#ifndef TETROMINO_H
#define TETROMINO_H

#include "Block.h"

/** Tetromino class
 * @brief Class containing the information about a Tetrimono
 * This class is immutable by design - a Tetromino's state should not be 
 * changed after it is created, rather replaced with a changed version, 
 * much like Strings in many OOP languages. This helps with collion detection.
 */
class Tetromino {
public:
    /**
     * Type enum
     * Contains the 7 possible tetrominos, represented by letters as on
     * the wikipedia page https://en.wikipedia.org/wiki/Tetromino
     */
    enum Type {
        I, O, T, J, L, S, Z
    };
    
    /**
     * @brief Gets a new Tetromino object from a given type
     * @param type The type of the tetromino
     * @returns A new Tetromino object
     */
    static Tetromino getTetrominoOfType(Tetromino::Type type);
    
    /**
     * @brief This Tetromino, but moved to the left
     * @returns A tetromino with the same blocks as this, but translated [-1, 0]
     */
    Tetromino movedLeft();
    
    /**
     * @brief This Tetromino, but moved to the right
     * @returns A tetromino with the same blocks as this, but translated [+1, 0]
     */
    Tetromino movedRight();
    
    /**
     * @brief This Tetromino, but moved down
     * @returns A tetromino with the same blocks as this, but translated [0, 1]
     */
    Tetromino movedDown();
    
    /**
     * @brief This Tetromino, but rotated clockwise
     * @returns A tetromino with the same blocks as this, but rotated
     * 90 degrees clockwise about the 'origin' (rounded down)
     */
    Tetromino rotatedClockwise();
    
    /**
     * @brief This Tetromino, but rotated anticlockwise
     * @returns A tetromino with the same blocks as this, but rotated
     * 90 degrees anticlockwise about the 'origin' (rounded down)
     */
    Tetromino rotatedAnticlockwise();
    
    /**
     * @brief Draw this Tetramino to the screen
     * @returns A tetromino with the same blocks as this, but rotated
     * 90 degrees about the 'origin' (rounded down)
     */
    void draw();
    
    /**
     * @brief Gets a random tetromino type
     * @returns A psudo-randomly selected Tetromino with rand()
     */
    static Type getRandomTetrominoType();
    
    /**
     * Destructor
     */
    ~Tetromino();
    
    /**
     * The 4 blocks that make up the tetromino
     */
    Block blocks[4];
    
    /**
     * @brief This Tetromino, but translated by [x, 0]
     * @param x The x coordinate to move the Tetromino to
     * @returns A tetromino with the same blocks as this, translated by [x, 0]
     */
    Tetromino teleportedTo(int x);
    
private:

    /**
     * @brief Returns a new immutable tetromino, translated by [dx, dy]
     * @param dx The amount to translate by on the X axis
     * @param dy The amount to translate by on the Y axis
     * @returns The tetromino translated by [dx, dy]
     */
    Tetromino translated(int dx, int dy);
    
    /**
     * @brief Returns a tetromino based on an input string representation, 
     * where a '*' represents a block, and a space represents nothing. For
     * example "* ***  " represents:
     * "* "
     * "**"
     * "* "
     * "  ", so the T shaped tetromino will be used.
     * @param str The string representation of the block as described above.
     * @returns The tetromino represented by the string.
     */
    static Tetromino getTetrominoFromString(const char * str);
    
    /**
     * @brief Constructor for a tetromino with 4 blocks
     * @param a The first block
     * @param a The second block
     * @param a The third block
     * @param a The fourth block
     */
    Tetromino(Block a, Block b, Block c, Block d);
};
#endif