#ifndef PREFS_H
#define PREFS_H

#include "mbed.h"
#include "SDFileSystem.h"

/**
 * Prefs class (Persistent Preferences)
 * @brief This class abstracts away data persistance to a key-value map.
 * It uses the Singleton design pattern, as only one instance should exist,
 * for efficiency reasons.
 */
class Prefs {
public:
    /**
     * Key enum
     * @brief The keys which are attached to val
     */
    enum Key {
        HIGHSCORE1, HIGHSCORE2, HIGHSCORE3,
        HIGHSCORE1_PLAYER, HIGHSCORE2_PLAYER, HIGHSCORE3_PLAYER,
        LAST_PLAYER, LAST_SCORE
    };
    
    /**
     * @brief Set the value of a specific key on the SD card
     * @param keys The key to set a value for
     * @param value The value to set the key to
     * 
     */
    void setKey(Key key, int value);
        
    /**
     * @param key The key to get
     * @returns The value associated with key on the sd card
     */
    int getKey(Key key);
        
    /**
     * @brief Resets the value of all keys to EMPTY
     * @see Prefs::EMPTY
     */
    void resetAll();
     
    /**
     * @brief used to get the singleton Prefs object,
     * or create it if it doesn't exist
     * @returns The singleton Prefs object
     */
    static Prefs * getInstance();
        
    /**
     * @brief used to signify a key has no value associated with it
     */
    static const int EMPTY = -1;
    
    /**
     * @brief Prints out all the keys and their values to stdout
     */
    void printAll();
private:
    /**
     * Creates a new Prefs object, creating a new file if needed. Called if
     * getInstance is called for the first time.
     */
    Prefs();
    
    /**
     * Destructor - closes the file and frees memory.
     */
    ~Prefs();
        
    /**
     * Pointer to the sd file system
     */
    SDFileSystem * sd;
     
    /**
     * Pointer to 'prefs.txt'
     */
    FILE * read_file; 
     
    /**
     * The singleton Prefs instance
     */
    static Prefs * instance;
     
    /**
     * The number of enumerated preferences
     */
    static const int NUMBER_OF_PREFS = 8;
     
    /**
     * Creates the text file, prefs.txt.
     */
    void createFile();
};

#endif