#ifndef GRID_H
#define GRID_H

#include "Tetromino.h"

/** 
 * Grid class
 * @brief Abstracts away access to a height x width 2d array, used to store
 * the 'landed' tetrominos.
 */
class Grid {
public:
    /**
     * @param tetromino A Tetromino to place on this grid
     * @returns A boolean value representing if the tetromino
     * could fit in the grid
     * @see Tetromino
     */
    bool isSpaceForTetromino(Tetromino tetromino);
    
    /**
     * @brief Commit a tetromino into the grid, setting values to true where
     * it overlaps the grid, updating the grid and clearing rows if needed
     * @param tetromino The tetromino to place
     * @returns The number of rows cleared by placing the tetromino on the grid
     * @see Tetromino
     */
    int placeTetromino(Tetromino tetromino);
        
    /**
     * @brief Use the Graphics functions to draw a block for every value
     * in the grid
     * @see Graphics::Game::drawBlock
     */
    void draw();
        
    /**
     * @brief Create an empty grid
     */
    Grid();
    
    /**
     * The amount of space for hidden blocks at the top of the screen
     */
    const static int HIDDEN_HEIGHT = 4;
        
    /**
     * The full height of the grid in blocks, including the hidden area
     */
    const static int GRID_HEIGHT = 24;
        
    /**
     * The width of the grid in blocks
     */
    const static int GRID_WIDTH = 10;
    
private:
    /**
     * @brief Checks for full rows and moves the blocks down if required,
     * "clearing" rows on the grid.
     * @returns The number of lines that have been cleared
     */
    int checkForLines();
    
    /**
     * @brief checks all rows and shifts them down if required
     */
    void moveDownIfNeeded();
    
    /**
     * @brief Moves the rows from row to row + 1, 'shifting' them down.
     * @param row The row to start shifting down from. All rows below this
     * will be translated down.
     */
    void shiftDownFrom(int row);
    
    /**
     * The grid as an array of booleans, true where there's a block 
     * in the space, false if it's empty
     */
    bool grid[GRID_WIDTH][GRID_HEIGHT];
    
    /**
     * @brief Checks if a block fits on the grid.
     * @param block The block to check
     * @returns True if the given block can fit in the grid, else false
     */
    bool isSpaceForBlock(Block block);
};
#endif