#ifndef GRAPHICS_H
#define GRAPHICS_H

#include "N5110.h"
#include "Bitmap.h"

/**
 * Graphics Namespace
 * @brief Unless stated otherwise, these functions all use the terms
 * 'horizontal'/'x' and 'vertical'/'y' to refer to the axis of the screen in
 * the rotated orientation the way the game is played.
 */
namespace Graphics {
    /**
     * @brief Clears the buffer, should be called every frame,
     * before all draw calls
     */
    void clear();
    
    /**
     * @brief Renders the buffer to the screen, should be called every frame,
     * after all draw calls
     */
    void render();
    
    /**
     * @brief Initialises the Graphics 'module', must be called to use any 
     * of the functions 
     */
    void init();
    
    /**
     * @brief Should be called to free the memory when the graphics module
     * is no longer needed
     */
    void deinit();
    
    /**
     * Turns the LCD backlight on if it's off, and vice versa
     */
    void toggleLight();
    
    /**
     * The height of the screen in pixels, when held in vertical orientation
     */
    const int SCREEN_HEIGHT = 84;
    
    /**
     * The width of the screen in pixels, when held in vertical orientation
     */
    const int SCREEN_WIDTH = 48;
    
    /**
     * Graphics::Game Namespace
     * @brief contains functions used by objects instanciated by the game class
     */
    namespace Game {
        /**
         * @brief draws a block to the screen
         * @param grid_x an x grid position
         * @param grid_y a y grid position
         */
        void drawBlock(int grid_x, int grid_y);
        
        /**
         * @brief draws a striped block to the screen, as in 
         * with lines along its diagonal
         * @param grid_x an x grid position
         * @param grid_y a y grid position
         */
        void drawStripedBlock(int grid_x, int grid_y);
        
        /**
         * @brief draws a dotted border around the grid
         */
        void drawBorder(); 
        
        /**
         * @brief 'Shake' or nudge the screen. The graphics functions natrually
         * 'lerp' the offset back to 0,0 after a few frames. It is additive, as
         * in calling shake(1,1) twice means twice as much camera offset.
         * @param dx Addititve amount of shake offset in the x direction
         * @param dy Addititve amount of shake offset in the x direction
         */
        void shake(int dx, int dy);   
    };
    
    /**
     * Graphics::UI Namespace
     * @brief contains functions used by the Menus::UI class
     */
    namespace UI {
        /**
         * The height of a character in pixels
         */
        const int CHAR_WIDTH = 4;
        
        /**
         * The height of a character in pixels
         */
        const int CHAR_HEIGHT = 6;
        
        /**
         * The space between characters in pixels
         */
        const int CHAR_SPACE = 1;
        
        /**
         * @brief draw n characters from a string
         * @see drawChar
         * @param x x coordiate of the text on the screen
         * @param y y coordiate of the text on the screen 
         * @param text the characters to write, terminated with \0
         */
        void drawText(int x, int y, const char * text);
        
        /**
         * @brief draw a box from (x1, y1) to (x2, y2)
         * @param x1 starting x coord
         * @param y1 starting y coord
         * @param x2 ending x coord
         * @param y2 ending y coord
         */
        void drawBorder(int x1, int y1, int x2, int y2);
        
        /**
         * @brief draws
         * @param bitmap the bitmap object to draw, in the 
         * @see Bitmap
         * @param x The actual x (in landscape orientation) coordinate
         * @param y The actual y (in landscape orientation) coordinate
         */
        void drawSprite(Bitmap * bitmap, int x, int y);
    };
};
#endif