#ifndef GAME_H
#define GAME_H

#include "Grid.h"

/**
 * Game class
 * @brief Holds everything about a session of a game of Tetris
 */
class Game {
public:
    /**
     * Difficulty enum
     * @brief Used to start a game in Easy, Medium or Hard mode, which changes
     * the fall speed of the blocks and the points recived for clearing a row.
     */
    enum Difficulty { EASY, MEDIUM, HARD };
    
    /**
     * @brief Create a new game with a specified difficulty
     */
    Game(Difficulty difficulty);
    
    /**
     * Destructor
     */
    ~Game();
    
    /**
     * @brief Update the game, responding to inputs and
     * moving the current tetromino down
     */
    void update();
     
    /**
     * @brief Draw all the elements of the game, the border and tetrominos
     */
    void draw();
    
private:
    /**
     * The current game's score
     */
    int score;
    
    /**
     * The amount of points the player gets for clearing a row, 
     * changes with difficulty mode
     */
    int score_for_row;
    
    /**
     * Move the current tetromino down one block, called every frame.
     * If it hits the grid, it is placed on the grid and a new tetromino is
     * instanciated.
     */
    void moveCurrentTetrominoDown();
    
    /**
     * The current grid instance, where the tetrominos are stored
     */
    Grid grid;
    
    /**
     * The upcoming tetromino type, that the next tetromino will have
     */
    Tetromino::Type next_tetromino_type;
    
    /**
     * The tetromino the player currently controls
     */
    Tetromino current_tetromino;
    
    /**
     * The number of frames before the tetromino will fall down one block
     */
    long move_frames;
    
    /**
     * Internal count of the number of frames elapsed
     */
    long frames;
    
    /**
     * The last time the tetromino fell down one block
     */
    long last_move_frame;
    
    /**
     * The initial x position for a new tetromino
     */
    int start_x;
    
    /**
     * Move the current tetromino down all the way until it hits the floor.
     * Called when 'B' is hit.
     */
    void dropCurrentTetromino();
    
    /**
     * Called when the grid is full, shows the game over menu
     */
    void gameOver();
    
    /**
     * Add to the players score based on the number of rows they cleared
     * @param rows_cleared The number of rows cleared by the last move
     */
    void addScore(int rows_cleared);
    
    /**
     * Put the players score on the leaderboard and rearrange it if needed
     */
    void updateLeaderboard();
    
    /**
     * The current player is assigned a number for the leaderboard
     */
    int player_number;
};

#endif