#ifndef ROSENENGINE_H
#define ROSENENGINE_H

#include "mbed.h"
#include <ctime>
#include "N5110.h"
#include "Gamepad.h"
#include "Ship.h"
#include "Weapons.h"
#include "Menu.h"
#include "Enemy.h"
#include "Health.h"
#include "Lore.h"

/** RosenEngine Class
@brief Library that used all other game libraries in order to make game functions
@author Ozoemena Adrian Ikrnna
@date 8th May 2019
*/
class RosenEngine
{

public:
    /** constructor */
   RosenEngine();
   /** deconstructor */
    ~RosenEngine();
    
    /** A mutator method that initializes ship size and speed, the default ship is the kestrel
    *@param ship_width, the width of the ship
    *@param ship_height, the height of the ship
    *@param ship_speed, the speed of the ship
    *@param ship_xpos, the ship's x co-ordinate
    *@param ship_ypos, the ship's y co-ordinate
    */
    void init(int ship_width,int ship_height,int ship_speed,int ship_xpos,int ship_ypos);
    /** A mutator method that resets some key values that neet to be reinitialized when the player
    *returns to the ment
    */
    void reset();
    /** An accessor method that reads the input from the gamepad
    *@param &pad address of the gamepad library used to read inputs and send outputs to the gamepad
    */
    void read_input(Gamepad &pad);
    /** A mutator method that updates all position and variable values based on inputs and the
    * time that has passed
    *@param &pad address of the gamepad library used to read inputs and send outputs to the gamepad
    */
    void update(Gamepad &pad);
    /** Draws all game assets on lcd display
    *@param &lcd address of the N5110 library used for the lcd display
    *@param &pad address of the gamepad library used to read inputs and send outputs to the gamepad
    */
    void draw(N5110 &lcd, Gamepad &pad);
    /** An accessor method that gets the position of all important game objects on screen
    */
    void get_pos();
    /** Displays the title screen on the lcd
    *@param &lcd address of the N5110 library used for the lcd display
    */
    void title(N5110 &lcd);
    /** An accessor method that gets the ycursor value to be used in the menu
    *@returns the  ycursor value
    */
    int get_ycursor();
    /** A mutator method used to set the ship being used in the game.
    * either the kestrel, the imperion or the orion
    */
    void set_shipUsed();
    /** Displays the ship option in the menu to allow player switch ships
    *@param &lcd address of the N5110 library used for the lcd display
    */
    void ship_select(N5110 &lcd);
    /** A mutator method that checks the enemies and players health. it resets an enemy if
    * its health falls to 0 and sets dead to true if the players health falls to 0.
    * either the kestrel, the imperion or the orion
    */
    void check_health();
    /** Calulates the amount of time past in seconds 
    *@param fps the frames per second that is set
    */
    float timer(int fps);
    /** An accessor method that returns true when a player dies
    */
    bool dead();
    /** Displays the intro to the game 
    *@param &lcd address of the N5110 library used for the lcd display
    */
    void intro(N5110 &lcd);
    /** An accessor method that gets the ships position
    *@returns the ships position
    */
    Vector2D get_shipPos();
    /** Checks for a collision between two 2D objects
    *@param xpos1, the x co-ordinate of the first object
    *@param ypos1, the y co-ordinate of the first object
    *@param width1, the width of the first object
    *@param height1, the height of the first object
    *@param xpos2, the x co-ordinate of the second object
    *@param ypos2, the y co-ordinate of the second object
    *@param width2, the width of the second object
    *@param height2, the height of the second object
    */
    bool check_collision(int xpos1, int ypos1,int width1,int height1,int xpos2, int ypos2,int width2,int height2);
    /** Checks if two objects positione intersect across a vertical line
    *used for the imperion weapon collisions
    *@param xpos1, the x co-ordinate of the first object
    *@param width1, the width of the first object
    *@param xpos2, the x co-ordinate of the second object
    *@param width2, the width of the second object
    */
    bool check_collision1(int xpos1,int width1,int xpos2,int width2);
    /** Displays help option
    **@param &lcd address of the N5110 library used for the lcd display
    */
    void help(N5110 &lcd);
        
private:
    //_____________________private-Methods________________________________________________________
    /** An accessor method used to get the current number of the two enemies, seeker and shooter, on screen
    @returns a 2D vector of the enemy numbers with the .x referring to shooters and the .y referring to seekers
    */
    Vector2D get_enemynum();
    /** A mutator method that updates the ships weapons
    *@param &pad address of the gamepad library used to read inputs and send outputs to the gamepad
    */
    void update_ship_weapon(Gamepad &pad);
    /** Draws the appropriate ship based on shipUsed (the ship being used)
    *@param &lcd address of the N5110 library used for the lcd display
    *@param &pad address of the gamepad library used to read inputs and send outputs to the gamepad
    */
    void draw_ship(N5110 &lcd, Gamepad &pad);
    /** A mutator function that changes ship_width appropriate ship based on shipUsed (the ship being used)
    */
    void set_ship_size();
    /** A mutator function that changes the players score
    *@param points, the amount of points the player scored
    */
    void score(int points);
    /** Checks if a seeker and the player ship collided
    *@param &pad address of the gamepad library used to read inputs and send outputs to the gamepad
    */
    void seeker_ship_collision(Gamepad &pad);
    /** Checks if a shooter and the player ship collided
    *@param &pad address of the gamepad library used to read inputs and send outputs to the gamepad
    */
    void shooter_ship_collision(Gamepad &pad);
    /** Checks if a shooter's projectile and the player ship collided
    *@param &pad address of the gamepad library used to read inputs and send outputs to the gamepad
    */
    void shooterw_ship_collision(Gamepad &pad);
    /** Checks if the kestrel ships's projectile and the seeker collided
    *@param &pad address of the gamepad library used to read inputs and send outputs to the gamepad
    */
    void kestrelw_seeker_collision(Gamepad &pad);
    /** Checks if the Imperions lazer and the seeker collided
    *@param &pad address of the gamepad library used to read inputs and send outputs to the gamepad
    */
    void imperionw_seeker_collision(Gamepad &pad);
    /** Checks if the kestrel ships's projectile and the shooter collided
    *@param &pad address of the gamepad library used to read inputs and send outputs to the gamepad
    */
    void kestrelw_shooter_collision(Gamepad &pad);
    /** Checks if the Imperions lazer and the shooter collided
    *@param &pad address of the gamepad library used to read inputs and send outputs to the gamepad
    */
    void imperionw_shooter_collision(Gamepad &pad);
    /** Checks if the orions weapon collided with any enemy
    *@param &pad address of the gamepad library used to read inputs and send outputs to the gamepad
    */
    void orionw_collision(Gamepad &pad);
    /** Gradually increased the difficulty of the game over time
    *@param time_elapsed, the amount of time since the player selected play from menu
    */
    void scaling(float time_elapsed);
    /** Generates a random number using the <ctime> library
    *@returns a random number
    */
    int rand_no();
    /** Displays a random tip on lcd
    *@param &lcd address of the N5110 library used for the lcd display
    */
    void game_over(N5110 &lcd);
    /** Displays the points the player has on screen
    *@param &lcd address of the N5110 library used for the lcd display
    */
    void disp_points(N5110 &lcd);
    /** Checks the seekers health
    */
    void check_se_health();
    /** Checks the shooter health
    */
    void check_sh_health();
    /** Finds the average distance between two points on the 2D screen
    *@param x1 the x co-ordinate of the first point
    *@param y1 the y co-ordinate of the first point
    *@param x2 the x co-ordinate of the second point
    *@param y2 the y co-ordinate of the second point
    *@returns the distance between the two points
    */
    int range(int x1, int y1, float x2, float y2);
    /** Finds the closest enemy ship to the player's ship on screen
    *@ returns the index and distance of closest enemy ship
    */
    Vector2D find_closest1();
    /** An accessor function that finds the position of the closest enemy on screen to ship
    *@param index, the index number of the closest ship (0-3) for shooter and (4-6) dor seekers
    *@returns the position of the closest enemy
    */
    Vector2D find_closest2(int index);
    //_______________Private-Variables__________________________________________
    Ship _ship;                 // The ship class object
    Weapons _weapons;           // The weapon class object
    Menu _menu;                 // The menu class object    
    Enemy _enemy;               // The enemy class object
    Health _health;             // The health class object
    Lore _lore;                 // The lore class object
    Vector2D _joystick;         // a vector containing the joystick values
    Direction _d;               // direction of the joystick, N,S,E, or W
    SHIP _shipUsed;             // a variable in of type enum SHIP that contains the ship being used by the player 
    int _ycursor;               // the _ycursor  value that indicates where the rectangular cursor is drawn in the menu                
    int _score;                 // the players score
    int _shipWidth;             // the ship's width
    int _shipHeight;            // the ship's height
    Vector2D _shipPos;          // the ships position
    Vector2D _shooterPos[3];    // an array of all the shooter positions
    Vector2D _shooterWPos[3];   // an array of all the shooter weapon positions
    Vector2D _seekerPos[3];     // an array of the seeker position           
    bool _dead;                 // true if player dies else false
    int _times_run;             // the amount of times the game loop is run
    int _no_shooters;           // the anount of shooters
    int _no_seekers;            // the amount of seeker
    bool _intro;                // true if intro has been shown else false     
    float _wait_time;           // amount of time before increasing difficulty

};

#endif
