#ifndef TANKENGINE_H
#define TANKENGINE_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"
#include "Projectile.h"
#include "Tank.h"
#include "Graphics.h"
#include "Menus.h"
#include "Scores.h"

/** TanksEngine class
@brief Class that holds the main game loop and game mechanics. It governs the 
       the use of game inputs like player actions through controls and game 
       outputs like updating the lcd, lighting LEDs and playing piezo tones.       
@version 1.0
@author Maxim C. Delacoe
@date April 2019
@code

#include "mbed.h"
#include "Gamepad.h"
#include "N5110.h"
#include "Bitmap.h"
#include "TanksEngine.h"
#include "Tank.h"
#include "Projectile.h"
#include "Graphics.h"
#include "Menus.h"
#include "Scores.h"

N5110 lcd(PTC9,PTC0,PTC7,PTD2,PTD1,PTC11);
TanksEngine engine;
Gamepad pad;
Graphics graphics;
Menus menus;
Scores scores;

int main()
{    // need to initialise LCD and Gamepad  
  lcd.init();
  pad.init();
      
  while(1) {  // infinite loop
    // Run start up screen and wait for player input.
    menus.start_up_screen(graphics, lcd, pad);
    // Run the menus and save any player preferences like mute enabled and
       starting health for the tanks.
    menus.main_menu(graphics, lcd, pad, scores); 
    // Initialise the game based on the players preferences. Setting starting 
       positions, health, etc.
    engine.initgame(menus);
    // Run the main game loop until a player reaches 0 health.
    engine.game_loop(graphics, lcd, pad, menus, scores);        
  }
}    

@endcode
*/
class TanksEngine
{
    
public:
  // Constructor and destructor.
  /**
  * @brief Constructor 
  * @details Sets the movement limits and gravity's acceleration.
  */
  TanksEngine();
  /**
  * @brief Destructor 
  * @details Non user specified.
  */
  ~TanksEngine(); 
  
  /**
  * @brief Resets member variables from previous games to their default values and sets the values specified in setting. 
  * @param menus * @details The menus object from Menus class.
  */   
  void initgame(Menus &menus);
  /**
  * @brief Runs the main gameloop updating the game as the player inputs controls and updating display based on game events. 
  * @param graphics * @details The graphics object from Graphics class.
  * @param lcd * @details The lcd object from N5110 class.
  * @param pad * @details The pad object from Gamepad class.
  * @param menus * @details The menus object from Menus class.
  * @param scores * @details The scores object from Scores class.
  */  
  void game_loop(Graphics &graphics, N5110 &lcd, Gamepad &pad,
                 Menus &menus, Scores &scores);
    
private:
  
    // Game Flow
  void _left_tank_turn(Graphics &graphics, Gamepad &pad);
  void _right_tank_turn(Graphics &graphics, Gamepad &pad);
  void _projectile_phase(N5110 &lcd, Gamepad &pad);
  void _change_turn();
  void _end(Graphics &graphics, N5110 &lcd, Gamepad &pad, Scores &scores); 
    // Game Mechanics
  void _tank_shoots(int x, int y, int turn, Gamepad &pad);
  void _object_hit(int x, int y, N5110 &lcd, Gamepad &pad);
  bool _collision_pl(Tank _tankl, Projectile _proj);
  bool _collision_pr(Tank _tankr, Projectile _proj);
  bool _collision_pm(int x, int y, N5110 &lcd, Projectile _proj);
    // Game Inputs 
  void _read_input(Gamepad &pad);
  void _decrement_cooldowns();
    // Game Outputs
  void _draw(Graphics graphics, N5110 &lcd);
  void _sounds(int n, Gamepad &pad);

  Tank _tankl;
  Tank _tankr;
    
  Projectile _proj;

  int _initial_health;       
  int _cooldown_l; 
  int _cooldown_r;
  int _move;
  int _turn;
  int _turn_counter;
  bool _fire;
  bool _mute;
  float _angle;
  float _mag;
  float _vel;
  float _grav;
  float _wind;
};

#endif // TANKSENGINE_H