#ifndef TANK_H
#define TANK_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"

/** Tank Class
* @brief Generates a tank object that can move position and generate a hitbox.
* @author Maxim C. Delacoe
* @date April 2019
 
@code
 
#include "mbed.h"
#include "N5110.h"
#include "Tank.h"
#include "Graphics.h"
 
N5110 lcd(PTC9,PTC0,PTC7,PTD2,PTD1,PTC11);
Tank _tank;
Graphics _graphics;
 
 
int main() {
    
  lcd.init();
  int i = 0;
  _tank.set_position(15, 0); // Set initial position
  _tank.set_movement_limits(0, 73);  // Set movement limits within lcd
  _tank.set_speed(-25);  // slow speed down
  
  while (1) {
    
    if (i < 1499) {
      _tank.move_position(1);   // tank moves right for the first 1500 iterations
    } else { 
      _tank.move_position(-1);  // otherwise the tank moves left
    }
    _tank.generate_hitbox();  // update the objects hitbox
    
    lcd.clear();
    int x = _tank.get_position_x();
    int y = _tank.get_position_y();           // get the tank's position and update sprite
    _graphics.draw_tank_l(x, y, lcd);
    _graphics.draw_turret_l(x, y, 90, lcd) ;
    lcd.refresh();
    
    i++;
    if (i > 3000) { // reset incrementer
      i = 0;
    wait_ms(1000/60);
    }
  }                
} 
 
@endcode
*/

class Tank
{
    
public:
  // Constructor and destructor.
  /**
  * @brief Constructor 
  * @details Sets the default movement speed.
  */ 
  Tank();
  /**
  * @brief Destructor 
  * @details Non user specified.
  */
  ~Tank();

  //Accessors
  /**
  * @brief Gets the tank's position in the x direction.
  * @returns the tank's position in the x direction
  */
  int get_position_x();
  /**
  * @brief Gets the tank's position in the y direction.
  * @returns the tank's position in the y direction
  */
  int get_position_y();
  /**
  * @brief Gets the tank's ith element of its hitbox array.
  * @returns a single hitbox integer value
  */
  int get_hitbox(int i);
  /**
  * @brief Gets the value of the tank's health.
  * @returns The tank's health value.
  */
  int get_health();
  //Mutators
  /**
  * @brief Sets the x and y positions of the tank.
  * @param x @details The tank's position in the x direction
  * @param y @details The tank's position in the y direction
  */
  void set_position(int x, int y);
  /**
  * @brief Sets the minimum and maximum positions the tank can hold in the x direction.
  * @param left @details The tank's minimum position in the x direction
  * @param right @details The tank's maximum position in the y direction
  */
  void set_movement_limits(int left, int right);
  /**
  * @brief Sets the tank's movement speed.
  * @param s @details The speed of the tank's movement
  */
  void Tank::set_speed(int s);
  /**
  * @brief Sets the tank's health.
  * @param h @details The value of the tank's health
  */
  void set_health(int h);

  //Member Methods
  /**
  * @brief Moves the tank left or right depending on the whether the input is negative or positive.
  * @param d @details The direction the tank should move
  */
  void move_position(int d);
  /**
  * @brief Decrements the tank's health.
  */
  void lose_health();    
  /**
  * @brief Fills a 40 integer array with all the values associated with the tank's area.
  */
  void generate_hitbox();

private:
    
  int _position_x;
  int _position_y;
  int _left_lim;
  int _right_lim;
  int _move_counter;
  int _speed; 
  int _health;
  int _hitbox[40];
};

#endif // TANK_H