#ifndef PROJECTILE_H
#define PROJECTILE_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"

/** Projectile Class
* @brief Generates a projectile object that has ballistic motion and generates a hitbox.
* @author Maxim C. Delacoe
* @date April 2019
 
@code

#include "mbed.h"
#include "N5110.h"
#include "Projectile.h"
#include "Graphics.h"
 
N5110 lcd(PTC9,PTC0,PTC7,PTD2,PTD1,PTC11);
Projectile _proj;
Graphics _graphics;
 
 
int main() {
    
  lcd.init();
  
  while (1) {
    
    srand(time(0));           // seed rand fucntion                                   
    int angle = rand() % 90;       // randomise launch angle from 0 to 90 degrees  
    _proj.set_launch_parameters(0, 0,  angle, 1.3, 0.02, 0);  // set launch parameters including randomised launch angle
    
    for (int i = 0; i < 300; i++) {
      
      _proj.update_flight(); 
      _proj.generate_hitbox();
      
      lcd.clear();
      int x = _proj.get_position_x();
      int y = _proj.get_position_y();  // get the projectile's position and update sprite
      _graphics.draw_projectile(x, y, lcd);
      lcd.refresh();
      
      wait_ms(1000/60);
    }
  }                
} 
@endcode
*/

class Projectile
{
    
public:
  // Constructor and destructor.
  /**
  * @brief Constructor 
  * @details Non user specified.
  */     
  Projectile();
  /**
  * @brief Destructor 
  * @details Non user specified.
  */
  ~Projectile();
    
  // Accessors
  /**
  * @brief Gets the projectile's position in the x direction.
  * @returns the projectile's position in the x direction
  */
  int get_position_x();
  /**
  * @brief Gets the projectile's position in the y direction.
  * @returns the projectile's position in the y direction
  */
  int get_position_y();
  /**
  * @brief Gets the projectile's ith element of its hitbox array.
  * @returns a single hitbox integer value
  */
  int get_hitbox(int i);
  // Mutators
  /**
  * @brief Sets the launch parameters for the projectile's ballistic motion.
  * @param x @details The projectile's position in the x direction
  * @param y @details The projectile's position in the y direction
  * @param ang @details The projectile's angle of launch from the ground
  * @param vel @details The projectile's initial velocity
  * @param grav @details The projectile's acceleration in the y direction due to gravity 
  * @param wind @details The projectile's acceleration in the x direction due to wind 
  */
  void set_launch_parameters(int x, int y, float ang, float vel, 
                             float grav, float wind);
  // Other Methods
  /**
  * @brief Fills a 4 integer array with all the values associated with the projectile's area.
  */
  void generate_hitbox();
  /**
  * @brief Increments the time component of the equations for ballistic mothion and updates x and y positions.
  */
  void update_flight();
  /**
  * @brief Checks whether the projectile has fallen outside the left, bottom or right of the screen.
  * @returns a boolean true if it is out of bounds
  */
  bool check_boundaries();
    
private:
    
  int _position_x;
  int _position_y;
  int _hitbox[5];
  int _init_x;
  int _init_y;
  float _time;
  float _lnch_ang;
  float _init_vel; 
  float _grav_acc; 
  float _wind_acc;
};

#endif // PROJECTILE_H
