#ifndef GRAPHICS_H
#define GRAPHICS_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"

/** Graphics Class
* @brief Holds the sprites and governs their use. Governs the use of LEDs in the game.
* @author Maxim C. Delacoe
* @date April 2019
 
@code

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"
#include "Graphics.h"
 
N5110 lcd(PTC9,PTC0,PTC7,PTD2,PTD1,PTC11);
Gamepad pad;
Graphics _graphics;
 
 
int main() {
    
  lcd.init();
  
  while (1) {
    
    int _mag = pad.get_mag(); 
    int _angle = pad.get_angle();
    
    lcd.clear();
    _graphics.draw_tank_l(10, 0, lcd); // Draw tank in appropriate position.                                                          
    _graphics.draw_turret_l(10, 0, _angle, lcd); 
    if (_mag >= 0.5) {                           // only draw the reticle if the joystick is not in
      _graphics.draw_reticle(10, 0, _angle, lcd);    // a neutral position.
    } 
    lcd.refresh();  
  }                
} 
@endcode
*/

class Graphics
{
    
public:
  // Constructor and destructor.
  /**
  * @brief Constructor 
  * @details Non user specified.
  */        
  Graphics();
  /**
  * @brief Destructor 
  * @details Non user specified.
  */
  ~Graphics();
   
  // Left Tank
  /**
  * @brief Draw the left tank sprite.
  * @param x * @details The tank's position in the x direction
  * @param y * @details The tank's position in the y direction
  * @param lcd * @details The lcd object from N5110 class
  */
  void draw_tank_l(int x, int y, N5110 &lcd);
  /**
  * @brief Draw the left tank's turret sprite.
  * @param x * @details The left tank turret's position in the x direction
  * @param y * @details The left tank turret's position in the y direction
  * @param angle * @details The tank turret's angle
  * @param lcd * @details The lcd object from N5110 class
  */
  void draw_turret_l(int x, int y, int angle, N5110 &lcd);
  /**
  * @brief Show left tank victory screen.
  * @param x * @details The left tank turret's position in the x direction
  * @param y * @details The left tank turret's position in the y direction
  * @param angle * @details The tank turret's angle
  * @param lcd * @details The lcd object from N5110 class
  */
  void draw_left_victory(N5110 &lcd);
  // Right Tank
  /**
  * @brief Draw the right tank sprite.
  * @param x * @details The tank's position in the x direction
  * @param y * @details The tank's position in the y direction
  * @param lcd * @details The lcd object from N5110 class
  */
  void draw_tank_r(int x, int y, N5110 &lcd);
  /**
  * @brief Draw the right tank's turret sprite.
  * @param x * @details The right tank turret's position in the x direction
  * @param y * @details The right tank turret's position in the y direction
  * @param angle * @details The tank turret's angle
  * @param lcd * @details The lcd object from N5110 class
  */
  void draw_turret_r(int x, int y, int angle, N5110 &lcd);
  /**
  * @brief Show right tank victory screen.
  * @param x * @details The right tank turret's position in the x direction
  * @param y * @details The right tank turret's position in the y direction
  * @param angle * @details The tank turret's angle
  * @param lcd * @details The lcd object from N5110 class
  */
  void draw_right_victory(N5110 &lcd);
  // Projectile
  /**
  * @brief Draw the projectile sprite.
  * @param x * @details The projectile's position in the x direction
  * @param y * @details The projectile's position in the y direction
  * @param lcd * @details The lcd object from N5110 class
  */
  void draw_projectile(int x, int y, N5110 &lcd);
  // Display
  /**
  * @brief Draws a bar/arrow that indicates the strength and direction of the wind.
  * @param wind * @details The acceleration due to wind in the x direction
  * @param lcd * @details The lcd object from N5110 class
  */
  void draw_wind_bar(float wind, N5110 &lcd);
  /**
  * @brief Draws a dot to indicate where the tank is aiming.
  * @param x * @details The left tank turret's position in the x direction
  * @param y * @details The left tank turret's position in the y direction
  * @param angle * @details The tank turret's angle
  * @param lcd * @details The lcd object from N5110 class
  */
  void draw_reticle(int x, int y, float angle, N5110 &lcd);
  // Maps
  /**
  * @brief Draws a building obstacle on the map.
  * @param x * @details The map object's position in the x direction
  * @param y * @details The map object's position in the y direction
  * @param lcd * @details The lcd object from N5110 class
  */
  void draw_parkinson_map(int x, int y, N5110 &lcd);
  // LEDs
  /**
  * @brief Lights LEDs equal to the input value.
  * @param current * @details The player's current health
  * @param pad * @details The pad object from Gamepad class
  */
  void show_health(int current, Gamepad &pad);
  /**
  * @brief Toggles three LEDS on at a time for the start up screen.
  * @param alt * @details The value of the alternator/incrementer
  * @param pad * @details The pad object from Gamepad class
  */
  void start_up(int alt, Gamepad &pad);
  // Start up Screen
  /**
  * @brief Draws the start up screen visuals.
  * @param lcd * @details The lcd object from N5110 class
  */
  void draw_start_up_screen(N5110 &lcd);
    
private:
  // Left Tank
  void _turret_angle_l1(int x, int y, N5110 &lcd);
  void _turret_angle_l2(int x, int y, N5110 &lcd);
  void _turret_angle_l3(int x, int y, N5110 &lcd);
  void _turret_angle_l4(int x, int y, N5110 &lcd);
  void _turret_angle_l5(int x, int y, N5110 &lcd);
  // Right Tank
  void _turret_angle_r1(int x, int y, N5110 &lcd);
  void _turret_angle_r2(int x, int y, N5110 &lcd);
  void _turret_angle_r3(int x, int y, N5110 &lcd);
  void _turret_angle_r4(int x, int y, N5110 &lcd);
  void _turret_angle_r5(int x, int y, N5110 &lcd);

};

#endif // GRAPHICS