#include "Engine.h"

Engine::Engine()
{

}

Engine::~Engine()
{

}

void Engine::init(int wall_width,int wall_gap,int stone_size,int speed) {
    
    // initialise the game parameters
    _wall_width = wall_width;
    _wall_gap = wall_gap; // wall gap presents the distance between the top wall and the bottom wall, which is licked to difficulty
    _stone_size = stone_size; // larger stone means it has larger area to trigger "_avenger.add_score();"
    _speed = speed; 

    // x position on screen - WIDTH is defined in N5110.h
    _w0x = WIDTH; // there is total 5 walls in the game, all are placed in distributed positions
    _w1x = WIDTH + 30;
    _w2x = WIDTH + 60;
    _w3x = WIDTH + 90;
    _w4x = WIDTH + 120;
    _avengerx = 10;
    _stonex = WIDTH + 105;

    _w0.init(_w0x,_wall_gap,_wall_width,_speed); // initiate all walls with random holes
    _w1.init(_w1x,_wall_gap,_wall_width,_speed);
    _w2.init(_w2x,_wall_gap,_wall_width,_speed);
    _w3.init(_w3x,_wall_gap,_wall_width,_speed);
    _w4.init(_w4x,_wall_gap,_wall_width,_speed);
    _avenger.init(_avengerx); // set the avenger in the middle
    _stone.init(_stonex,_stone_size,_speed); // initiate stone size and a random y position
}

void Engine::read_input(Gamepad &pad) { // get joystick reading
    
    _d = pad.get_direction();
    _mag = pad.get_mag();
    _mapped_coord = pad.get_mapped_coord();
}

void Engine::draw(N5110 &lcd) {
    // draw the elements in the LCD buffer
    // pitch
    lcd.drawRect(0,0,WIDTH,HEIGHT,FILL_TRANSPARENT);
    // walls
    _w0.draw1(lcd);
    _w0.draw2(lcd);
    _w1.draw1(lcd);
    _w1.draw2(lcd);
    _w2.draw1(lcd);
    _w2.draw2(lcd);
    _w3.draw1(lcd);
    _w3.draw2(lcd);
    _w4.draw1(lcd);
    _w4.draw2(lcd);
    // avenger
    _avenger.draw(lcd);
    // stone
    _stone.draw(lcd);
}


void Engine::update(Gamepad &pad) {
    
    // important to update the walls, the avenger and the stone before checking collisions so the program can correct for it before updating the display
    _w0.update();
    _w1.update();
    _w2.update();
    _w3.update();
    _w4.update();
    _avenger.update(_d,_mag,_mapped_coord);
    _stone.update();
    check_wall_collision(pad);
    check_score(pad);
}

void Engine::check_wall_collision(Gamepad &pad) {
    
    // read the current avenger's position
    Vector2D avenger_pos = _avenger.get_pos();
    
    //read all the walls position
    int w0_x = _w0.get_x();
    int w1_x = _w1.get_x();
    int w2_x = _w2.get_x();
    int w3_x = _w3.get_x();
    int w4_x = _w4.get_x();
    
    if (w0_x <= 1) {  //  when the wall goes behind the avenger, it reset back to the original pos to continue the game
        _w0.reset();
    }
    if (w1_x <= 1) {
        _w1.reset();
    }
    if (w2_x <= 1) {
        _w2.reset();
    }
    if (w3_x <= 1) {
        _w3.reset();
    }
    if (w4_x <= 1) {
        _w4.reset();
    }
    
    int w0_height = _w0.get_height(); // get the specific, random height for each of the walls
    int w1_height = _w1.get_height();
    int w2_height = _w2.get_height();
    int w3_height = _w3.get_height();
    int w4_height = _w4.get_height();
    
    if ((
         (avenger_pos.y + 5 >= 0) && (avenger_pos.y + 1 <= w0_height) && // check all the top wall corner collision and the bottom for all the walls
         (avenger_pos.x + 10 >= _w0.get_x()) && (avenger_pos.x + 10 <= _w0.get_x() + 1) // left and right collisions
        ) || (
         (avenger_pos.y + 5 >= 0) && (avenger_pos.y + 1 <= w1_height) &&
         (avenger_pos.x + 10 >= _w1.get_x()) && (avenger_pos.x + 10 <= _w1.get_x() + 1)
        ) || (
         (avenger_pos.y + 5 >= 0) && (avenger_pos.y + 1 <= w2_height) &&
         (avenger_pos.x + 10 >= _w2.get_x()) && (avenger_pos.x + 10 <= _w2.get_x() + 1)
        ) || (
         (avenger_pos.y + 5 >= 0) && (avenger_pos.y + 1 <= w3_height) &&
         (avenger_pos.x + 10 >= _w3.get_x()) && (avenger_pos.x + 10 <= _w3.get_x() + 1)
        ) || (
         (avenger_pos.y + 5 >= 0) && (avenger_pos.y + 1 <= w4_height) &&
         (avenger_pos.x + 10 >= _w4.get_x()) && (avenger_pos.x + 10 <= _w4.get_x() + 1)
        )
       ) {
        _avenger.lose_score(); // if collision does happen, a stone is lost due to unnecessary time interference
        
        pad.tone(100.0,0.1); // sad sound playing
        wait(0.1);
        pad.tone(200.0,0.1);
        wait(0.1);
        pad.tone(300.0,0.1);
        wait(0.1);
        pad.tone(400.0,0.1);
        wait(0.1);
        pad.tone(500.0,0.1);
        wait(0.1);
    }
    
    if ((
         (avenger_pos.y + 5 >= _wall_gap + w0_height) && (avenger_pos.y + 1 <= HEIGHT) && // check collisions for the bottom wall
         (avenger_pos.x + 10 >= _w0.get_x()) && (avenger_pos.x + 10 <= _w0.get_x() + 1) // left and right side
        ) || (
         (avenger_pos.y + 5 >= _wall_gap + w1_height) && (avenger_pos.y + 1 <= HEIGHT) &&
         (avenger_pos.x + 10 >= _w1.get_x()) && (avenger_pos.x + 10 <= _w1.get_x() + 1)
        ) || (
         (avenger_pos.y + 5 >= _wall_gap + w2_height) && (avenger_pos.y + 1 <= HEIGHT) &&
         (avenger_pos.x + 10 >= _w2.get_x()) && (avenger_pos.x + 10 <= _w2.get_x() + 1)
        ) || (
         (avenger_pos.y + 5 >= _wall_gap + w3_height) && (avenger_pos.y + 1 <= HEIGHT) &&
         (avenger_pos.x + 10 >= _w3.get_x()) && (avenger_pos.x + 10 <= _w3.get_x() + 1)
        ) || (
         (avenger_pos.y + 5 >= _wall_gap + w4_height) && (avenger_pos.y + 1 <= HEIGHT) &&
         (avenger_pos.x + 10 >= _w4.get_x()) && (avenger_pos.x + 10 <= _w4.get_x() + 1)
        )
       ) {
        _avenger.lose_score(); // score lost
        
        pad.tone(100.0,0.1); // sad sound
        wait(0.1);
        pad.tone(200.0,0.1);
        wait(0.1);
        pad.tone(300.0,0.1);
        wait(0.1);
        pad.tone(400.0,0.1);
        wait(0.1);
        pad.tone(500.0,0.1);
        wait(0.1);
    }
}

void Engine::check_score(Gamepad &pad) { //check the score to trigger how many leds should be turned on
    
    int avenger_score = _avenger.get_score(); // get score
    
    if (avenger_score == 0) { // 7 if statements represents 7 conditions
        pad.leds_off();
    }
    if (avenger_score == 1) { // getting one stone turns on 1 led
        pad.led(1,1.0f);
        pad.led(2,0.0f);
        pad.led(2,0.0f);
        pad.led(2,0.0f);
        pad.led(2,0.0f);
        pad.led(2,0.0f);
    }
    if (avenger_score == 2) { // two stone, two led
        pad.led(1,1.0f);
        pad.led(2,1.0f);
        pad.led(3,0.0f);
        pad.led(4,0.0f);
        pad.led(5,0.0f);
        pad.led(6,0.0f);
    }
    if (avenger_score == 3) {
        pad.led(1,1.0f);
        pad.led(2,1.0f);
        pad.led(3,1.0f);
        pad.led(4,0.0f);
        pad.led(5,0.0f);
        pad.led(6,0.0f);
    }
    if (avenger_score == 4) {
        pad.led(1,1.0f);
        pad.led(2,1.0f);
        pad.led(3,1.0f);
        pad.led(4,1.0f);
        pad.led(5,0.0f);
        pad.led(6,0.0f);
    }
    if (avenger_score == 5) {
        pad.led(1,1.0f);
        pad.led(2,1.0f);
        pad.led(3,1.0f);
        pad.led(4,1.0f);
        pad.led(5,1.0f);
        pad.led(6,0.0f);
    }
    if (avenger_score == 6) {
        pad.led(1,1.0f);
        pad.led(2,1.0f);
        pad.led(3,1.0f);
        pad.led(4,1.0f);
        pad.led(5,1.0f);
        pad.led(6,1.0f);
    }
    
    Vector2D avenger_pos = _avenger.get_pos(); // this part also check avenger-stone collision, and here gets the avenger's and the stone current location 
    // read current ball attributes
    Vector2D stone_pos = _stone.get_pos();
    
    if (stone_pos.x <= 1) {  // if the stone leaves the left side of the screen, replace it to the right side for the next stone retrieval
        _stone.reset();
    }

    if (
        (avenger_pos.y + 4 >= stone_pos.y - _stone_size) && (avenger_pos.y + 2 <= stone_pos.y + _stone_size) // check if avenger and the stone are overlap
        && //bottom
        (avenger_pos.x + 10 >= stone_pos.x) && (avenger_pos.x + 10 <= stone_pos.x + 1)  //right
    ) {
        // if it has, fix position and reflect x velocity
        _avenger.add_score();
        pad.tone(500.0,0.25);
        wait(0.25);
        pad.tone(1000.0,0.25);
        wait(0.25);
    }
}

int Engine::get_score() { // get score for the main function to determine the ending
    return _avenger.get_score();
}