#include "mbed.h"
#include "N5110.h"
#include "spaceship.h"
#include "Joystick.h"
#include "weapons.h"
#include "Asteroids.h"
#include "SongPlayer.h"

/**Main

@brief Game Implementation
@brief Revision 2.5

@author Taurai Mbakada
@date   02 May 2017

**/

//Game Objects//

N5110 lcd(PTC10 , PTC9, PTC0, PTC7, PTD2, PTD1, PTC11);
Joystick joystick(PTB10, PTB11, PTC16);

//Prototypes//
Spaceship ship;
bulletType1 bullet;
Asteroids astroid;
Gamepad pad;


/* Original/Reset vectors for astroid positions and velocity */
static const Vector2D og_pos[8]    = {{0,24},{80,24},{42,0},{42,48},{70,48},{72,0},{10,0},{10,48}};
static const Vector2D og_vel[8]    = {{1, 0},{-1, 0},{ 0,1},{ 0,-1},{-1,-1},{-1,1},{1,1} ,{1,-1 }};

/* Actual/Runtime vectors for astroid positions and velocity */
static Vector2D astr_pos[8]       = {{0,24},{80,24},{42,0},{42,48},{70,48},{72,0},{10,0},{10,48}};
static Vector2D astr_vel[8]       = {{1, 0},{-1, 0},{ 0,1},{ 0,-1},{-1,-1},{-1,1},{1,1} ,{1,-1 }};

/* Function prototypes */
static void game_init( void );               //intialise game
static void gameRender( void );              //Render game objects
static void bullet_run( Direction joy_dir ); //Bullet track
static void astroid_run( void );             //Astroid trcks
static void mainMenu();                      //Menu Initiliastion
static void lcdControl(void);                //Brightness

/* Miscellaneous */
static void bullet_out_of_bounds( void );
static void bullet_hits( void );
static void asteroid_hits( void );
static void welcome( void );
static void origins (void);
static void gameOver(Gamepad &pad, N5110 &lcd);
static void printScore(N5110 &lcd);


/******************************************************************************
* Name: main( void)
*
* Description: Entry point for the game. Including welcome screen,
* game instructions and scores.
*
* Parameters: void
*
* Return: should never return
*
*******************************************************************************/
int main()
{
    int fps = 12;       // frames per second
    // printf("frames per second set");
    Direction joy_dir;  // read the direction of the joysick

    /* Initialise peripherals and game components */
    game_init();
    //printf("peripherals set");

    welcome(); //initiate welcome screen
    //printf("welcome screen displayed");

    mainMenu();

    origins(); //initiate instrcutions
    //printf("instructions shown");

    gameRender(); //render game components
    //printf("game rendered");
    wait(1.0f/fps);

    lcd.refresh();


    /* Ininfinite loop for executing game commands */
    while (true) {

        lcd.inverseMode();

        lcdControl();
        //printf("potentiometer changed")

        /* Fetch current joystick direction*/
        joy_dir = joystick.get_direction();

        /* Change orientation of ship based on joystick */
        ship.update_ship_pos( joy_dir );

        /* Check if we need to add/reset bullets*/
        bullet_run(joy_dir);

        /* Astroid spawn engine */
        astroid_run();

        /* Update all bullets position */
        for(char i = 0; i < 8; i++) {
            bullet.update( i );
        }

        /* Check for any bullet-astroid hits*/
        bullet_hits();

        /* Check for any ship-astroid hits */
        asteroid_hits();

        /* Render lcd game components */
        gameRender();
        wait(1.0f/fps);
    }
}

/******************************************************************************
* Name: static gameRender( void)
*
* Description: Game rendering function. Ship is put in place, along
* with the astroids and bullets.
*
* Parameters: void
*
* Return: void
*
*******************************************************************************/
static void gameRender( void )
{
    lcd.clear();

    ship.draw(lcd);
    // printf("ship drawn");

    printScore(lcd);
    //printf("score displayed");

    /* Add all astroids to the lcd canvas */
    for(char i = 0; i < 8; i++) {
        astroid.draw(lcd, i);
    }
    // printf("astroids spawning");

    /* Add all bullets to the lcd canvas */
    for(char i = 0; i < 8; i++) {
        bullet.draw(lcd, i);
    }
    //printf("bullets in place");

    lcd.refresh();
}

/*******************************************************************************
* Name: static void welcome(void)
*
* Description: Welcome screen. First interaction with the user. Includes music
* player.
*
* Parameters: void
*
* Return: void
******************************************************************************/
static void welcome(void)
{

    lcd.printString("  S.T.A.R  ",8,0);
    lcd.printString("SpecialTactics", 0,2);
    lcd.printString("  and Rescue  ", 0,3);
    lcd.printString(" Press Start ", 0,5);
    lcd.refresh();
    //printf("Strings Created");


    float note[18]= {1568.0,1396.9,1244.5,1244.5,1396.9,1568.0,1568.0,1568.0,1396.9,
                     1244.5,1396.9,1568.0,1396.9,1244.5,1174.7,1244.5,1244.5, 0.0
                    };
    float duration[40]= {0.48,0.24,0.72,0.48,0.24,0.48,0.24,0.24,0.24,
                         0.24,0.24,0.24,0.24,0.48,0.24,0.48,0.48, 0.0
                        };
    //printf("music initialised");

    // setup instance of new SongPlayer class, mySpeaker using pin 26
    // the pin must be a PWM output pin
    SongPlayer mySpeaker(PTC10);
    // Start song and return once playing starts
    mySpeaker.PlaySong(note,duration);

    // wait flashing LEDs until start button is pressed
    while ( pad.check_event(Gamepad::START_PRESSED) == false) {
        pad.leds_on();
        wait(0.1);
        pad.leds_off();
        wait(0.1);
    }
    //printf("start button pressed");
}


/******************************************************************************
* Name: static game_init ( void)
*
* Description: Game initialisation function
*
* Parameters: void
*
* Return: void
*
*******************************************************************************/
static void game_init(void)
{
    /* Initialise game components and peripherals */
    lcd.init();
    joystick.init();
    pad.init();
    bullet.init();
    astroid.init();
}

/******************************************************************************
* Name: static bullet_run( Direction joy_dir )
*
* Description: This function is the main loop/engine for
*              spawning bullets from the ship. Bullets are meant to be created
               behind the ship.
*
* Parameters: Direction joy_dir - Current joystick direction
*
* NOTE: POINT IN DIRECTION OF ASTROID USING THE JOYSTICK TO SHOOT BULLET. 
*       JOYSICK MUST BE KEPT IN PLACE.
*
* Return: void
*******************************************************************************/
static void bullet_run( Direction joy_dir )
{
    Vector2D pos;
    Vector2D vel;
    static char bull_id = 0;
    //printf("bullet id set to zero");

    /* If button is pressed, add bullet depending on ship orientation */
    if( pad.check_event(Gamepad::A_PRESSED) == true) {
        switch(joy_dir) {
            case N:
                pos.x = 43;
                pos.y = 22;
                vel.x =  0;
                vel.y = -1;
                break;
            case NE:
                pos.x = 48;
                pos.y = 24;
                vel.x =  1;
                vel.y = -1;
                break;
            case E:
                pos.x = 48;
                pos.y = 24;
                vel.x =  1;
                vel.y =  0;
                break;
            case SE:
                pos.x = 48;
                pos.y = 24;
                vel.x =  1;
                vel.y =  1;
                break;
            case S:
                pos.x = 43;
                pos.y = 24;
                vel.x =  0;
                vel.y =  1;
                break;
            case SW:
                pos.x = 42;
                pos.y = 18;
                vel.x = -1;
                vel.y =  1;
                break;
            case W:
                pos.x = 40;
                pos.y = 24;
                vel.x = -1;
                vel.y =  0;
                break;
            case NW:
                pos.x = 44;
                pos.y = 32;
                vel.x = -1;
                vel.y = -1;
                break;
            default:
                pos.x = 34;
                pos.y = 26;
                vel.x =  0;
                vel.y = -1;
                break;
        }
        pad.tone(3000.0,0.1);

        /* Lets add a new bullet to the canvas*/
        bullet.set_velocity(vel, bull_id);
        bullet.set_pos(pos, bull_id);
        //printf("bullet created");
        bull_id++;
        /* Don't exceed 8 bullets */
        if(bull_id > 7) {
            bull_id = 0;
        }
    }
    /* Check if any bullet is now outside lcd limits */
    bullet_out_of_bounds();
    //printf("bullet out of bounds");
}

/******************************************************************************
* Name: static bullet_out_of_bounds( void )
*
* Description: This function is the main loop/engine for
*              spawning astroids
*
* Parameters: void
*
* Return: void
*
*****************************************************************************/
static void astroid_run( void )
{
    /* Velocity update-rate for astroids, the bigger the value the slower the astroid */
    const char cons_slow_down[8] = {8,10,8,15,9,11,14,20};
    static char slow_down[8]     = {8,10,8,15,9,11,14,20};

    /* Add astroids to the canvas */
    for(char i = 0; i < 8; i++) {
        if(slow_down[i] == 0) {
            astroid.set_velocity(astr_vel[i], i);
            astroid.set_pos(astr_pos[i], i);
            astroid.update(i);
            astr_pos[i] = astroid.get_pos(i);

            slow_down[i] = cons_slow_down[i];
        }
        //printf("astroids slowed down");

        /* Reset the astroid to original position if now outside bounds */
        if(
            (astr_pos[i].x > 80 ) ||
            ( astr_pos[i].x < 1 ) ||
            ( astr_pos[i].y > 49 )||
            ( astr_pos[i].y < 1 )
        ) {
            astr_pos[i] = og_pos[i];
            astr_vel[i] = og_vel[i];
            //printf("astroid position reset");
        }
        slow_down[i]--;
    }
}

/*******************************************************************************
* Name: static bullet_out_of_bounds( void )
*
* Description: This function checks if a bullet is outside
*              bounds/limits of lcd
*
* Parameters: void
*
* Return: void
*
*******************************************************************************/
static void bullet_out_of_bounds( void )
{
    Vector2D pos;
    Vector2D vel = {0,0};

    /* Reset/clear bullet if it's now outside lcd bounds */
    for(char i = 0; i < 8; i++) {

        /* Get bullet current position */
        pos= bullet.get_pos(i);

        /* Check if now outside lcd limits */
        if(
            (pos.x > 80 ) ||
            ( pos.x < 1 ) ||
            ( pos.y > 49 )|| ( pos.y < 1 )
        ) {
            pos.x = 44;
            pos.y = 24;
            bullet.set_pos(pos, i);
            bullet.set_velocity(vel, i);
        }
        //printf("bullet reset");
    }
}

/*****************************************************************************
* Name: static void bullet_hits( void )
*
* Description: This function checks for collisions between bullets and astroids*
*
* Parameters: void
*
* Return: void
*
******************************************************************************/
static void bullet_hits( void )
{
    Vector2D bull_pos;
    Vector2D ast_pos;

    /* Check if any bullet on the canvas has hit an astroid */
    for(char i = 0; i < 8; i++) {

        /* Get current position of bullet */
        bull_pos = bullet.get_pos(i);
        //printf("bullet position set");

        for(char idx = 0; idx < 8; idx++) {

            /* Get current position of astroid */
            ast_pos = astroid.get_pos(idx);
            //printf("astroid position set");

            ship.get_userScore();
            //printf("user score retrieved");

            /* Check for collision by creating a hit box plus/minus 1 unit around
            the x and y coordinates ofthe astroids*/

            if
            (
                ((bull_pos.x == ast_pos.x ) && (bull_pos.y  == ast_pos.y))||
                (((bull_pos.x + 1) == ast_pos.x ) && (bull_pos.y  == ast_pos.y))||
                ((bull_pos.x == ast_pos.x ) && ((bull_pos.y + 1)  == ast_pos.y))||
                (((bull_pos.x - 1) == ast_pos.x ) && (bull_pos.y  == ast_pos.y))||
                ((bull_pos.x == ast_pos.x ) && ((bull_pos.y - 1)  == ast_pos.y))||
                (((bull_pos.x - 1)== ast_pos.x ) && ((bull_pos.y - 1)  == ast_pos.y))||
                (((bull_pos.x + 1) == ast_pos.x ) && ((bull_pos.y + 1) == ast_pos.y))||
                (((bull_pos.x - 1) == ast_pos.x ) && ((bull_pos.y + 1)  == ast_pos.y))||
                (((bull_pos.x + 1) == ast_pos.x ) && ((bull_pos.y - 1)  == ast_pos.y))

            ) {
                //printf("collision observed");
                astr_pos[idx] = og_pos[0];
                astr_vel[idx] = og_vel[0];
                //printf("astroids returned");

                ship.add_userScore();
                //printf("score updated");

                /* Returning the bullet after a collision */
                Direction joy_dir;
                Vector2D pos;
                Vector2D vel;
                static char bull_id = 0;
                switch(joy_dir) {
                    default:
                        pos.x = 44;
                        pos.y = 22;
                        vel.x =  0;
                        vel.y = 0;
                        break;
                }
                bullet.set_velocity(vel, bull_id);
                bullet.set_pos(pos, bull_id);
                bull_id++;

                if(bull_id > 7) {
                    bull_id = 0;
                }
                //printf("bullet returned");

                lcd.refresh();
            }
        }
    }
}

/**************************************************************************
*  Name: static void gameOver(Gamepad &pad, N5110 &lcd)
*
* Description: This function creates the game over sequence that is displayed
*              when any one of the astroids collides with the ship.
*
* Parameters: &pad and &lcd
*
* Return: Does not return
*
*
***************************************************************************/
static void gameOver(Gamepad &pad, N5110 &lcd)
{
    lcd.clear();
    lcd.normalMode();
    //First Messgae
    lcd.printString("Mission Failed",0,2);
    lcd.printString("  Commandor!  ",0,3);
    lcd.refresh();
    wait(2);
    //printf("first message displayed");

    pad.tone(250.0,0.1); //end sequence tone initiated
    wait(0.4);
    pad.tone(240.0,0.1);
    wait(0.3);
    pad.tone(150.0,0.1);
    wait(0.2);
    pad.tone(50.0,0.1);
    //printf("end sequence tone played");

    lcd.clear();
    pad.leds_on();

    int score = ship.get_userScore();
    //sprintf("%2d", score);

    // print to LCD
    char buffer1[21];
    sprintf(buffer1,"%2d",score);
    //Second message
    lcd.printString("Score"  ,28,2);
    lcd.printString(buffer1,36,3);
    //Users score determines users overall performance rank
    while(1)  {
        if(score <= 50) {
            lcd.printString("  Try Harder!  ",0,0);
            //printf("Try Harder");
        } else if(score > 50 && score <= 100) {
            lcd.printString("Valient Effort!",0,0);
            //printf("Valient Effort");
        } else {
            lcd.printString("Medal of Honor!",0,0);
            //printf("Medal of Honor");
        }
        lcd.refresh();
        wait(1.5);
    }
}

/*****************************************************************************
* Name: static void asteroid_hits( void )
*
* Description: This function checks for collisions
*              between ship and astroids
*
* Parameters: void
*
* Return: void
*
******************************************************************************/
static void asteroid_hits( void )
{
    Vector2D ship_pos;
    Vector2D ast_pos;

    ship_pos = ship.get_pos();
    //printf("ship postition updated");

    for(char idx = 0; idx < 8; idx++) {
        ast_pos = astroid.get_pos(idx);

        /*checking for collisions between the astroids and the ship. Bullets are
        given a wider impact area*/
        if
        (
            ((ast_pos.x == ship_pos.x ) || (ast_pos.y== ship_pos.y))||
            (((ast_pos.x + 2) == ship_pos.x) || (ast_pos.y == ship_pos.y))||
            ((ast_pos.x  == ship_pos.x) || ((ast_pos.y + 2) == ship_pos.y))||
            (((ast_pos.x - 2) == ship_pos.x) || (ast_pos.y == ship_pos.y))||
            ((ast_pos.x  == ship_pos.x) ||((ast_pos.y - 2) == ship_pos.y))||
            (((ast_pos.x - 2) == ship_pos.x) || ((ast_pos.y - 2)== ship_pos.y))||
            (((ast_pos.x + 2) == ship_pos.x) || ((ast_pos.y + 2) == ship_pos.y))
        ) {
            //printf("Collision observed");
            pad.tone (500, 0.5);
            gameOver(pad, lcd);
        }
    }
}

/******************************************************************************
* Name: countDown() and origins()
*
* Description: Gives the background to the game and allows the user the oppoturnity
*              to read the instrctions.
*
* Parameters: void
*
* Return: void
*
*******************************************************************************/
static void countDown(void)
{
    lcd.clear();

    lcd.printString("######",26,0);
    lcd.printString("#",56,1);
    lcd.printString("######",26,2);
    lcd.printString("#",56,3);
    lcd.printString("######",26,4);
    wait(1);
    lcd.refresh();
    //printf("3");

    lcd.clear();
    lcd.printString("######",26,0);
    lcd.printString("#",56,1);
    lcd.printString("######",26,2);
    lcd.printString("#",26,3);
    lcd.printString("######",26,4);
    wait(1);
    lcd.refresh();
    ////printf("2");

    lcd.clear();
    lcd.printString("######",26,0);
    lcd.printString("#",56,1);
    lcd.printString("######",26,2);
    lcd.printString("#",26,3);
    lcd.printString("######",26,4);
    wait(1);
    lcd.refresh();
    //printf("1");
}

static void origins(void)
{
    lcd.clear();

    //Background - pt1
    lcd.printString("Incoming Alert",0,0);
    lcd.printString("  Approaching ",0,2);
    lcd.printString(" Astroid Belt! ", 0,3);
    lcd.printString(" Avoid at all ",0,4);
    lcd.printString("    costs!   ",0,5);
    lcd.refresh();
    wait(3);
    //pt2
    lcd.clear();
    lcd.printString("  Commander,  ",0,1);
    lcd.printString("Weapons AreHot",0,2);
    lcd.printString("  I repeat ", 0,3);
    lcd.printString("WeaponsareHot!",0,4);
    lcd.refresh();
    wait(2);

    countDown();
}

/*****************************************************************************
* Name: printScore(N5110 &lcd)
*
* Description: Printing the scores.
*
* Parameters: &lcd only.
*
* Return: Does not return.
*
******************************************************************************/
static void printScore(N5110 &lcd)
{
    // get scores from paddles
    int score = ship.get_userScore();

    // print to LCD
    char buffer1[14];
    sprintf(buffer1,"%2d",score);
    lcd.printString(buffer1,20,0);
}

/*****************************************************************************
* Name: Menu Initialisation
*
* Description: Creating a menu for the user.
*
* Parameters: void
*
* Return: Does not return.
*
******************************************************************************/
static void subMenu(void)
{
    lcd.clear();
    lcd.printString("  >Main Menu<  ",0,0);
    lcd.printString("@  START - X ",0,2);
    lcd.printString("@  HELP - Y ",0,3);
    lcd.refresh();
}

static void mainMenu(void)
{
    /* Inital screen the user interacts with */
    subMenu();

    while ( pad.check_event(Gamepad::X_PRESSED) == false) {
        // settings
        if (pad.check_event(Gamepad::Y_PRESSED) == true) {
            lcd.clear();
            pad.tone(200,0.1);
            lcd.printString("  >> HELP <<   ",0,0);
            lcd.printString("@SHOOT - A  ",0,2);
            lcd.printString("@Bullet-JoyDirc",0,3);
            lcd.printString("@Brightness- P",0,4);
            lcd.printString("@Back - Back",0,5);
            lcd.refresh();
        }

        lcd.clear();
        // back function
        if (pad.check_event(Gamepad::BACK_PRESSED) == true) {
            subMenu();
            pad.tone(200,0.1);

        }
        lcd.clear();
    }
}

/*****************************************************************************
* Name: lcdControl()
*
* Description: This function is used to alter the brightness of the lcd display.
*              The user can change the brightness by altering the angle of the
*              potentiometer.
*
*              Used once the game officially starts.
*
* Parameters: void
*
* Return: Does not return.
*
******************************************************************************/
static void lcdControl(void)
{
    float LCD_backlight = pad.read_pot();
    lcd.setBrightness(LCD_backlight);
    lcd.refresh();
}
