#ifndef PLAYER_H
#define PLAYER_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"

// Constants
#define GRAVITY 1
#define JUMPSPEED -5

/** N5110 Class
@brief A class that handles all the player's actions, as well as stores
@brief his position and velocity

@brief Revision 1.0

@author Aleksejs Sladzevskis
@date   3rd May 2017
*/

class Player
{
public:

    /** Constructor */
    Player();

    /** Destructor */
    ~Player();

    /** Initialise the player
    *@param intVector2D player's starting position
    */
    void init(intVector2D XY);

    /** Draw player
    *@param lcd output
    */
    void draw(N5110 &lcd);

    /** Update player's data
    *@param jx magnitude of joystick's displacement in x-axis
    *@param pc serial out for debugging
    *@param lvl current level number
    *@param theSwitch state of the switch (on/off)
    *@param pad gamepad
    *@param soundOn current sound setting (on/off)
    */
    void update(float jx, Serial &pc, int lvl, bool theSwitch, Gamepad &pad, bool soundOn);

    /** Get player's position
    *@returns a 2D integer vector of player's position
    */
    intVector2D get_pos();

    /** Set player's position
    *@param p player's current position
    */
    void set_pos(intVector2D p);

    /** Get player's velocity
    *@returns a 2D integer vector of player's position
    */
    Vector2D get_velocity();

    /** Set player's velocity
    *@param v player's current velocity
    */
    void set_velocity(Vector2D v);

    /** Reset player's velocity to zero */
    void reset_velocity();

    /** Applies constant acceleration to player's velocity in y-axis
    *@param  lvl current level
    *@param  theSwitch current state of the switch (on/off)
    */
    void apply_gravity(int lvl, bool theSwitch);

    /** Projects the player up with a set speed
    *@param  lvl current level
    *@param  theSwitch current state of the switch (on/off)
    */
    void apply_jump(int lvl, bool theSwitch);

    /** An array that stores the collision points of player's bounding box */
    intVector2D collisionPoint[8];

    /** Keeps track of whether the player released the jump key */
    bool _didReleaseJumpKey;

    /** Keeps track of whether the player is in the air */
    bool _isJumping;

    /** Keeps track of whether the player pressed the jump key */
    bool _didPressJump;

    /** Keeps track of whether the player is dead or alive */
    bool _isDead;


private:
    
    int _x;
    int _y;
    Vector2D _velocity;

};
#endif