#ifndef MenuEngine_H
#define MenuEngine_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"
#include "GameObject.h"
#include "Player.h"
#include "Animation.h"
#include "Level.h"

/** MenuEngine class
@brief Ties together the interactions between the player and the game objects.
@brief Similar to GameEngine class, however modified to remove unnecessary elements
@brief and bring in some different elements.

@brief Revision 1.0

@author Aleksejs Sladzevskis
@date   4th May 2017
*/

class MenuEngine
{
public:

    /** Constructor */
    MenuEngine();

    /** Destructor */
    ~MenuEngine();

    /** Initialise the menu engine
    *@param currentLevel feed in the value of current level to display in the menu
    */
    void init(int currentLevel);

    /** Read gamepad input
    *@param pad gamepad object
    *@param pc serial out for debugging
    */
    void read_input(Gamepad &pad, Serial &pc);

    /** Process inputs, apply menu's physics and check for setting changes
    *@param pad gamepad object
    *@param pc serial out for debugging
    *@param lcd output
    */
    void update(Gamepad &pad, Serial &pc, N5110 &lcd);

    /** Draw the updated objects on the screen
    *@param lcd output
    */
    void draw(N5110 &lcd, Serial &pc);

    /** Check whether the player has reached the exit
    *@param pad gamepad object
    *@param lcd output
    *@param pc serial out for debugging
    */
    void check_finish(Gamepad &pad, N5110 &lcd, Serial &pc);

    /** A flag to exit the menu and start launch the game engine*/
    bool gameStarted;

    /** Stores the currently selected level */
    int selectedLevel;

    /** Stores the currently selected sound setting (on/off) */
    bool soundIsOn;

private:

    // private functions
    void load_objects();
    void draw_objects(N5110 &lcd);
    void draw_object(N5110 &lcd, int i);
    void draw_note(N5110 &lcd);
    void print_selected_level(N5110 &lcd, Serial &pc);

    void check_for_collisions(Gamepad &pad, Serial &pc, N5110 &lcd);
    void get_player_data();
    void discrete_contact_solver(Gamepad &pad);
    void speculative_contact_solver(Gamepad &pad);
    void handle_speculative_contacts(int direction);
    bool contact_detected(int direction);
    void determine_contact_type();
    void handle_contacts();
    void update_player_data();

    // Private objects
    Level level;
    Animation animate;
    Player _player;
    GameObject _worldObjects[MENUOBJECTCOUNT];

    // Menu entrance and exit positions
    intVector2D starting_position;
    intVector2D finish_position;

    // A variable for storing joystick's X axis input
    float _jx;

    // ===================================================
    // Collision Detection
    // ===================================================

    // Iteration variable
    int o;

    // Maximum number of iterations
    static const int iterations = 3;

    // Flags to specify what kind of collision has occurred
    bool contactX;
    bool contactYbottom;
    bool contactYtop;

    // Speculative testing variables
    float vectorLength;
    int segments;

    // The move player was supposed to make before collision detection was applied
    Vector2D originalMove;

    // A vector used for collision detection processing
    Vector2D projectedMove;

    // Corrected move that the player is to make
    Vector2D nextMove;

    // Vectors for player's position and velocity
    intVector2D player_position;
    Vector2D player_velocity;

};
#endif