#include "MenuEngine.h"

// Constructor
MenuEngine::MenuEngine()
{

}

// Destructor
MenuEngine::~MenuEngine()
{

}

void MenuEngine::init(int currentLevel)
{
    // Menu is level 0
    level.number = 0;

    // Load all the object data from the level object into the array
    load_objects();

    // Level's starting and finish positions are relative to the entrance/finish tubes
    starting_position.x = _worldObjects[7].position.x + 2;
    starting_position.y = _worldObjects[7].position.y + 2;
    finish_position.x = _worldObjects[8].position.x - 3;
    finish_position.y = _worldObjects[8].position.y + 2;

    // Initialise the player with calculated starting position
    _player.init(starting_position);

    // If it's the first tiem in the menu and the level hasn't been set yet - set it to 1
    // Otherwise set it to the last level played in the game
    selectedLevel = currentLevel < 1? 1 : currentLevel;

    // Sound is on by default
    soundIsOn = true;

    // Clear flag that launches the game
    gameStarted = false;

    // Trigger spawn animation during the next draw() cycle
    animate.spawnCycle = 0;

}

void MenuEngine::load_objects()
{
    // Temporary array to load object data into
    GameObject worldObjects[MENUOBJECTCOUNT];

    // Loop through every object and load relevant data
    for (int i = 0; i < MENUOBJECTCOUNT; i++)
    {
        worldObjects[i].type = level.load_type(i);
        worldObjects[i].position.x = level.load_x(i);
        worldObjects[i].position.y = level.load_y(i);
        worldObjects[i].width = level.load_width(i);
        worldObjects[i].height = level.load_height(i);
    }

    // Store the loaded data into the engine class array
    for (int i = 0; i < MENUOBJECTCOUNT; i++) {
		_worldObjects[i] = worldObjects[i];
    };
}

void MenuEngine::update(Gamepad &pad, Serial &pc, N5110 &lcd)
{
    _player.update(_jx,pc,level.number, false, pad, soundIsOn);
    check_for_collisions(pad, pc, lcd);
}

void MenuEngine::read_input(Gamepad &pad, Serial &pc)
{
    // Store the joystick displacement on the x-axis
    _jx = pad.get_mapped_coord().x;

    // Check whether the player pressed jump and set the flag accordingly
    _player._didPressJump = pad.check_event(Gamepad::A_PRESSED);
}

void MenuEngine::draw(N5110 &lcd, Serial &pc)
{
    // Draw menu objects
    lcd.clear();
    draw_objects(lcd);
    draw_note(lcd);

    // Print out the menu title
    lcd.printString("Main Menu", 16, 1);

    print_selected_level(lcd,pc);
    animate.spawn(lcd, starting_position);
    _player.draw(lcd);
    lcd.refresh();
}

void MenuEngine::print_selected_level(N5110 &lcd, Serial &pc)
{
    // Determine currently selected level and print out the appropriate number
    // in the level selection box
    if (selectedLevel == 1) {
        lcd.printString("1", 28, 3);
    }
    else if (selectedLevel == 2) {
        lcd.printString("2", 27, 3);
    }
    else if (selectedLevel == 3) {
        lcd.printString("3", 27, 3);
    }
    else if (selectedLevel == 4) {
        lcd.printString("4", 27, 3);
    }
    else if (selectedLevel == 5) {
        lcd.printString("5", 27, 3);
    }
    else pc.printf("%d\n", selectedLevel); // debug
}

void MenuEngine::draw_objects(N5110 &lcd)
{
    // Go through each object in the array in draw it
    for (int i = 0; i < MENUOBJECTCOUNT; i++)
    {
        draw_object(lcd, i);
    };
}

void MenuEngine::draw_object(N5110 &lcd, int i)
{
    // If the object is a platfrom, draw a filled rectangle
    if (_worldObjects[i].type == 0)
    {
        lcd.drawRect(_worldObjects[i].position.x, _worldObjects[i].position.y, _worldObjects[i].width, _worldObjects[i].height, FILL_BLACK);
    }
    // If the object is a menu box, draw a transparent rectangle
    else if (_worldObjects[i].type == 7)
    {
        lcd.drawRect(_worldObjects[i].position.x, _worldObjects[i].position.y, _worldObjects[i].width, _worldObjects[i].height, FILL_TRANSPARENT);
    }
}

void MenuEngine::draw_note(N5110 &lcd)
{
    // Set the appropriate pixels to draw a note
    lcd.setPixel(_worldObjects[5].position.x + 5, _worldObjects[5].position.y + 10);
    lcd.setPixel(_worldObjects[5].position.x + 5, _worldObjects[5].position.y + 11);

    lcd.setPixel(_worldObjects[5].position.x + 6, _worldObjects[5].position.y + 9);
    lcd.setPixel(_worldObjects[5].position.x + 6, _worldObjects[5].position.y + 10);
    lcd.setPixel(_worldObjects[5].position.x + 6, _worldObjects[5].position.y + 11);
    lcd.setPixel(_worldObjects[5].position.x + 6, _worldObjects[5].position.y + 12);

    lcd.setPixel(_worldObjects[5].position.x + 7, _worldObjects[5].position.y + 9);
    lcd.setPixel(_worldObjects[5].position.x + 7, _worldObjects[5].position.y + 10);
    lcd.setPixel(_worldObjects[5].position.x + 7, _worldObjects[5].position.y + 11);
    lcd.setPixel(_worldObjects[5].position.x + 7, _worldObjects[5].position.y + 12);

    lcd.setPixel(_worldObjects[5].position.x + 8, _worldObjects[5].position.y + 2);
    lcd.setPixel(_worldObjects[5].position.x + 8, _worldObjects[5].position.y + 3);
    lcd.setPixel(_worldObjects[5].position.x + 8, _worldObjects[5].position.y + 4);
    lcd.setPixel(_worldObjects[5].position.x + 8, _worldObjects[5].position.y + 5);
    lcd.setPixel(_worldObjects[5].position.x + 8, _worldObjects[5].position.y + 6);
    lcd.setPixel(_worldObjects[5].position.x + 8, _worldObjects[5].position.y + 7);
    lcd.setPixel(_worldObjects[5].position.x + 8, _worldObjects[5].position.y + 8);
    lcd.setPixel(_worldObjects[5].position.x + 8, _worldObjects[5].position.y + 9);
    lcd.setPixel(_worldObjects[5].position.x + 8, _worldObjects[5].position.y + 10);
    lcd.setPixel(_worldObjects[5].position.x + 8, _worldObjects[5].position.y + 11);

    lcd.setPixel(_worldObjects[5].position.x + 9, _worldObjects[5].position.y + 3);
    lcd.setPixel(_worldObjects[5].position.x + 9, _worldObjects[5].position.y + 4);

    lcd.setPixel(_worldObjects[5].position.x + 10, _worldObjects[5].position.y + 4);
    lcd.setPixel(_worldObjects[5].position.x + 10, _worldObjects[5].position.y + 5);

    // Cross the symbol out if the sound is off
    if (!soundIsOn)
    {
        lcd.drawLine(46, 20, 61, 34, 1);
        lcd.drawLine(46, 34, 61, 20, 1);
    }
}

void MenuEngine::check_for_collisions(Gamepad &pad, Serial &pc, N5110 &lcd)
{
    // Read player's position and velocity and store them inside engine class
    get_player_data();

    // Assume there is a collision to trigger the check
    contactX = contactYbottom = contactYtop = true;

    // Loop while there is a contact detected or maximum number of iterations has been reached
    for (int iteration = 0; iteration < iterations && (contactX || contactYbottom || contactYtop); iteration++)
	{
		// Get the amount of X and Y movement expected by the player this frame
		nextMove.x = player_velocity.x;
        nextMove.y = player_velocity.y;

		// No collisions found yet
		contactX = contactYbottom = contactYtop = false;

		// Store the original final expected movement of the player so we can
		// see if it has been modified due to a collision or potential collision later
		originalMove = nextMove;

        // Iterate over each object with the player's bounding box until a collision is found
		for ( o = 0; o < MENUOBJECTCOUNT && !contactX && !contactYbottom && !contactYtop && !_player._isDead; o++)
		{
            // Check for speculative contacts
            speculative_contact_solver(pad);

            /*
            if (nextMove.y != originalMove.y) {
                pc.printf("Original move = %f,%f\nCorrected move = %f,%f\n\n", originalMove.x, originalMove.y, nextMove.x, nextMove.y);
            }*/

            // Check for discrete contacts
            discrete_contact_solver(pad);

            // If there was a contact, determine it's type
            determine_contact_type();
		}
        // Resolve any detected contacts
        handle_contacts();
	}
    // Update player's data using corrected position and velocity
    update_player_data();
}

void MenuEngine::get_player_data()
{
    // Read player's position and velocity
    player_position.x = _player.get_pos().x;
    player_position.y = _player.get_pos().y;
    player_velocity.x = _player.get_velocity().x;
    player_velocity.y = _player.get_velocity().y;
}

void MenuEngine::speculative_contact_solver(Gamepad &pad)
{
    // ================================================================================
    // Speculative contacts section
    //
    // We will traverse along the movement vector of the player from his/her current
    // position until the final position for the frame to check if any geometry lies
    // in the way. If so, the vector is adjusted to end at the geometry's intersection
    // with the player's movement vector. This eliminates the possibility of passing
    // through objects that lie along the player's path.
    // ================================================================================

    // We will test the four possible directions of player movement individually
    // dir: 0 = top, 1 = bottom, 2 = left, 3 = right
    for (int dir = 0; dir < 4; dir++)
    {
        // Skip the test if the expected direction of movement makes the test irrelevant
        if (dir == 0 && nextMove.y > 0) continue;
        if (dir == 1 && nextMove.y < 0) continue;
        if (dir == 2 && nextMove.x > 0) continue;
        if (dir == 3 && nextMove.x < 0) continue;

        // Our current position along the anticipated movement vector of the player this frame
        projectedMove.x = 0;
        projectedMove.y = 0;

        // Calculate the length of the movement vector using Pythagoras
        vectorLength = sqrt((nextMove.x * nextMove.x) + (nextMove.y * nextMove.y));
        segments = 0;

        // Advance along the vector until it intersects with some geometry
        // or we reach the end
        while (!contact_detected(dir) && segments < vectorLength)
        {
            projectedMove.x += nextMove.x / vectorLength;
            projectedMove.y += nextMove.y / vectorLength;
            segments++;
        }

        // If an intersection occurred
        if (segments < vectorLength)
        {
            handle_speculative_contacts(dir);
        }
    }
}

void MenuEngine::handle_speculative_contacts(int direction)
{
    // Apply correction for over-movement
    if (segments > 0)
    {
        // Move one segment back
        projectedMove.x -= nextMove.x / vectorLength;
        projectedMove.y -= nextMove.y / vectorLength;
    }

    // If the player jumps into level selection box - increment current level
    // Go back to level 1 after level 5 is reached
    if (o == 4) {
        selectedLevel = (selectedLevel > 5? 1 : (selectedLevel + 1));
    }
    // If the player jumps into sound setting box - turn the sound on/off
    else if (o == 5) {
        soundIsOn = !soundIsOn;
    }

    // Adjust the X or Y component of the vector depending on
    // which direction we are currently testing
    if (direction >= 2 && direction <= 3) {
        nextMove.x = projectedMove.x;
    }
    if (direction >= 0 && direction <= 1) {
        nextMove.y = projectedMove.y;
    }
}

void MenuEngine::discrete_contact_solver(Gamepad &pad)
{
    // ================================================================================
    // Discrete contact solver
    //
    // Here we look for existing collisions and nudge the player in the opposite
    // direction until the collision is solved. The purpose of iteration is because
    // the correction may cause collisions with other pieces of geometry.
    // ================================================================================

    // dir: 0 = top, 1 = bottom, 2 = left, 3 = right
    for (int dir = 0; dir < 4; dir++)
    {
        // Skip the test if the expected direction of movement makes the test irrelevant
        if (dir == 0 && nextMove.y > 0) continue;
        if (dir == 1 && nextMove.y < 0) continue;
        if (dir == 2 && nextMove.x > 0) continue;
        if (dir == 3 && nextMove.x < 0) continue;

        // Make a working copy of the expected move in appropriate direction
        projectedMove.x = (dir >= 2? nextMove.x : 0);
        projectedMove.y = (dir <  2? nextMove.y : 0);

        // Traverse backwards in X or Y (but not both at the same time)
        // until the player is no longer colliding with the geometry
        while (contact_detected(dir))
        {
            // Move one pixel into the opposite direction, away from the object
            if (dir == 0) {
                projectedMove.y++;
            }
            if (dir == 1) {
                projectedMove.y--;
            }
            if (dir == 2) {
                projectedMove.x++;
            }
            if (dir == 3) {
                projectedMove.x--;
            }

            // If the player jumps into level selection box - increment current level
            // Go back to level 1 after level 5 is reached
            if (o == 4) {
                selectedLevel = (selectedLevel > 4? 1 : (selectedLevel + 1));
            }
            // If the player jumps into sound setting box - turn the sound on/off
            else if (o == 5) {
                soundIsOn = !soundIsOn;
            }
        }

        // If direction being checked is horizontal - update horizontal vector
        if (dir >= 2 && dir <= 3)
        {
            nextMove.x = projectedMove.x;
        }

        // If direction being checked is vertical - update vertical vector
        if (dir >= 0 && dir <= 1)
        {
            nextMove.y = projectedMove.y;
        }
    }
}

void MenuEngine::determine_contact_type()
{
    // Detect what type of contact has occurred based on a comparison of
    // the original expected movement vector and the new one
    if (nextMove.y > originalMove.y && originalMove.y < 0)
    {
        contactYtop = true;
    }

    if (nextMove.y < originalMove.y && originalMove.y > 0)
    {
        contactYbottom = true;
    }

    // A slight error margin to account for the fact that floating numbers
    // tend to "float" (not required for y-axis since there is no floating input there)
    if (fabs(nextMove.x - originalMove.x) > 0.01f)
    {
        contactX = true;
    }
}

void MenuEngine::handle_contacts()
{
    // If a contact has been detected, apply the re-calculated movement vector
    // and disable any further movement this frame (in either X or Y as appropriate)
    if (contactYbottom || contactYtop)
    {
        // Apply corrected vector and limit the movement
        player_position.y += nextMove.y;
        player_velocity.y = 0;

        // If the player has hit the floor - reset jumping flag
        if (contactYbottom)
        {
            _player._isJumping = false;
        }
    }

    // If there is a contact with a wall - limit movement in that direction
    // and reset vertical speed to allow the player to grab on to the walls
    if (contactX)
    {
        player_position.x += nextMove.x;
        player_velocity.x = 0;
        player_velocity.y = 0;
        nextMove.y = 0;
    }
}

void MenuEngine::update_player_data()
{
    // Update player's position and velocity
    player_position.x += player_velocity.x;
    player_position.y += player_velocity.y;
    _player.set_velocity(player_velocity);
    _player.set_pos(player_position);
}

bool MenuEngine::contact_detected(int direction)
{
    // Check the coordinates of players collision points that are relevant to the direction of movement
    // against the space that the object being check occupies. If there is any intersection return TRUE.
    if (_worldObjects[o].containsPoint(static_cast<int>(_player.collisionPoint[direction*2].x + player_position.x + projectedMove.x),
                                        static_cast<int>(_player.collisionPoint[direction*2].y + player_position.y + projectedMove.y))
        || _worldObjects[o].containsPoint(static_cast<int>(_player.collisionPoint[direction*2+1].x + player_position.x + projectedMove.x),
                                            static_cast<int>(_player.collisionPoint[direction*2+1].y + player_position.y + projectedMove.y)))
    {
        return true;
    }

    else
    {
        return false;
    }
}

void MenuEngine::check_finish(Gamepad &pad, N5110 &lcd, Serial &pc)
{
    // Check whether the finish position next to the exit tube is reached
    if (_player.get_pos().x == finish_position.x && _player.get_pos().y == finish_position.y)
    {
        // Set flag that exits the menu and launches the game
        gameStarted = true;

        // Redraw the objects
        draw(lcd,pc);

        // Animate the player exiting through the exit
        animate.finish(lcd, _player.get_pos(), pad, soundIsOn);

        // Reset his position and velocity for next level
        _player.set_pos(starting_position);
        _player.reset_velocity();
    }
}