#ifndef GAMEENGINE_H
#define GAMEENGINE_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"
#include "Player.h"
#include "GameObject.h"
#include "Level.h"
#include "Animation.h"

/** GameEngine class
@brief Ties together the interactions between the player and the game objects,
@brief as well as determines the level objectives.

@brief Revision 1.0

@author Aleksejs Sladzevskis
@date   3rd May 2017
*/

class GameEngine
{

public:

    /** Constructor */
    GameEngine();

    /** Destructor */
    ~GameEngine();

    /** Initialise the game engine
    *@param startingLevel the number of the level to load
    *@param sound current sound setting (on/off)
    *@param pad gamepad object
    */
    void init(int startingLevel, bool sound, Gamepad &pad);

    /** Read gamepad input
    *@param pad gamepad object
    *@param pc serial out for debugging
    */
    void read_input(Gamepad &pad, Serial &pc);

    /** Process inputs, apply game's physics and check for level objectives completion
    *@param pad gamepad object
    *@param pc serial out for debugging
    *@param lcd output
    */
    void update(Gamepad &pad, Serial &pc, N5110 &lcd);

    /** Draw the updated objects on the screen
    *@param lcd output
    */
    void draw(N5110 &lcd);

    /** Check whether the player has reached the exit
    *@param pad gamepad object
    *@param lcd output
    */
    void check_finish(Gamepad &pad, N5110 &lcd);

    /** Lever number getter function */
    int get_level();

    /** This is set low when the player exits the game */
    bool isActive;


private:

    // Private Functions
    void load_level_objects();
    void announce_level(N5110 &lcd);
    void print_level_number(N5110 &lcd);

    void check_for_collisions(Gamepad &pad, Serial &pc, N5110 &lcd);
    void get_player_data();
    void discrete_contact_solver(Gamepad &pad);
    void speculative_contact_solver(Gamepad &pad);
    void handle_speculative_contacts(int direction);
    bool contact_detected(int direction);
    void determine_contact_type();
    void handle_contacts();
    void update_player_data();
    void apply_level_specifics(Gamepad &pad, Serial &pc);

    void level_3_specifics(Gamepad &pad, Serial &pc);
    void level_4_specifics(Gamepad &pad, Serial &pc);
    void level_5_specifics(Gamepad &pad, Serial &pc);

    void check_death(Gamepad &pad, N5110 &lcd);
    void check_switch(Gamepad &pad);

    void check_limits(Gamepad &pad, Serial &pc);

    void draw_objects(N5110 &lcd);
    void draw_object(N5110 &lcd, int i);

    void reset_coins();
    void activate_coins();

    // Private objects
    Player _player;
    Animation animate;
    Level level;
    GameObject _worldObjects[OBJECTCOUNT];

    // Level entrance and exit positions
    intVector2D starting_position;
    intVector2D finish_position;

    // Flag for entering a new level
    bool newLevelEntered;

    // Variable used for animation
    char frames;

    // A variable for storing joystick's X axis input
    float _jx;

    // In-game button state
    bool switchIsOn;

    // Door state
    bool doorIsOpened;

    // Sound setting
    bool soundIsOn;

    // ===================================================
    // Collision Detection
    // ===================================================

    // Iteration variable
    int o;

    // Maximum number of iterations
    static const int iterations = 3;

    // Flags to specify what kind of collision has occurred
    bool contactX;
    bool contactYbottom;
    bool contactYtop;

    // Speculative testing variables
    float vectorLength;
    int segments;

    // The move player was supposed to make before collision detection was applied
    Vector2D originalMove;

    // A vector used for collision detection processing
    Vector2D projectedMove;

    // Corrected move that the player is to make
    Vector2D nextMove;

    // Vectors for player's position and velocity
    intVector2D player_position;
    Vector2D player_velocity;

};

#endif
