// Credits:
// Game Engine design based on Pong Engine Example code by Craig Evans
// Collision detection code based on https://katyscode.wordpress.com/2013/01/18/2d-platform-games-collision-detection-for-dummies/

#include "GameEngine.h"
#include <math.h>

// Constructor
GameEngine::GameEngine()
{

}

// Destructor
GameEngine::~GameEngine()
{

}


void GameEngine::init(int startingLevel, bool sound, Gamepad &pad)
{
    // Set current level to the value passed from the menu
    level.number = startingLevel;

    // Load all the object data from the level object into the array
    load_level_objects();

    // Level's starting and finish positions are relative to the entrance/finish tubes
    starting_position.x = _worldObjects[5].position.x + 2;
    starting_position.y = _worldObjects[5].position.y + 2;
    finish_position.x = _worldObjects[7].position.x - 3;
    finish_position.y = _worldObjects[7].position.y + 2;

    // Initialise the player with calculated starting position
    _player.init(starting_position);

    // Set default positions of the switch and the door
    doorIsOpened = false;
    switchIsOn = false;

    // Trigger spawn animation during the next draw() cycle
    animate.spawnCycle = 0;

    // Turn sound on or off depending on what setting was selected in the menu
    soundIsOn = sound;

    // This need to remain true for the game to continue running
    isActive = true;

    // Trigger the level announcement
    newLevelEntered = true;

    // Reset the flag for BACK button in case it was pressed while the menu was active
    pad.reset_event(Gamepad::BACK_PRESSED);
}

void GameEngine::read_input(Gamepad &pad, Serial &pc)
{
    // Store the joystick displacement on the x-axis
    _jx = pad.get_mapped_coord().x;

    // Check whether the player pressed jump and set the flag accordingly
    _player._didPressJump = pad.check_event(Gamepad::A_PRESSED);

    // If Y is pressed - commit suicide
    _player._isDead = (pad.check_event(Gamepad::Y_PRESSED)? true : _player._isDead);

    // Quit back to main menu if player pressed BACK
    isActive = (pad.check_event(Gamepad::BACK_PRESSED)? false : true);

    //pc.printf("%d\n", _player._didPressJump);
    //pc.printf("%f\n", _jx);
}

void GameEngine::update(Gamepad &pad, Serial &pc, N5110 &lcd)
{
    announce_level(lcd);
    pad.leds_off();
    apply_level_specifics(pad,pc);
    _player.update(_jx,pc,level.number, switchIsOn, pad, soundIsOn);
    check_death(pad, lcd);
    check_for_collisions(pad, pc, lcd);
    check_switch(pad);
    check_limits(pad, pc);
}

void GameEngine::announce_level(N5110 &lcd)
{
    // If the player has entered a new level display the announcement
    if (newLevelEntered)
    {
        lcd.clear();

        // Draw a frame
        lcd.drawLine(0, 0, 83, 0, 1);
        lcd.drawLine(0, 0, 0, 47, 1);
        lcd.drawLine(0, 47, 83, 47, 1);
        lcd.drawLine(83, 0, 83, 47, 1);

        print_level_number(lcd);
        lcd.refresh();
        wait(2.0);

        // Disable the announcement until the next level is reached
        newLevelEntered = false;

        // Trigger spawn animation during next draw() execution
        animate.spawnCycle = 0;
    }
}

void GameEngine::print_level_number(N5110 &lcd)
{
    // Read current level number and print out the appropriate string
    if (level.number == 1) {
        lcd.printString("Level 1", 23, 2);
    }
    else if (level.number == 2) {
        lcd.printString("Level 2", 23, 2);
    }
    else if (level.number == 3) {
        lcd.printString("Level 3", 23, 2);
    }
    else if (level.number == 4) {
        lcd.printString("Level 4", 23, 2);
    }
    else if (level.number == 5) {
        lcd.printString("Level 5", 23, 2);
    }
}

void GameEngine::draw(N5110 &lcd)
{
    lcd.clear();
    draw_objects(lcd);
    animate.spawn(lcd, starting_position);
    _player.draw(lcd);
    lcd.refresh();
}

void GameEngine::draw_objects(N5110 &lcd)
{
    // Go through each object in the array in draw it
    for (int i = 0; i < OBJECTCOUNT; i++)
    {
        draw_object(lcd, i);
    };
}

void GameEngine::draw_object(N5110 &lcd, int i)
{
    // If the object is a platfrom, draw a filled rectangle
    if (_worldObjects[i].type == 0)
    {
        lcd.drawRect(_worldObjects[i].position.x, _worldObjects[i].position.y, _worldObjects[i].width, _worldObjects[i].height, FILL_BLACK);
    }

    // If the object is horizontal spikes, draw a set of horizontal rectangles
    else if (_worldObjects[i].type == 1)
    {
        for (int y = 0; y < _worldObjects[i].height; y = y + 2)
        {
            lcd.drawRect(_worldObjects[i].position.x, _worldObjects[i].position.y + y, _worldObjects[i].width, 1, FILL_BLACK);
        }
    }

    // If the object is vertical spikes, draw a set of vertical rectangles
    else if (_worldObjects[i].type == 2)
    {
        for (int x = 0; x < _worldObjects[i].width; x = x + 2)
        {
            lcd.drawRect(_worldObjects[i].position.x + x, _worldObjects[i].position.y, 1, _worldObjects[i].height, FILL_BLACK);
        }
    }

    // If the object is a door and it's closed, draw a vertical line
    else if (_worldObjects[i].type == 3 && doorIsOpened == false)
    {
        lcd.drawRect(_worldObjects[i].position.x, _worldObjects[i].position.y, _worldObjects[i].width, _worldObjects[i].height, FILL_BLACK);
    }

    // If the object is a switch, draw its appropriate state
    else if (_worldObjects[i].type == 4)
    {

        if (switchIsOn)
        {
            lcd.drawLine(_worldObjects[i].position.x, _worldObjects[i].position.y, _worldObjects[i].position.x + 2, _worldObjects[i].position.y - 2, 1);
        }

        else
        {
            lcd.drawLine(_worldObjects[i].position.x, _worldObjects[i].position.y, _worldObjects[i].position.x - 2, _worldObjects[i].position.y - 2, 1);
        }
    }

    // If the object is an uncollected coin - draw it
    else if (_worldObjects[i].type == 5)
    {
        lcd.setPixel(_worldObjects[i].position.x + 1, _worldObjects[i].position.y);
        lcd.setPixel(_worldObjects[i].position.x + 1, _worldObjects[i].position.y + 2);
        lcd.setPixel(_worldObjects[i].position.x, _worldObjects[i].position.y + 1);
        lcd.setPixel(_worldObjects[i].position.x + 2, _worldObjects[i].position.y + 1);
    }
}

void GameEngine::check_for_collisions(Gamepad &pad, Serial &pc, N5110 &lcd)
{
    // Read player's position and velocity and store them inside engine class
    get_player_data();

    // Assume there is a collision to trigger the check
    contactX = contactYbottom = contactYtop = true;

    // Loop while there is a contact detected or maximum number of iterations has been reached
    for (int iteration = 0; iteration < iterations && (contactX || contactYbottom || contactYtop); iteration++)
	{
		// Get the amount of X and Y movement expected by the player this frame
		nextMove.x = player_velocity.x;
        nextMove.y = player_velocity.y;

		// No collisions found yet
		contactX = contactYbottom = contactYtop = false;

		// Store the original final expected movement of the player so we can
		// see if it has been modified due to a collision or potential collision later
		originalMove = nextMove;

        // Iterate over each object with the player's bounding box until a collision is found
		for ( o = 0; o < OBJECTCOUNT && !contactX && !contactYbottom && !contactYtop && !_player._isDead; o++)
		{
            // Don't check collisions against an open door or a collected coin
            if (_worldObjects[o].type == 3 && doorIsOpened == true) continue;
            if (_worldObjects[o].type == 6) continue;

            // Check for speculative contacts
            speculative_contact_solver(pad);

            /*
            if (nextMove.y != originalMove.y) {
                pc.printf("Original move = %f,%f\nCorrected move = %f,%f\n\n", originalMove.x, originalMove.y, nextMove.x, nextMove.y);
            }*/

            // If the coin was collected along the way no point of checking discrete collisions
            if (_worldObjects[o].type == 6) continue;

            // Check for discrete contacts
            discrete_contact_solver(pad);

            // If there was a contact, determine it's type
            determine_contact_type();
		}
        // Resolve any detected contacts
        handle_contacts();
	}
    // Update player's data using corrected position and velocity
    update_player_data();
}

void GameEngine::load_level_objects()
{
    // Temporary array to load object data into
    GameObject worldObjects[OBJECTCOUNT];

    // Loop through every object and load relevant data
    for (int i = 0; i < OBJECTCOUNT; i++)
    {
        worldObjects[i].type = level.load_type(i);
        worldObjects[i].position.x = level.load_x(i);
        worldObjects[i].position.y = level.load_y(i);
        worldObjects[i].width = level.load_width(i);
        worldObjects[i].height = level.load_height(i);
    }

    // Store the loaded data into the engine class array
    for (int i = 0; i < OBJECTCOUNT; i++) {
		_worldObjects[i] = worldObjects[i];
    };
}

void GameEngine::get_player_data()
{
    // Read and record player's position and velocity
    player_position.x = _player.get_pos().x;
    player_position.y = _player.get_pos().y;
    player_velocity.x = _player.get_velocity().x;
    player_velocity.y = _player.get_velocity().y;
}

void GameEngine::speculative_contact_solver(Gamepad &pad)
{
    // ================================================================================
    // Speculative contacts section
    //
    // We will traverse along the movement vector of the player from his/her current
    // position until the final position for the frame to check if any geometry lies
    // in the way. If so, the vector is adjusted to end at the geometry's intersection
    // with the player's movement vector. This eliminates the possibility of passing
    // through objects that lie along the player's path.
    // ================================================================================

    // We will test the four possible directions of player movement individually
    // dir: 0 = top, 1 = bottom, 2 = left, 3 = right
    for (int dir = 0; dir < 4; dir++)
    {
        // Skip the test if the expected direction of movement makes the test irrelevant
        if (dir == 0 && nextMove.y > 0) continue;
        if (dir == 1 && nextMove.y < 0) continue;
        if (dir == 2 && nextMove.x > 0) continue;
        if (dir == 3 && nextMove.x < 0) continue;

        // Our current position along the anticipated movement vector of the player this frame
        projectedMove.x = 0;
        projectedMove.y = 0;

        // Calculate the length of the movement vector using Pythagoras
        vectorLength = sqrt((nextMove.x * nextMove.x) + (nextMove.y * nextMove.y));
        segments = 0;

        // If the collision against coins are checked - no need to modify movement vector
        if (_worldObjects[o].type == 5 || _worldObjects[o].type == 6)
        {
            // Traverse along the whole movement vector one segment by one
            for (segments = 0; segments < vectorLength; segments++)
            {
                if (contact_detected(dir))
                {
                    // If there is a contact with any type of coin - mark it collected
                    _worldObjects[o].type = 6;

                    // And play a sound if it is turned on
                    if (soundIsOn)
                    {
                        pad.tone(783.99,0.02f);
                        wait(0.02f);
                        pad.tone(1046.50,0.05f);
                    }
                    // No need to make any further checks
                    break;
                }
            }
        }

        else
        {
            // Advance along the vector until it intersects with some geometry
            // or we reach the end
            while (!contact_detected(dir) && segments < vectorLength)
            {
                projectedMove.x += nextMove.x / vectorLength;
                projectedMove.y += nextMove.y / vectorLength;
                segments++;
            }

            // If an intersection occurred
            if (segments < vectorLength)
            {
                handle_speculative_contacts(dir);
            }
        }
    }
}

void GameEngine::handle_speculative_contacts(int direction)
{
    // Apply correction for over-movement
    if (segments > 0)
    {
        // Move one segment back
        projectedMove.x -= nextMove.x / vectorLength;
        projectedMove.y -= nextMove.y / vectorLength;
    }

    // Adjust the X or Y component of the vector depending on
    // which direction we are currently testing
    if (direction >= 2 && direction <= 3) {
        nextMove.x = projectedMove.x;
    }
    if (direction >= 0 && direction <= 1) {
        nextMove.y = projectedMove.y;
    }

    // Mark the player dead if he has hit the spikes
    if (_worldObjects[o].type == 1 || _worldObjects[o].type == 2)
    {
        _player._isDead = true;
    }
}

void GameEngine::discrete_contact_solver(Gamepad &pad)
{
    // ================================================================================
    // Discrete contact solver
    //
    // Here we look for existing collisions and nudge the player in the opposite
    // direction until the collision is solved. The purpose of iteration is because
    // the correction may cause collisions with other pieces of geometry.
    // ================================================================================

    // dir: 0 = top, 1 = bottom, 2 = left, 3 = right
    // No point of iterating if player is dead
    for (int dir = 0; dir < 4 && !_player._isDead; dir++)
    {
        // Skip the test if the expected direction of movement makes the test irrelevant
        if (dir == 0 && nextMove.y > 0) continue;
        if (dir == 1 && nextMove.y < 0) continue;
        if (dir == 2 && nextMove.x > 0) continue;
        if (dir == 3 && nextMove.x < 0) continue;

        // Make a working copy of the expected move in appropriate direction
        projectedMove.x = (dir >= 2? nextMove.x : 0);
        projectedMove.y = (dir <  2? nextMove.y : 0);

        // Traverse backwards in X or Y (but not both at the same time)
        // until the player is no longer colliding with the geometry
        while (contact_detected(dir))
        {
            // Again, no adjustments needed if its a coin
            if (_worldObjects[o].type == 5 || _worldObjects[o].type == 6)
            {
                // Do mark it collected though
                _worldObjects[o].type = 6;

                // Play sound if it's on
                if (soundIsOn)
                {
                    pad.tone(783.99f,0.02f);
                    wait(0.02f);
                    pad.tone(1046.50,0.05f);
                }
                // Exit the loop to avoid corrections
                break;
            }

            // Move one pixel into the opposite direction, away from the object
            if (dir == 0) {
                projectedMove.y++;
            }
            if (dir == 1) {
                projectedMove.y--;
            }
            if (dir == 2) {
                projectedMove.x++;
            }
            if (dir == 3) {
                projectedMove.x--;
            }

            // Mark the player dead if he has hit the spikes
            if (_worldObjects[o].type == 1 || _worldObjects[o].type == 2) {
                _player._isDead = true;
            }
        }

        // If direction being checked is horizontal - update horizontal vector
        if (dir >= 2 && dir <= 3)
        {
            nextMove.x = projectedMove.x;
        }

        // If direction being checked is vertical - update vertical vector
        if (dir >= 0 && dir <= 1)
        {
            nextMove.y = projectedMove.y;
        }
    }
}

void GameEngine::determine_contact_type()
{
    // Detect what type of contact has occurred based on a comparison of
    // the original expected movement vector and the new one
    if (nextMove.y > originalMove.y && originalMove.y < 0)
    {
        contactYtop = true;
    }

    if (nextMove.y < originalMove.y && originalMove.y > 0)
    {
        contactYbottom = true;
    }

    // A slight error margin to account for the fact that floating numbers
    // tend to "float" (not required for y-axis since there is no floating input there)
    if (fabs(nextMove.x - originalMove.x) > 0.01f)
    {
        contactX = true;
    }
}

void GameEngine::handle_contacts()
{
    // If a contact has been detected, apply the re-calculated movement vector
    // and disable any further movement this frame (in either X or Y as appropriate)
    if (contactYbottom || contactYtop)
    {
        // Apply corrected vector and limit the movement
        player_position.y += nextMove.y;
        player_velocity.y = 0;

        // Allow jumping when you stand on the ceiling after the switch is flipped in levels 2 and 3
        if (contactYtop && (level.number == 2 || level.number == 3) && switchIsOn)
        {
            _player._isJumping = false;
        }

        // If player is on the floor on any other level - reset jumping flag
        else if (contactYbottom)
        {
            _player._isJumping = (((level.number == 2 || level.number == 3) && switchIsOn)? _player._isJumping : false);
        }
    }

    // If there is a contact with a wall - limit movement in that direction
    // and reset vertical speed to allow the player to grab on to the walls
    if (contactX)
    {
        player_position.x += nextMove.x;
        player_velocity.x = 0;
        player_velocity.y = 0;
        nextMove.y = 0;
    }
}

void GameEngine::update_player_data()
{
    // Update player's position and velocity
    player_position.x += player_velocity.x;
    player_position.y += player_velocity.y;
    _player.set_velocity(player_velocity);
    _player.set_pos(player_position);
}

bool GameEngine::contact_detected(int direction)
{
    // Check the coordinates of players collision points that are relevant to the direction of movement
    // against the space that the object being check occupies. If there is any intersection return TRUE.
    if (_worldObjects[o].containsPoint(static_cast<int>(_player.collisionPoint[direction*2].x + player_position.x + projectedMove.x),
                                        static_cast<int>(_player.collisionPoint[direction*2].y + player_position.y + projectedMove.y))
        || _worldObjects[o].containsPoint(static_cast<int>(_player.collisionPoint[direction*2+1].x + player_position.x + projectedMove.x),
                                            static_cast<int>(_player.collisionPoint[direction*2+1].y + player_position.y + projectedMove.y)))
    {
        return true;
    }

    else
    {
        return false;
    }
}

void GameEngine::check_switch(Gamepad &pad)
{
    // Only check if switch is not yet turned on
    // There is a sweet spot that is three pixels wide that activates the switch
    if (!switchIsOn
        &&  player_position.y == _worldObjects[38].position.y - 2
        &&  (      player_position.x == _worldObjects[38].position.x - 2
                || player_position.x == _worldObjects[38].position.x - 1
                || player_position.x == _worldObjects[38].position.x
            )
        )
    {
        // Set the switch flag on
        switchIsOn = true;

        // Play a sound if it is on
        if (soundIsOn) pad.tone(784.0f, 0.1f);

        // In the first two levels the switch opens the door
        if (level.number < 3) doorIsOpened = true;

        // In level four it makes the coins appear
        if (level.number == 4) activate_coins();
    }
}

void GameEngine::check_death(Gamepad &pad, N5110 &lcd)
{
    if (_player._isDead)
    {
        // Register player's death position
        intVector2D position = _player.get_pos();

        // Animate player's death at that position
        for (int frame = 0; frame < 4; frame++)
        {
            animate.death(lcd, position, frame);
            draw_objects(lcd);
            lcd.refresh();
            wait(0.06);
        }

        // Play a sad C minor chord to acknowledge the death :(
        if (soundIsOn)
        {
        pad.tone(392.00f, 0.15f);
        wait(0.15);
        pad.tone(311.13f, 0.15f);
        wait(0.15);
        pad.tone(261.63f, 0.15f);
        }

        // Revive the player at the level's starting position
        _player.set_pos(starting_position);

        // Reset velocity
        _player.reset_velocity();

        // Clear the death flag
        _player._isDead = false;

        // Trigger spawn animation during next draw() cycle
        animate.spawnCycle = 0;

        // Clear flags for the switch and the door as the new level has been entered
        switchIsOn = false;
        doorIsOpened = false;

        // Reset the picked up coins on level 4
        if (level.number == 4) reset_coins();
    }
}

void GameEngine::check_limits(Gamepad &pad, Serial &pc)
{
    // Kill player if he gets outside of the screen as a result of a bug
    // and print an appropriate message
    if (player_position.x < 0 || player_position.x > 83)
    {
        _player._isDead = true;
        pc.printf("Out of bounds (x-axis)");
    }

    if (player_position.y < 0 || player_position.y > 47)
    {
        _player._isDead = true;
        pc.printf("Out of bounds (y-axis)");
    }
}

void GameEngine::check_finish(Gamepad &pad, N5110 &lcd)
{
    // Check whether the finish position next to the exit tube is reached
    if (_player.get_pos().x == finish_position.x && _player.get_pos().y == finish_position.y)
    {
        // Redraw the player at the finish position
        lcd.clear();
        draw_objects(lcd);
        _player.draw(lcd);

        // Animate the player exiting through the exit
        animate.finish(lcd, _player.get_pos(), pad, soundIsOn);

        // Reset his position and velocity for next level
        _player.set_pos(starting_position);
        _player.reset_velocity();

        // Reset the flags for next level
        doorIsOpened = false;
        switchIsOn = false;
        _player._isJumping = true;

        // Set the flag to trigger next level announcement
        newLevelEntered = true;

        // Finishing last level exits back to main menu
        if (level.number == 5) {
            _worldObjects[37].height = 7;
            isActive = false;
            newLevelEntered = false;
        }

        // Increase the level number unless it's the last level
        if (level.number < 5) level.number++;
    }
}

void GameEngine::apply_level_specifics(Gamepad &pad, Serial &pc)
{
    // Some events and objects are specific to certain levels or events
    if (level.number == 3 && switchIsOn && !doorIsOpened) level_3_specifics(pad,pc);
    else if (level.number == 4 && switchIsOn) level_4_specifics(pad,pc);
    else if (level.number == 5 && switchIsOn) level_5_specifics(pad,pc);
}

void GameEngine::level_3_specifics(Gamepad &pad, Serial &pc)
{
    // The player essentially has to find the secret spot to open the door
    // There are 6 LEDs to guide him : 3 on the left and 3 on the right
    // A red, yellow and green on each side
    // They hint based on "Warmer/Colder" principle
    // Left 3 LEDs are X-axis hint, right 3 LEDs - y-axis
    // When player is far away from the secret spot on that axis: red LED is on
    // When the player gets close: yellow lights up instead
    // Green LED means he found the exact coordinate on that axis
    // Making both LEDs green and hence finding the secret spot triggers the door opening
    // The secret coordinates are (80,9)

    // X-coordinate hint
    if (player_position.x < 62)
    {
        pad.led(3, 1); // red LED
    }
    else if (player_position.x != 80)
    {
        pad.led(2, 1); // yellow LED
    }
    else if (player_position.x == 80)
    {
        pad.led(1, 1); // green LED
    }

    // Y-coordinate hint
    if (player_position.y > 24)
    {
        pad.led(6, 1); // red LED
    }
    else if (player_position.y != 9)
    {
        pad.led(5, 1); // yellow LED
    }
    else if (player_position.y == 9)
    {
        pad.led(4, 1); // green LED
    }

    // Unlock door if secret spot discovered
    if (player_position.x == 80 && player_position.y == 9)
    {
        doorIsOpened = true;

        // Play a note three times to hint to the player that something happened
        // (if the sound is on)
        if (soundIsOn)
        {
            pad.tone(784.0f, 0.1);
            wait(0.1);
            pad.tone(784.0f, 0.1);
            wait(0.1);
            pad.tone(784.0f, 0.1);
        }
    }
}

void GameEngine::reset_coins()
{
    // Mark all coins collected to get them out of view
    for (int i = 41; i < 51; i++)
    {
        _worldObjects[i].type = 6;
    }
}

void GameEngine::activate_coins()
{
    // Mark all coins uncollected to bring them into view
    for (int i = 41; i < 51; i++)
    {
        _worldObjects[i].type = 5;
    }
}

void GameEngine::level_4_specifics(Gamepad &pad, Serial &pc)
{
    // When the player flips the switch - 10 coins appear
    // He must collect them all tho open the door

    // Assume the door is opened
    doorIsOpened = true;

    // Loop through every coin and if at least one is uncollected - close the door
    for (int i = 41; i < 51; i++)
    {
        if (_worldObjects[i].type == 5)
        {
            doorIsOpened = false;
            break;
        }
    }
}

void GameEngine::level_5_specifics(Gamepad &pad, Serial &pc)
{
    // To open the door on the final level the player must think outside the box
    // The door is opened by rotating the analogue potentiometer on the PCB

    // Read and store the potentiometer value
    float pot_value = pad.read_pot();

    // Gradually open the door as the potentiometer is rotated
    if (pot_value < 0.143f)
    {
        _worldObjects[37].height = 7;
    }
    else if (pot_value < 0.286f)
    {
        _worldObjects[37].height = 6;
    }
    else if (pot_value < 0.286f)
    {
        _worldObjects[37].height = 5;
    }
    else if (pot_value < 0.429f)
    {
        _worldObjects[37].height = 4;
    }
    else if (pot_value < 0.572f)
    {
        _worldObjects[37].height = 3;
    }
    else if (pot_value < 0.715f)
    {
        _worldObjects[37].height = 2;
    }
    else if (pot_value < 0.858f)
    {
        _worldObjects[37].height = 1;
    }
    else
    {
        _worldObjects[37].height = 0;
    }
}

int GameEngine::get_level()
{
    // return the current level number
    int temp = level.number;
    return temp;
}