/** Pokemon Game
@brief Game based on the main series Pokemon games by the pokemon company and Gamefreak.
@brief whole game is a battle simulator, where you can level up your companion by battling other wild pokemon
@brief You can view your pokemons stats and change the screen brightness from the menus.

@brief Revision 2

@author Aaron Lad
@date   2nd May 2017

@code
*/

#include "mbed.h"
#include "Gamepad.h"
#include "N5110.h"
#include "Pokemon.h"
#include "Images.h"
#include "Bitmap.h"
#include "Script.h"

//rewritten code to implement new/better ideas. part of the code is same as before but lots of changes were needed.
//tried several attempts to Use a GameEngine class rather than using the main.cpp but ultimately was not able to

//-------------------------------- objects -------------------------------------
N5110 lcd(PTC9,PTC0,PTC7,PTD2,PTD1,PTC11);
Gamepad pad;
Pokemon pk;
Images sp;
Script sc;
//--------------------------- Structs/Typedefs ---------------------------------
struct joyInput {
    Direction d; //direction of joystick to navigate menu's
};

typedef enum State {START, PARTNER, MENU, FIGHT, POKEMON, SETTINGS} Gamepage;
//assigning names to different states

void init();
void drawStart();
void select(int x, int y);
void deselect(int _x, int _y);
void balls();
float drawPartner();
float partnerChoice(int choice);
void choice(int p);
float drawMenu();
void drawFight();
void drawPoke();
void settings();
void ledCycle();
float YesNo();
void battleStats();
float rndm(int RA);
//-----------------------------Functions----------------------------------------

int main()
{
    init();  //initialising screen and gamepad
    drawStart();  //screen shows startup screen
    Gamepage state = START;  //go to start state
/*-----------------------GAME LOOP START-------------------------------
Uses a game loop to move through the states of the game system
*/
    while(1) {
        if (state == START) {

            state = PARTNER;//change next state to Partner state
            //printf("state = Partner");
        } else if (state == PARTNER) { //choosing your partner

            int partner = drawPartner();
            int correct = partnerChoice(partner);
            if( correct == 2) {
                state = START; //change next state to Start state
            } else if(correct == 1) {
                choice(partner);
                lcd.refresh();
                state = MENU; //change next state to Menu state
                //printf("state = MENU")
            }
        }

        else if (state == FIGHT) { //fight with wild pokemon

            drawFight();
            state = MENU;//change next state to Menu state
            //printf("state = MENU")
        } else if (state == SETTINGS) {  //settings, screen brightness

            settings();
            state = MENU;//change next state to Menu state
            //printf("state = MENU")

        } else if (state == MENU) { // main menu

            wait(1.0);
            int box = drawMenu();
            if (box == 0) {
                state = FIGHT;//change next state to Fight state
                //printf("state = FIGHT")
                lcd.clear();
            } else if (box ==1) {
                state = POKEMON;//change next state to Pokemon state
                //printf("state = POKEMON")
                lcd.clear();
            } else if (box ==2) {
                state = SETTINGS;//change next state to Settings state
                //printf("state = SETTINGS")
                lcd.clear();
            }
            lcd.refresh();
            wait(2.0);

        } else if (state == POKEMON) { //check partner pokemon stats

            state = MENU;//change next state to Menu state
            //printf("state = MENU")
            drawPoke();
        }
    }
}

/** initialise screen and pad
 *powers up the display and configures the gamepad
 *sets the screen brightness to 100%
*/
void init()
{
    //need to initialise gamepad and LCD screen
    pad.init();
    lcd.init();
    //set screen brightness to full by default
    lcd.setBrightness(1.0);
}

/** LED Cycle
 *turns the leds of the game pad on and off in sequence
 *using 1 to turn on LED and 0 to turn off
*/
void ledCycle()
{
    pad.led(1,1); //cycle through LED's on gamepad
    pad.led(6,1);
    wait(0.5);
    pad.led(2,1);
    pad.led(5,1);
    wait(0.5);
    //all LEDS on
    pad.leds_on();
    wait(0.5);
    pad.led(1,0);
    pad.led(6,0);
    wait(0.5);
    pad.led(2,0);
    pad.led(5,0);
    wait(0.5);
    pad.leds_off();
    //all LEDs off
    wait(0.5);
}

/** Draw the Start state
 *Draws the startup screen for the game
 *uses a check for the start button to start the game loop
*/
void  drawStart()
{
    //load script from Library
    sc.Start(lcd);
    //detect when Start button is pressed
    while( pad.check_event(Gamepad::START_PRESSED) == false) { // until start is pressed, this loop cycles
        ledCycle();
        /*if(pad.check_event(Gamepad::START_PRESSED) == true){
            printf("START Pressed")};*/
    }
    
    wait(0.5);
    //reset gamepad to counteract button bounce
    pad.init();
    //clear screen
    lcd.clear();
}
/** draw select Cursor
 *Draws the cursor used for selecting options 
 *refreshes screen every time called to make sure cursors are shown
 * @param x - the column number (0-83)
 * @param y - the row number (0-42)
*/
void select(int x,int y)
{
    //Print 3 pixel cursor to screen
    lcd.setPixel(x,y);
    lcd.setPixel(x-1,y+1);
    lcd.setPixel(x-1,y-1);
    lcd.refresh();
}
/** removes select Cursor
 *clears the cursor used for selecting options 
 *refreshes screen every time called to make sure cursors are removed from the screen
 * @param x - the column number (0-83)
 * @param y - the row number (0-42)
*/
void deselect(int _x,int _y)
{
    //remove 3 pixel cursor from screen buffer
    lcd.setPixel(_x,_y,false);
    lcd.setPixel(_x-1,_y+1,false);
    lcd.setPixel(_x-1,_y-1,false);
    lcd.refresh();
}
/** Print the Ball sprites
 *Calls the bitmap from the Images class and prints the Pokeball sprite to the buffer in the required loactions
 *refreshes screeen when called to make sure Pokeballs are displayed
*/
void balls() //show the pokeballs on the screen for player to choose from
{
    //call sprite library to print balls
    sp.ball(lcd, pad);
    lcd.refresh();
    wait(1.0);
}

/** Draw Partner state
 *Allows player to select a pokeball which would contain the Pokemon used as their starter
 *uses the joystick to move a cursor and the A button to select which ball
 *Returns the value of the chosen ball
*/
float drawPartner() //choice of pokeball
{
    lcd.clear();
    //set variable for Pokemon choice
    int offset = 1;
    //detect for A button press
    while(pad.check_event(Gamepad::A_PRESSED) == false) {
        /*if(pad.check_event(Gamepad::A_PRESSED) == true){
            printf("A Pressed")};*/
        int _d = pad.get_direction();
        //detect for Joystick pushed right
        if(_d == E && offset == 1||_d == E && offset == 0) {
            offset = offset + 1;
        //detect fo Joystick pushed left
        } else if(_d == W && offset == 1 || _d == W && offset == 2 ) {
            offset = offset - 1;
        }
        //Print and remove cursors
        if(offset == 0) {
            deselect(33,25);
            deselect(57,25);
            select(9,25);
            balls();
            //printf("select0");
        } else if(offset == 1) {
            deselect(9,25);
            deselect(57,25);
            select(33,25);
            balls();
            //printf("select1");
        } else if(offset == 2) {
            deselect(9,25);
            deselect(33,25);
            select(57,25);
            balls();
            //printf("select2");
        }
    }
    wait(0.5);
    //reset gamepad to counteract button bounce
    pad.init();
    return offset;
}

/** Yes/No Choice
 *For use in the partner state to allow players to confirm choice of starters
 *Uses the use of the A and B buttons
*/
float YesNo()
{
    //Call text from Script library
    sc.YN(lcd);
    int end = 0;
    wait(1.0);
    //loop until a choice is made
    while(end == 0) {
        lcd.refresh();
        if(pad.check_event(Gamepad::A_PRESSED)== true) { //choose yes
        /*if(pad.check_event(Gamepad::A_PRESSED) == true){
            printf("A Pressed");}*/
            end = 1;
        } else if(pad.check_event(Gamepad::B_PRESSED)== true) {//choose no
        /*if(pad.check_event(Gamepad::B_PRESSED) == true){
            printf("B Pressed");}*/
            end = 2;
        }
    }
    //use return to load next scren in choice screen
    return end;
}

/** Load partner choice
 *Displays the Starter pokemon from the Chosen pokemon
 *Then passes on the choice of wether the player wants to choose that starter
*/
float partnerChoice(int choice) //shows pokemon inside ball, gives choice to choose that mon
{
    lcd.clear();
    //using choice value to determine pokeball chosen
    if(choice == 0) {
        sp.bulbasaur(lcd, pad); //grass starter
        //printf("Bulbasaur");
    } else if(choice == 1) {
        sp.charmander(lcd, pad);//fire starter
        //printf("Charmander");
    } else if(choice == 2) {
        sp.squirtle(lcd, pad);//water starter
        //printf("Squirtle");
    }
    lcd.refresh();
    wait(3.5);
    lcd.clear();
    //load decision
    int end = YesNo();
    wait(0.5);
    //reset gamepad to counteract button bounce
    pad.init();
    return end;

}

/** Choice setting
 *Sets the type for your chosen pokemon, and display the name of your pokemon
 * @param p - used to select which pokemon was chosen by player
*/
void choice(int p) //sets up class for your chosen starter
{
    lcd.clear();
    //using input to set pokemon used as Starter
    if (p == 0) {
        //Setting Pokemon Type as grass - Bulbasaur
        pk.setType(Grass);
        lcd.printString("You Chose",16,2);
        lcd.printString("Bulbasaur",15,3);
    } else if (p == 1) {
        //Setting Pokemon Type as Fire - Charmander
        pk.setType(Fire);
        lcd.printString("You Chose",16,2);
        lcd.printString("Charmander",15,3);
    } else if (p == 2) {
        //Setting Pokemon Type as Water - Bulbasaur
        pk.setType(Water);
        lcd.printString("You Chose ",16,2);
        lcd.printString("Squirtle",15,3);
    }
}
/** Draw Menu State
 *Prints the menu to the screen
 *uses a cursor to allow user to choose wether they want to fight, check pokemon or go to settings
 *returns the choice of the player
*/

float drawMenu()
{
    //load menu script from script library
    sc.menu(lcd);
    //set initial values for X and Y for select
    int Xo = 6;
    int Yo = 11;
    select(Xo, Yo);
    lcd.refresh();
    wait(1.0);
    int box = 0;
    //Check for A button pressed, loop until it is pressed
    while( pad.check_event(Gamepad::A_PRESSED) == false) {
        /*if(pad.check_event(Gamepad::A_PRESSED) == true){
            printf("A Pressed");}*/
        //get direction from joystick
        int d = pad.get_direction();
        //check for stick moved down
        if (d == S && box == 0 ||d == S && box == 1) {
            //move cursor
            deselect(Xo,Yo);
            Yo = Yo + 16;
            select(Xo,Yo);
            lcd.refresh();
            box = box + 1;
            wait(1.0);
            //check for stick moved up
        } else if (d == N && box == 1||d == N && box == 2) {
            //move cursor
            deselect(Xo,Yo);
            Yo = Yo - 16;
            select(Xo,Yo);
            lcd.refresh();
            box = box - 1;
            wait(1.0);
        }
    }
    //reset gamepad to counteract button bounce
    pad.init();
    return box;
}

/** Randomiser float
 *used to select and return a random number
 *keeps the Pokemon type in correspondence to the pokemon the player comes up against
 * @param RA - random number which chose the wild pokemon you come up against, used to chose pokemon type
 */
float rndm(int RA)
{
    int R = rand() %2;

    if (R ==1) {
    } else {
        if ( RA <= 6 ) {
            R = R;
        } else if ( RA >= 13 ) {
            R = R + 4;
        } else {
            R = R + 2;
        }
    }
    return R;
}

/**Battle stats
 * Displays the Name of the pokemon you come against
 * Also displays name and level of your pokemon
*/
void battleStats()
{
    //taking information from Pokemon class, adding them to strings
    std::string pkn = pk.Name();
    std::string pkl = pk.Level();
    char aa[50];
    char bb[50];
    //copying strings to buffers, so they can be added to buffer
    strncpy(aa, pkn.c_str(), sizeof(aa));
    strncpy(bb, pkl.c_str(), sizeof(bb));
    lcd.printString(aa,2,4);
    lcd.printString(bb,2,5);
    lcd.refresh();
    wait(2.0);
    lcd.clear();
    lcd.printString(aa,2,4);
}
/** Draw fight class
 * randomise 2 integers to get a pokemon name and stats for said pokemon for yours to figth
 * shows HP as battle progresses
 * pulls battle functions and typings from the pokemon class
*/
void drawFight()
{
    lcd.clear();
    //setting randomiser
    srand(time(NULL));
    //Pokemon name list, ordered Fire, Grass then water
    std::string PokemonNames[18] = {"Charmander","Cyndaquil","Torchic","Chimchar","Tepig","Fennekin","Bulbasaur","Chikorita","Treeko","Turtwig","Snivy","Chespin","Squirtle","Totodile","Mudkip","Piplup","Oshawott","Froakie"};
    //Pokemon level and type list
    Pokemon enemy[6] = {Pokemon(5,20,Fire), Pokemon(10,30,Fire), Pokemon(5,20,Grass), Pokemon(10,30,Grass), Pokemon(5,20,Water), Pokemon(10,30,Water)};
    //setting a value for the name randomiser and type randomiser
    int ene = rand() % 18;
    int ene2 = rndm(ene);
    //Copying pokemon name to buffer to be printed onto the screen
    std::string Pokename = PokemonNames[ene];
    Pokemon opponent = enemy[ene2];
    char buffer[64];
    std::string nameStr = Pokename;
    strncpy(buffer, nameStr.c_str(), sizeof(buffer));
    lcd.printString("A wild",25,2);
    lcd.printString(buffer,18,3);
    lcd.printString("appeared",20,4);
    lcd.refresh();
    wait(2.0);
    lcd.clear();
    lcd.printString(buffer,35,0);
    lcd.printString("LVL:??",35,1);
    battleStats();
    lcd.printString(buffer,35,0);
    //Actual battle taking place
    char HP1[10];
    char HP2[10];
    //Setting HP values
    int HPOT;
    int HPAT;
    int HPO = HPOT = pk.HPO(opponent);
    int HPA = HPAT = pk.HPA();
    int win = 0; //1 = win, 2 = loss, 3 = draw
    // After HP cycles past 0, The HP counter goes to above 429,000,000 ;
    //whle loop until one pokemons HP hits 0
    while ( win == 0 ) {
        //update HP values
        HPAT = HPAT - pk.OpponentTurn(opponent);
        HPOT = HPOT - pk.YourTurn(opponent);
        wait(1.8);
        //player wins
        if( HPOT >> HPO) {
            win = 1;
            lcd.printString("HP:0",2,5);
            pk.win(lcd);
            //printf("win");
        //wild pokemon wins
        } else if( HPAT >> HPA) {
            win = 2;
            lcd.printString("HP:0",35,1);
            //printf("lose");
        //both pokemon faint
        } else if( HPOT >> HPO && HPAT >> HPA) {
            win = 3;
            lcd.printString("HP:0",2,5);
            lcd.printString("HP:0",35,1);
            //printf("draw");
        //game still looping
        } else {
            lcd.drawRect(2,40,50,8,FILL_WHITE);
            lcd.drawRect(35,8,50,8,FILL_WHITE);
            //update screen with new HP
            sprintf(HP1, "HP:%u",HPAT);
            sprintf(HP2, "HP:%u",HPOT);
            lcd.printString(HP1,2,5);
            lcd.printString(HP2,35,1);
            lcd.refresh();
        }
    }
    //print correct script for win condition from script library
    sc.Battle(lcd, win);
    lcd.refresh();
    wait(2.0);
    //reset gamepad to counteract button bounce
    pad.init();
}

/** Draw the Pokemon class
 * Prints all your pokemons stats to the screen
 * uses B button to go back to Menu state
*/
void drawPoke()
{
    lcd.clear();
    //check for B button pressed, loop until.
    while(pad.check_event(Gamepad::B_PRESSED) == false) {
        /*if(pad.check_event(Gamepad::B_PRESSED) == true){
            printf("B Pressed")};*/
        //get Pokemon stats, copy to buffers to be printed to screen
        std::string pkn = pk.Name();
        std::string pkl = pk.Level();
        std::string pkh = pk.HP();
        std::string pkt = pk.Type();
        char a[50];
        char b[50];
        char c[50];
        char d[50];
        strncpy(a, pkn.c_str(), sizeof(a));
        strncpy(b, pkl.c_str(), sizeof(b));
        strncpy(c, pkt.c_str(), sizeof(c));
        strncpy(d, pkh.c_str(), sizeof(d));
        lcd.printString("Pokemon:",2,0);
        lcd.printString(a,2,1);
        lcd.printString(b,2,2);
        lcd.printString(c,2,3);
        lcd.printString(d,2,4);
        lcd.printString(" B = Back ",24,5);
        lcd.refresh();
    }
}

/** Draw the settings class
 * Used to set screen brightness
 * uses B button to go back to menu
*/
void settings()
{
    lcd.clear();
    lcd.refresh();
    lcd.printString("SET BRIGHTNESS", 0, 2);
    lcd.printString("B = BACK", 20, 4);
    lcd.refresh();
    wait(1.0);
    //check for B button pressed, loop until.
    while(pad.check_event(Gamepad::B_PRESSED) == false) {
        /*if(pad.check_event(Gamepad::B_PRESSED) == true){
            printf("B Pressed")};*/
        //get value of potentiometer from gamepad library
        //update brightness with float value
        float brightness = pad.read_pot();
        lcd.setBrightness(brightness);
        lcd.printString("SET BRIGHTNESS", 0, 2);
        lcd.printString("B = BACK", 20, 4);
        lcd.refresh();
    }
    pad.init();
}

