/**
@file main.h
@brief Header file contains functions and variables
@brief Snake Game - Embedded Systems Project
@brief Revision 1.0
@author Thokozile Tembo
@Date 05/05/2016
*/

#include "mbed.h"
#include "N5110.h"
#define DIRECTION_TOLERANCE 0.01

//Objects/Pin activation 
/**
@namespace lcd
@brief Creates N5110 object to allow interaction with lcd
*/
N5110 lcd(PTE26 , PTA0 , PTC4 , PTD0 , PTD2 , PTD1 , PTC3);

/**
@namespace Serial
@brief Creates Serial Communication with PC
*/
Serial pc(USBTX, USBRX);


/**
@namespace AnalogueIn
@brief Creates connection with Joystick & potentiometer
*/
AnalogIn joy_x(PTB2);  // joysitc x directions potentiometer
AnalogIn joy_y(PTB3); // joystick y directions potentiometer 
AnalogIn pot(PTB10); // brightness adjusting potentiometer

/**
@namespace Buttons
@brief Digital Inputs for joy button and pcb button
*/

DigitalIn joy_button(PTB11); //  needs pull-down (active high)
InterruptIn pcb_button(PTB18); // needs pull-down (active high)
// PWM Output (LED and Buzzer)
PwmOut pcb_led(PTC2);  // Green LED
PwmOut pcb_buzzer(PTA2); // sound speaker

/**
@namespace Ticker
@brief Creates ticker update for joystick& potentiometer adjustment
*/
Ticker joystick_update_ticker; // read values of joystick
Ticker brightness_update_ticker; // read values of potentiometer

/**
Initialises the K64F display and presents the welcome screen
@returns void
*/
void init_K64F();

void welcomeScreen(); /*!< Welcome screen*/




/** Defining possible joystick directions (enum)
@returns void
*/
enum DirectionName {
    UP,
    DOWN,
    LEFT,
    RIGHT,
    CENTRE,
    UNKNOWN
};

/** struct for Joystick, containing x,y values, button status, direction and centered values
@returns void
*/
struct JoyStick {
    double x;    /// current x value
    double x0;   /// 'centred' x value
    double y;    /// current y value
    double y0;   /// 'centred' y value
    int button; /// button state (assume pull-down used, so 1 = pressed, 0 = unpressed)
    DirectionName direction;  // current direction
} joystick;

/** struct for Apple, containing x,y values
@returns void*/

struct AppleType {
    int x; // Apple X
    int y; // Apple Y
};

/**  Defining possible states of the game (enum)
@returns void*/

enum GAME_STATES {
        INIT, // 0
        WAIT_FOR_USER, // 1
        PLAY, // 2
        GAME_OVER // 3
};

GAME_STATES game_state = INIT; // Default state of the game


volatile int g_pcb_button_flag = 0; /*!<Global variables*/


/** Function prototypes @returns void*/

void calibrateJoystick(); /*!< initial positions in the range 0.0 to 1.0*/
void updateJoystick();/*!< read current joystick values relative to calibrated values*/
void updateBrightness(); /*!< updates brightness of screen based on potentiometer reading*/
void snakeGame(); /*!< executres full functionality of game */

/**These functions are used to map game
@returns void*/

void drawBlock(int field_x, int field_y); /*!< Enable square (4x4 pixels) at x, y */

void drawApple(int field_x, int field_y);/*!< Draw Apple (4x4 pixels) at x, y*/

void eraseBlock(int field_x, int field_y); /*!< Disable square (4x4 pixels) at x, y*/


int getBlock(int field_x, int field_y); /*!<Get the status of the square (4x4 pixels) at x, y*/

AppleType placeRandomApple(); /*!< Place an apple at random location (returns food structure)*/

void pcb_button_isr(); /*!< pcb_button interrupt function*/


void blinkLEDandBuzz();/*!< This function make LED blink and buzzer make a sound*/