/**
Thokozile M Tembo 200822004
@file main.cpp
@brief Snake Game

*/
#include "main.h"

int main() {

    lcd.init(); // Initialise LCD
    // Set brightness accordingly to the potentiometer
    lcd.setBrightness(pot.read());
    init_K64F(); // Initialise connection with buttons
    welcomeScreen(); // Display welcome screen
    calibrateJoystick(); // Calibrate the joystick 
    
    // Read joystick values every 20 ms (50 Hz)
    joystick_update_ticker.attach(&updateJoystick, 0.02);
    // Read potentiometer value every 300 ms
    brightness_update_ticker.attach(&updateBrightness, 0.3);
    
    // Initialising random number generator
    srand(time(NULL));
            
    snakeGame();
}

void init_K64F() {
    // Configure serial comms
    pc.baud(115200);
    // Call pcb_button_isr function
    pcb_button.rise(&pcb_button_isr); 
    pcb_button.mode(PullDown); // use PullDown
}

void welcomeScreen() {
    lcd.printString("Welcome to" ,15,0); //inserts text for welcome on the first line hence the '1'
    lcd.printString("Snake Game",15,1); // prints the text in the brackets on the secondline hence '2'
    lcd.printString("Thokozile",0,3); //prints text in the third line
    lcd.printString("Tembo",0,4); // prints name in fifth line
    lcd.printString("200822004",0,5); // prints name in fifth line
    wait(2); // waits for 2 seconds
    lcd.clear(); 
    lcd.refresh(); // resets the screen
}
void calibrateJoystick() {
    // must not move during calibration
    joystick.x0 = joy_x;  // initial positions in the range 0.0 to 1.0 (0.5 if centred exactly)
    joystick.y0 = joy_y;
}

void updateJoystick() {
    // read current joystick values relative to calibrated values (in range -0.5 to 0.5, 0.0 is centred)
    joystick.x = joy_x - joystick.x0;
    joystick.y = joy_y - joystick.y0;
    joystick.button = joy_button; // read button state

    // calculate direction depending on x,y values
    // tolerance allows a little lee-way in case joystick not exactly in the stated direction
    if (fabs(joystick.y) < DIRECTION_TOLERANCE && fabs(joystick.x) < DIRECTION_TOLERANCE) {
        joystick.direction = CENTRE;
    } else if ( joystick.y > DIRECTION_TOLERANCE && fabs(joystick.x) < DIRECTION_TOLERANCE) {
        joystick.direction = UP;
    } else if ( joystick.y < DIRECTION_TOLERANCE && fabs(joystick.x) < DIRECTION_TOLERANCE) {
        joystick.direction = DOWN;
    } else if ( joystick.x > DIRECTION_TOLERANCE && fabs(joystick.y) < DIRECTION_TOLERANCE) {
        joystick.direction = LEFT;
    } else if ( joystick.x < DIRECTION_TOLERANCE && fabs(joystick.y) < DIRECTION_TOLERANCE) {
        joystick.direction = RIGHT;
    } else {
        joystick.direction = UNKNOWN;
    }
}

/* Snake Game Logic */

void snakeGame() {
    
    // Game Variables
    int snake_x[220]; // This is the array of pixels that refers to the snake parts, horizontally for X values
    int snake_y[220]; // This is the array of pixels that refers to the snake parts, vertically for Y values
    
    AppleType apple; // The function responsible for Creating the food
        
    // Game Variables
    int snake_size; // variable for snake length 
    int x_direction; //variable for snake  X direction
    int y_direction; // vairable for snake Y direction
    int next_x; //  the next X varaible to be occupied by snake
    int next_y; //  the next X varaible to be occupied by snake
    int score; //Stores the value of score based on how many fruits have been eaten by snake
    int snake_speed; // the rate at which the snake moves 

    while(1) {
        switch(game_state) { // initial state consturcts game environment snake, border food etc
            case INIT:
                x_direction = 0; // set initially to 0 
                y_direction = 0;
                score = 0;
                snake_size = 5; // Initial size of snake (set to 5 blocks)
                snake_speed = 0; // 0 - 100 lies between 0 and a 100
                // Clear Screen
                lcd.clear();
                // Draw border                        
                lcd.drawRect(1,1,81,45,0); 
                
                // Reset snake parts
                for (int i = 0; i < 220; i++) {//resets the coordinates of the snake body across the screen within border
                    snake_x[i] = 0;
                    snake_y[i] = 0;
                }
                // Draw Initial snake
                for (int i = 0; i < snake_size; i++) {  // draw the initial snake body, 5 blocks = 5time 4x4 
                    snake_x[i] = (snake_size-1) - i;
                    snake_y[i] = 0;
                    drawBlock(snake_x[i], snake_y[i]); // draws 4x4 pixel block 
                }
                
                apple = placeRandomApple(); // function responsible for setting random coordinates for food
                
                lcd.refresh();
                
                game_state = WAIT_FOR_USER; // Enters game state set by enum
                
                break;
            case WAIT_FOR_USER:
                // Wait until user moves joystick to the right or down (waits for game to start
                while ((joystick.direction != RIGHT) && (joystick.direction != DOWN)) {
                    sleep(); // ignores command when joystick is moved up or left at beginning of game
                }
                // Go to Play state
                game_state = PLAY;                
                break;            // if joystick is moved right or down, then PLAY commences
            case PLAY:
                while (game_state == PLAY) { //hence while in game play state  
                    
                    switch (joystick.direction) {// define joystick conditions for right
                        case RIGHT:
                            if (x_direction != -1) { // If joystick not pushed left
                                x_direction = 1; //then x direction is right
                                y_direction = 0; //y direction is 0 because joystick is not in vertical postion
                            }
                            break;
                        case LEFT:
                            if (x_direction != 1) { // If joystic is not pushed right
                                x_direction = -1; //then the diretion is left
                                y_direction = 0; //y remains zero because joystick is not moved vertically
                            }
                            break;
                        case DOWN:
                            if (y_direction != -1) { // If not pushed up, 
                                x_direction = 0; //remains 0 because joystick is not in horizontal position
                                y_direction = 1; // Hence direction is down
                            }
                            break;
                        case UP:
                            if (y_direction != 1) { // If joystick not pushed down
                                x_direction = 0; // then x direction is 0 because joystick is not in horizontal position 
                                y_direction = -1; // Hence direction is up
                            }
                            break;
                    }
                    
                    next_x = snake_x[0] + x_direction; // defines new X postion of each snake pixel
                    next_y = snake_y[0] + y_direction; // defines new Y position of each snake pixel
                    
                    if (getBlock(next_x, next_y) && (next_x != apple.x) && (next_y != apple.y)) {                      
                        // If snake runs into itself,
                        game_state = GAME_OVER; // then game is over
                    } else if ((next_x > 19) || (next_y > 10) || (next_x < 0) || (next_y < 0)) {
                        // if the snake hits the border, 
                        game_state = GAME_OVER; // game is over 
                    } else {
                        // Nothing happens
                        if ((next_x == apple.x) && (next_y == apple.y)) {
                            // If snake consumes food
                            
                            snake_size++; // the size of snake increases. 
                            score++; // the score increase by 1
                            snake_speed = ((snake_speed + 10) < 150) ? (snake_speed + 10) : snake_speed; // snake speed increase, gets more difficult 
                            
                            apple = placeRandomApple(); // generate new position of food 
                            // Does not remove tail block to allow increase in snake size
                        } else {                        
                            // Remove tail
                            eraseBlock(snake_x[snake_size-1], snake_y[snake_size-1]); // if snake did not eat food then erase tail block to maintain inital size
                        } // if snake did not eat food then erase tail block to maintain inital size

                        
                        
                        for (int i = snake_size-1; i > 0; i--) { // shifts all the pixels of the snake body, except head
                            snake_x[i] = snake_x[i - 1];
                            snake_y[i] = snake_y[i - 1]; 
                        }
                        
                        
                        snake_x[0] = next_x; // hence moves snake head to lead for new position
                        snake_y[0] = next_y;
                                                
                        
                        drawBlock(next_x, next_y); // constructs new head postion
                        
                        // Redraw things on LCD
                        lcd.refresh(); 
                                
                        wait_ms(200 - snake_speed); // alters snake speed based on rate of the loop
                    }
                }
                break;
            case GAME_OVER:
                char score_string[14]; // number of characters for score
                sprintf(score_string, "Score: %d", score); // prints score value estimated based on number of food eaten.
                // Clear Screen
                lcd.clear();
                // Print score
                lcd.printString("GAME OVER!",0,0); //prints 'GAME OVER' 
                lcd.printString(score_string,0,1);
                
                lcd.printString("REPLAY",48,5); // Replay message at bottom right corner next to PCB 
                
                // Blink LED and Buzz until user presses the
                // PCB Button
                blinkLEDandBuzz(); // Runs until the pcb_button pressed
                // When blinkLEDandBuzz finished go to Init state
                game_state = INIT; // returns to INIT state at beginning of loop to start new game
                break;              
            default:
                game_state = INIT;
        }
    }
}

AppleType placeRandomApple() { // responsible for placing food
 
    int good_location = 0; //varaible for suitable location for food  
    AppleType apple;
    // Look for a new location for an apple
    while (!good_location) { //while not in good location
        apple.x = rand() % 20; // 0 to 19  // random x coordinate 
        apple.y = rand() % 10; // 0 to 9 // random y coordinate
        // Don't put apple on the snake
        if (!getBlock(apple.x, apple.y)) good_location = 1; // checks for other pixels in the 20x11 field (within border)  
    }                                                    // avoids snake pixels, so food isnt generate on snake 
    drawApple(apple.x, apple.y); //hence draws food
    return apple; // returns food 
}

// field_x (0..19), field_y (0..10)
void drawBlock(int field_x, int field_y) { // responsible for the construct of the 4x4 pixel 
    // field_x*(block_size)+(border_offset) 
    lcd.drawRect(field_x*4+2, field_y*4+2, 3, 3, 1); 
}

void eraseBlock(int field_x, int field_y) { // erase block when snake moves, from preivous position
    // Draw an empty square on top
    // field_x*(block_size)+(border_offset)    
    lcd.drawRect(field_x*4+2, field_y*4+2, 3, 3, 3);
}

void drawApple(int field_x, int field_y) { //construct food
    int offset_x = field_x*4+2; // offset of 2 pixels
    int offset_y = field_y*4+2;
    // Drawing an apple pixel by pixel (8 dots)
    lcd.setPixel(offset_x+1,offset_y); 
    lcd.setPixel(offset_x+2,offset_y);
    lcd.setPixel(offset_x,offset_y+1);
    lcd.setPixel(offset_x,offset_y+2);
    lcd.setPixel(offset_x+3,offset_y+1);
    lcd.setPixel(offset_x+3,offset_y+2);
    lcd.setPixel(offset_x+1,offset_y+3);
    lcd.setPixel(offset_x+2,offset_y+3);
}

int getBlock(int field_x, int field_y) { // check for block on apply and snake to allow food to be eaten
    int offset_x = field_x*4+2;
    int offset_y = field_y*4+2;
    // Pixel at (offset_x+1,offset_y) will be active
    // for both apple and the snake block
    return lcd.getPixel(offset_x+1,offset_y);
}

// This function is called when button is pressed
void pcb_button_isr() {
    g_pcb_button_flag = 1;
}

void blinkLEDandBuzz() {
    // Reset button interrupt flag
    // in case if it was pressed during the game PLAY state 
    g_pcb_button_flag = 0;
                
    // Set the buzzer period to 0.002 s
    pcb_buzzer.period(0.002); // frequency of sound 
    // Flag is needed to switch between
    // on and off states repeatedly    
    int flag = 0;
    
    // Wait for the button to be pressed
    while(!g_pcb_button_flag) { // while not equal to 1, while button unpressed
        
        if (flag == 0) {
            // enable buzzer and led to ring and blink
            pcb_led.write(1);
            pcb_buzzer.write(0.5); // duty cycle period of 0.5sec and LED 1 sec
            flag = 1;
        
        } else if (flag == 1) {
            // disable buzzer and led when button is pressed 
            pcb_led.write(0);
            pcb_buzzer.write(0);
            flag = 0;
        }                
        
        wait(0.1); // delay before switching
    }
    
    // Disable LED and Buzzer 
    pcb_led.write(0);
    pcb_buzzer.write(0);
}

// updateBrightness function called
// by the ticker every 300 ms

void updateBrightness() {
    // Static means that the value will be kept
    // in the memory between iterations
    static float lcd_brighntness = 0;
    
    float pot_value = pot.read(); //value of potentiometer
    float pot_allowed_difference = 0.1; 
    
    if (abs(lcd_brighntness - pot_value) > pot_allowed_difference) {
        // Only update when potentiometer reading changes by
        // a certain amount
        lcd.setBrightness(pot_value);
        lcd_brighntness = pot_value;
    }
    
}