/**
@file main.h
@brief Header file containing functions prototypes, defines and global variables.
@brief Leeds University ELEC2645 sensor project
@author Bader Alathri
@date   May 2016
*/

#ifndef MAIN_H
#define MAIN_H


#include "mbed.h"
#include "N5110.h"
#include "SRF02.h"
#include "FXOS8700Q.h"  /**
@see https://developer.mbed.org/teams/Freescale/code/Hello_FXOS8700Q/
*/



/**
@namespace lcd
@brief output pins for N5110 LCD
@brief (PWR,SCE,RST,D/C,MOSI,SCLK,LED)****
*/
N5110 lcd(PTA1,PTA0,PTC4,PTD0,PTD2,PTD1,PTC3);
  

/**
@namespace sensor
@brief output pins for SRF02 sensor
*/
SRF02 sensor(PTE25,PTE24);

/**
@namespace acc
@brief output pins for FXOS8700
@brief onboard accelerometer
*/
FXOS8700Q_acc acc( PTE25, PTE24, FXOS8700CQ_SLAVE_ADDR1);

/**
@namespace pc
@brief UART connection for PC
*/
Serial pc(USBTX,USBRX);

/**
@see https://developer.mbed.org/teams/Freescale/code/Hello_FXOS8700Q/
@brief retrieves data from accelerometer
@brief taken from FXOS8700Q library
*/
MotionSensorDataUnits acc_data; 


/**
@namespace r_led
@brief GPIO output for external red LED
*/
DigitalOut r_led(PTC8);

/**
@namespace g_led
@brief GPIO output for external green LED
*/
DigitalOut g_led(PTC1);

/**
@namespace red_led
@brief output for onboard red LED
*/
DigitalOut red_led(LED_RED);

/**
@namespace green_led
@brief output for onboard green LED
*/
DigitalOut green_led(LED_GREEN);

/**
@namespace blue_led
@brief output for onboard blue LED
*/
DigitalOut blue_led(LED_BLUE); 

/**
@namespace toggle
@brief GPIO interrupt input for external toggle button
*/
InterruptIn toggle(PTC5);

/**
@namespace select
@brief GPIO interrupt input for external select button
*/
InterruptIn select(PTC7);

/**
@namespace ticker1
@brief periodically calls timer1_isr
*/
Ticker ticker1;


volatile int g_select_flag = 0; /*!< flag to be set in ISR when select button is pressed*/
volatile int g_toggle_flag = 0; /*!< flag to be set in ISR when toggle button is pressed*/
volatile int g_timer1_flag =0; /*!< flag to be set in ISR when periodically by ticker1 is pressed*/
volatile int g_error_flag = 0; /*!< flag to be set when measurement error is detected*/

float total_distance; /*!< stores the sum of the 10 SRF02 readings */
float average_distance_cm; /*!< stores the average distance calculated in cm*/
float average_distance_in; /*!< stores the average distance calculated in inches */
float average_z; /*!< stores the calculated average z-axis value */
float brightness; /*!< stores the brightness value*/


/**
Interrupt service routine for select button
sets select button flag
*/
void select_isr();

/**
Interrupt service routine for toggle button
sets toggle button flag
*/
void toggle_isr();

/**
Interrupt service routine for ticker1
sets timer1 flag
*/
void timer1_isr();

/**
lights up external red LED and displays error message on LCD
*/
void measurement_error();

/**
displays measurement transition screen and turns off external LEDs
*/
void measuring_transition();

/**
calculates average distance in cm from 10 SRF02 readings
and saves it in average_distance_cm variable;
*/
void get_average_distance_cm();

/**
calculates average distance in inches from 10 SRF02 readings
and saves it in average_distance_in variable;
*/
void get_average_distance_in();

/**
displays calculated average distance in cm on lcd
including the units
*/
void display_distance_cm();

/**
displays calculated average distance in inches on lcd
including the units
*/
void display_distance_in();

/**
gets average z-axis value from 10 readings in 0.1 seconds
of the onboard FXOS8700 accelerometer
and saves it in the average_z variable
*/
void get_z_axis_value();

/**
according to the calculated average_z value, 
displays on lcd whether surface is flat or not
specifying whether it is flat horizontally or vertically
*/
void display_flatness();

/**
increases brightness value by 0.2 if it is less than 1.0,
and changes brightness to 0 if it is already at 1.0
*/
void adjust_brightness();

/**
displays brightness as a percentage on lcd
*/
void display_brightness();

/**
error function hangs flashing onboard LED
*/
void error();

/**
setup serial port
*/
void init_serial();

/**
set-up the on-board LEDs
*/
void init_K64F();




#endif
