/**
@file main.h
@brief Header file containing functions prototypes, defines and global variables.
@brief Shows examples of creating Doxygen documentation.
@brief Revision 1.0.
@author Martin Woodhams
@date   March 2015
*/

#ifndef MAIN_H
#define MAIN_H

#include "mbed.h"
#include "N5110.h"
#include "Arrays.h"


#define UP  0
#define UPRIGHT 1
#define RIGHT 2
#define DOWNRIGHT 3
#define DOWN 4
#define DOWNLEFT 5
#define LEFT 6
#define UPLEFT 7
#define STAY 8

#define DIRECTION_TOLERANCE 0.025f

/** 
@namespace lcd
@brief This allows the mbed to interface with the N5110 screen
@param VCC - PTE26
@param SCE - PTA0
@param RST - PTC4
@param D/C - PTD0
@param MOSI - PTD2
@param SCLK - PTD1
@param LED - PTC3 
*/
N5110 lcd (PTE26 , PTA0 , PTC4 , PTD0 , PTD2 , PTD1 , PTC3);

/**
@namespace b1
@brief Sets up a button that allows for interrupts to be called on pin PTB19
*/
InterruptIn b1 (PTB19);

/**
@namespace b2
@brief Sets up a button that allows for interrupts to be called on pin PTB18
*/
InterruptIn b2 (PTB18);

/**
@namespace xPot
@brief Sets up the potentiometer that determines the horizontal position of the joystick
*/
AnalogIn xPot(PTB2);

/**
@namespace yPot
@brief Sets up the potentiometer that determines the vertical position of the joystick
*/
AnalogIn yPot(PTB3);

/**
@namespace ticker
@brief A ticker that is attatched to a time based interrupt to allow the screen to periodically refresh
*/
Ticker ticker;

/**
@namespace timer
@brief A timer that is used as the seed to generate a random number for asteroid creation
*/
Timer timer;

volatile int g_b1_flag = 0; /*!<A global flag which is called on when the interrupt on b1 is called */
volatile int g_b2_flag = 0; /*!<A global flag which is called on when the interrupt on b2 is called */
volatile int g_timer_flag = 0; /*!<A global flag which is called on when the timer flags */
int score = 0; /*!< A variable that keeps track of the score the player is achieving */
int bullet_number=0; /*!< A variable that keeps count of how many bullets are on screen */
int shipDirection = 0; /*!< A variable determined by the FSM, this variable chooses which ship array is displayed and which direction bullets are shot*/
int game_over = 0; /*!< A variable that determines when the game is over */
int printFlag = 0; /*!<a flag to show that the joystick can be read*/
int state = 1;  /*!< set initial state of the FSM state 1 UP*/
int direction = UP; /*!<the initial direction of the ship is to travel UP */
int settings = 0;   /*!<aa flag to tell if the settings option has been selected*/

struct State {
    int shipDirection;  // current state
    int next_state[9]; // next state can be up determined by up to 9 different inputs

};  /*!< creates the structure for the FSM*/ 

typedef const struct State STyp;

STyp fsm[8] = {
    {0,{0,1,1,1,1,7,7,7,0}},
    {1,{0,1,2,2,2,2,0,0,1}},
    {2,{1,1,2,3,3,3,3,1,2}},
    {3,{2,2,2,3,4,4,4,4,3}},
    {4,{5,3,3,3,4,5,5,5,4}},
    {5,{6,6,4,4,4,5,6,6,5}},
    {6,{7,7,7,5,5,5,6,7,6}},
    {7,{0,0,0,0,6,6,6,7,7}},

}; /*!< The FSM has 9 possible inputs and 8 possible outcomes the inputs are taken from the joystick direction and go as following
UP = 0, UPRIGHT = 1, RIGHT = 2, DOWNRIGHT = 3, DOWN = 4, DOWNLEFT = 5, LEFT = 6, UPLEFT = 7, STAY = 8
The ships direction follows the same pattern and the FSM aims to get the ship to face that direction in the quickest way possible
If the joystick is pushed in a direction completly opposite to the ship the ship will spin clockwise
*/

/**
A function which alters the b1 flag from 0 to 1 when an interrupt occurs
*/
void b1_isr();

/**
A function which alters the b2 flag from 0 to 1 when an interrupt occurs
*/ 
void b2_isr();

/**
A function which alters the timer flag from 0 to 1 when the time interrupt occurs
*/ 
void timer_isr();

/**
Reads the initial positon of the joystick and sets this as centered
*/
void calibrateJoystick();

/**
Compares the current positon of the joystick to the centred position of the joystick and calculates which way the joystick has been pushed
*/
void updateJoystick();

/**
The main game function in it the 
*/
void game();

/**
A function that reads the ships direction and fires a bullet in that direction.
The function also checks any exsisting bullets and moves them in the appropriate direction.
*/
void shoots();

/**
A function that using random numbers determines which size asteroid to generate and from which direction
The function also checks to see if there is a bullet or ship in front of it, if there isn't the asteroid moves forward
*/ 
void asteroidMove();

/**
checks to see if the bullet has left the screen without hitting anything if it has the bullet is erased
*/
void checkbullet();

#endif