/**
@file main.h

@brief Header file containing functions prototypes, defines and global variables.
@brief Revision 1.1
@author Joel W. Webb
@date   March 2016
*/
 
#ifndef MAIN_H
#define MAIN_H
 
#define PI 3.14159265359
#define DIRECTION_TOLERANCE 0.2
 
#include "mbed.h"
#include "N5110.h"
#include "SDFileSystem.h"



        // GPIO Objects
/*
@namespace lcd
@brief N5110 object for interfacing with Nokia 5110 LCD
@param VCC - PTE26
@param SCE - PTA0
@param RST - PTC4
@param D/C - PTD0
@param MOSI - PTD2
@param SCLK - PTD1
@param LED - PTC3 
*/
N5110 lcd(PTE26, PTA0, PTC4, PTD0, PTD2, PTD1, PTC3);

/*
@namespace sd
@param MOSI - PTE3
@param MISO - PTE1
@param SCK - PTE4
@param CS - PTE4
*/
SDFileSystem sd(PTE3, PTE1, PTE2, PTE4, "sd"); // MOSI, MISO, SCK, CS

/**  
@namespace errorLED
@brief GPIO output for error LED
*/
DigitalOut errorLED(LED1);

/**
@namespace buzzer
@brief PWM output for buzzer
*/
PwmOut buzzer(PTC10);

/**
@namespace xpot
@brief The joystick x axis potentiometer
*/
AnalogIn xPot(PTB11);

/**
@namespace ypot
@brief The joystick y axis potentiometer
*/
AnalogIn yPot(PTB10);

/**
@namespace buttonjoy
@brief Interrupt input for Joystick button
*/
InterruptIn buttonjoy(PTB9);

/**
@namespace buttonA
@brief Interrupt input for Button A
*/
InterruptIn buttonA(PTB23);

/**
@namespace buttonB
@brief Interrupt input for Button B
*/
InterruptIn buttonB(PTA1);



        //Data Structures

/// stringList struct for dealing with printing strings to LCD easier
struct stringList {
    char str[14];
    int offset;
};
typedef stringList stringList;

/// cell struct for dealing with coordinates
struct cell {
    int x;
    int y;
};
typedef cell cell;

/// vector struct for dealing with vectors
struct vector {
    float x;
    float y;
};
typedef vector vector;

/// plinkvar struct is used to pass information to and from the plink functions without resorting to many global variables
struct plinkvar {
        int difficulty;
        vector pos;
        vector vel;
        vector acc;
        int height;
        int gameOver;
        int platformWidth;
        cell powerUp;
        int powerUpHeight;
        int powerUpRadius;
        int ballRadius;
        cell platforms[20];
        float tickerDelay;
};
typedef plinkvar plinkvar;


        // Global variables
        
volatile int g_buttonA_flag; /*!< Button A flag set in ISR */
volatile int g_buttonB_flag; /*!< Button B flag set in ISR */
volatile int g_buttonjoy_flag; /*!< Joystick Button flag set in ISR */
volatile int g_joystick_flag; /*!< Joystick x and y values have been polled in ISR */
volatile int g_gametick_flag; /*!< gametick flag is set in gametick_isr by Ticker gametick */
volatile int g_screentick_flag; /*!< screentick flag is set in screentick_isr by Ticker screentick */
volatile const float* noteArray; /*!< Float pointer to next note in array. Used in sound Timeout ISR */

// Menu stringList arrays kept in FLASH
const stringList menuList[] = {{"Games",20},{"Snake",0},{"Plink",0}};
const stringList snakeList[] = {{"Snake",20},{"Start",0},{"Difficulty",0},{"Highscore",0},{"Back",0}};
const stringList difficultyList[] = {{"Difficulty",10},{"Easy",0},{"Medium",0},{"Hard",0}};
const stringList plinkList[] = {{"Plink",20},{"Start",0},{"Difficulty",0},{"Highscore",0},{"Back",0}};

 
 
        // Function Prototypes
        
    //ISR
/**
Interrupt service routine for button A
@brief Sets g_buttonA_flag
*/
void buttonA_isr();
/**
Interrupt service routine for button B
@brief Sets g_buttonB_flag
*/
void buttonB_isr();
/**
Interrupt service routine for buttonjoy
@brief Set g_buttonjoy_flag
*/
void buttonjoy_isr();

/**
Interrupt service routine for playing the current note
@brief Alters the current PWM signal to the current note frequency read from noteArray
*/
void sound_isr();

/**
Interrupt service routine for Ticker gametick
@brief Used to control the update speed of games
*/
void gametick_isr();
/**
Interrupt service routine for Ticker screentick
@brief Used to control the update speed of the LCD display
*/
void screentick_isr();


    // Snake functions
/**
Main Snake game function
@brief The snake function is called from the main loop and contains everything needed to play the snake game
@param difficulty - The difficulty is easy(0) medium(1) or hard(2) and it alters how fast the game ticker is applied
*/
void snake(int difficulty);


    // Plink functions
/**
Main Plink game function
@brief The plink function calls its sub-functions to allow the user to play a physics based jumping game
@param difficulty - The difficulty is easy(0) medium(1) or hard(2) and it alters how long the platforms are
*/
void plink(int difficulty);
/**
Initialising plink variables function
@brief The initpvar function sets all of the intiial plink variables
@param pvar - Struct that contains all necessary plink variables
*/
plinkvar initpvar(plinkvar pvar);
/**
Scrolls the objects on the LCD display
@brief Handles when the ball exceeds a certain height and moves all objects up in Y value
@param pvar - Struct that contains all necessary plink variables
*/
plinkvar plinkScroll(plinkvar pvar);
/**
Platform Generation function
@brief By default only 20 platforms are saved at any one time so new platforms need to be generated when old ones are no longer needed
@param pvar - Struct that contains all necessary plink variables
*/
plinkvar plinkPlatGen(plinkvar pvar);
/**
Object collision handling function
@brief When objects such as the ball, platforms, walls and power ups collide this function detects it and performs the corresponding action
@param pvar - Struct that contains all necessary plink variables
*/
plinkvar plinkCollisions(plinkvar pvar);
/**
Plink Physics Engine function
@brief Handles all the physics related movements with newtons laws of motion every game iteration
@param pvar - Struct that contains all necessary plink variables
*/
plinkvar plinkPhysicsEngine(plinkvar pvar);
/**
LCD Screen drawing function
@brief This function takes all objects and draws them on the LCD display
@param pvar - Struct that contains all necessary plink variables
*/
plinkvar plinkDrawScreen(plinkvar pvar);
/**
Game Over Function
@brief This function handles all the high score saving and printing to LCD display
@param pvar - Struct that contains all necessary plink variables
*/
plinkvar plinkGameOver(plinkvar pvar);


    // Menu functions
/**
Menu screen Controller
@brief menu is a function that handles the selection of different menu selections
@param menuList - The pointer to the array of strings handled in the menu (Must be 6 elements or less)
@param line - The number of lines in the menu list (Must be 6 or less)
@returns Selected menu element (0-4)
*/
int menu(const stringList* menuList,int lines);
/**
The drawString function
@brief This function is used to simplify the menu function
@param stringList - a const string array containing the strings needed to be written on each line
@param lines - Integer representing how many strings should be written
*/
void drawStrings(const stringList* list,int lines);


    // Sound and error functions
/**
Attaches Timeout to play sounds immediately
@brief playSound is a function to provide ease fo use when calling sound arrays
@param sound - The pointer to the noteArray
*/
void playSound(const float* sound);
/**
Error function
@brief Hangs while flashing errorLED
*/
void error();


    // Initialising and joystick
/**
Initializes Inputs
@brief Used to group code that will run only once at startup
*/
void initInputs();
/**
Initializes the joystick
@brief Aquires xpot and ypot default values
*/
void calibrateJoystick();
/**
updateJoystick
@brief Updates direction the joystick is pointing in on a Ticker event
*/
void updateJoystick();



#include "sounds.h"
 
#endif
 