/**
@file main.h
@brief Header file containing functions prototypes, defines and global variables.
@brief Revision 1.0.
@author Daniel Gibbons
@date   May 2016
*/

#ifndef MAIN_H
#define MAIN_H

#define PI 3.14159265359

#include "mbed.h"
#include "N5110.h"
#include "SRF02.h"
#include "FXOS8700Q.h"
#include "SDFileSystem.h"


/**  
@namespace lcd
@brief N5110 LCD connections
@namespace srf02
@brief SRF02 sensor connections
@namespace i2c
@brief K64F I2C connections for FXOS8700Q on-board Accelerometer
@namespace acc
@brief K64F FXOS8700Q on-board Accelerometer connections
@namespace sd
@brief SD card connections
@namespace pc
@brief UART connection to PC for debugging
@namespace led_alarm
@brief LED to indicate status of the alarm: flashing -> setting or triggered ; constant -> set
@namespace buzzer
@brief Indicates status of the alarm: buzzes when alarm is setting and when triggered
@namespace button_0
@brief First button interrupt
@namespace button_1
@brief Second button interrupt
@namespace button_c
@brief Third button interrupt
@namespace sw2
@brief K64F on-board switch
@namespace sw3
@brief K64F on-board switch
@namespace setting_distance
@brief This ticker fires to read the distance when the system is setting
@namespace intruder_distance
@brief This ticker fires to read the distance when the system is set
@namespace alerts
@brief This ticker fires to turn and off the led and buzzer
@namespace pin_timeout
@brief This ticker fires to timeout the user if they don't enter the pin within an alotted time of triggering the device
@namespace setting_screen
@brief This ticker fires to display the setting animation on the setting screen
@namespace accelerometer
@brief This ticker fires whilst in the setting screen to get accelerometer data
@namespace transition
@brief This timeout transitions the system to the set state, five seconds after entering the setting state, or to the menu if the device is moved in the setting state
@namespace tamper_transition
@brief This timeout transitions the system  to the menu if the device is moved in the setting state
@namespace buzz
@brief This timeout turns off the buzzer one second after the device is set
@namespace confirm
@brief This timeout transitions to a screen asking the user to confirm the pin that they have just entered
@namespace r_led
@brief K64F on-board red LED
@namespace g_led
@brief K64F on-board green LED
@namespace b_led
@brief K64F on-board blue LED

*/

//         VCC, SCE, RST, D/C, MOSI, SCLK, LED
N5110 lcd(PTE26,PTA0,PTC4,PTD0,PTD2,PTD1,PTC3);
SRF02 srf02(I2C_SDA,I2C_SCL);
I2C i2c(PTE25, PTE24);
FXOS8700QAccelerometer acc(i2c, FXOS8700CQ_SLAVE_ADDR1);
//              MOSI, MISO, SCK, CS
SDFileSystem sd(PTE3, PTE1, PTE2, PTE4, "sd"); 
Serial pc(USBTX,USBRX);
PwmOut led_alarm(PTC10);
PwmOut buzzer(PTC11);
InterruptIn button_0(PTB23);
InterruptIn button_1(PTA2);
InterruptIn button_c(PTC2);
InterruptIn sw2(SW2);
InterruptIn sw3(SW3);
Ticker setting_distance;
Ticker intruder_distance;
Ticker alerts;
Ticker pin_timeout;
Ticker setting_screen;
Ticker accelerometer;
Timeout transition;
Timeout tamper_transition;
Timeout buzz;
Timeout confirm;
DigitalOut r_led(LED_RED);
DigitalOut g_led(LED_GREEN);
DigitalOut b_led(LED_BLUE);

struct State {
    
    /* array of next states which depend on which button was pressed
       - nextState[0] ----> button_0 pressed
       - nextState[1] ----> button_1 pressed
       - nextState[2] ----> button_c pressed (Option 1)
       - nextState[3] ----> button_c pressed (Option 2)
    */
       
    int nextState[4];  
    
};
typedef const struct State STyp;

// fsm stores all nine states that 'Mobile Security System' is comprised off
STyp fsm[9] = {
    {0,0,0,0},  // 0 - initialisation (title screen)
    {2,3,1,1},  // 1 - main menu (Set alarm or set new password)
    {2,2,4,1}, // 2 - set alarm (enter password)
    {3,3,1,1}, // 3 - set new password (enter new password)
    {4,4,4,4}, // 4 - setting calibration
    {5,6,5,5}, // 5 - alarm activated
    {6,6,1,5}, // 6 - deactivate without triggering (enter password)
    {7,7,1,8}, // 7 - alarm triggered (enter password)
    {8,1,8,8}   // 8 - alarm triggered
};

FILE *pin; /*!< File pointer */

int entered_pin[4]; /*!< Pin user enters */ 
int set_pin[4]; /*!< Pin that is saved to SD card and this is the current pin for the system */
int index_array[4]; /*!< Array to index the reading from the SD card */

int pin_counter; /*!< Increments to store the entered pin */
int incorrect_pin_flag; /*!< If the entered pin doesn't match the set pin this flag is incremented to 1 */

float acc_X; /*!< Current x-axis reading from the accelerometer */
float acc_Y; /*!< Current y-axis reading from the accelerometer */
float acc_Z; /*!< Current z-axis reading from the accelerometer */
float setting_acc_X; /*!< X-axis reading from the accelerometer when setting screen is initialised */
float setting_acc_Y; /*!< Y-axis reading from the accelerometer when setting screen is initialised */
float setting_acc_Z; /*!< Z-axis reading from the accelerometer when setting screen is initialised */

int g_current_state; /*!< The current state of the system */
int g_next_state; /*!< The next state of the system  */

float distance[10]; /*!< Stores 10 distance readings from SRF02 */
float one_second_distance; /*!< The total of all 10 distance readings taken in a second */
double one_second_avg_distance; /*!< one_second_distance divided by 10 */
double initial_setting_distance; /*!< The one_second_avg_distance calculated when in the setting state (state 4)  */

int setting_distance_counter; /*!< increments to 10 distance readings can be stored over 1 second */
int intruder_distance_counter; /*!< increments to 10 distance readings can be stored over 1 second */
int pin_timeout_counter; /*!< increments to 20 to enable the 20 second timeout when the alarm is triggered */
int setting_alarm_counter; /*!< increments in steps of five and is used for the setting screen animation */

int seconds_till_timeout; /*!< 21 - pin_timeout_counter */

volatile int g_setting_distance_flag; /*!< Flag in setting_distance_isr */
volatile int g_intruder_distance_flag; /*!< Flag in intruder_distance_isr */
volatile int g_button_0_flag; /*!< Flag in button_0_isr */
volatile int g_button_1_flag; /*!< Flag in button_1_isr */
volatile int g_button_c_flag; /*!< Flag in button_c_isr */
volatile int g_led_buzzer_flag; /*!< Flag in led_buzzer_isr */
volatile int g_acc_flag; /*!< Flag in acc_isr */
volatile int g_pin_timeout_flag; /*!< Flag in pin_timeout_isr */
volatile int g_setting_screen_flag; /*!< Flag in setting_screen_isr */

char buffer[14]; /*!< Stores strings that are comprised of variables that are going to be displayed on the LCD */
int length; /*!< Stores the length of strings that are comprised of variables that are going to be displayed on the LCD */

///
/// Initialisation Functions
///
void error(); /*!< Hangs flashing on_board LED */
  
void init_serial(); /*!< Set-up the serial port */

void init_K64F(); /*!< Set-up the on-board LEDs and switches */

void init_buttons(); /*!< Set-up the three external buttons */

void init_variables(); /*!< Initialises all variables to zero */
///
/// Interrupt Service Routines (ISRs)
///
void setting_distance_isr(); /*!< Interrupt Service Routine that triggers when setting_distance ticker fires */

void intruder_distance_isr(); /*!< Interrupt Service Routine that triggers when intruder_distance ticker fires */

void button_0_isr(); /*!< Interrupt Service Routine that triggers when button_0 is pressed by the user */

void button_1_isr(); /*!< Interrupt Service Routine that triggers when button_1 is pressed by the user */

void button_c_isr(); /*!< Interrupt Service Routine that triggers when button_c is pressed by the user */

void led_buzzer_isr(); /*!< Interrupt Service Routine that triggers when alerts ticker fires */

void acc_isr(); /*!< Interrupt Service Routine that triggers when accelerometer ticker fires */

void pin_timeout_isr(); /*!< Interrupt Service Routine that triggers when pin_timeout ticker fires */

void setting_screen_isr(); /*!< Interrupt Service Routine that triggers when setting_screen ticker fires */
///
/// Button Functions
///
void button_0_protocol(); /*!< Protocol when button_0 is pressed */

void button_1_protocol(); /*!< Protocol when button_1 is pressed */

void button_c_protocol(); /*!< Protocol when button_c is pressed */
///
/// State Functions
///
void state_0_screen(); /*!< Sets the screen and the necessary tickers and timeouts for state 0 */

void state_1_screen(); /*!< Sets the screen and the necessary tickers and timeouts for state 1 */

void state_2_screen(); /*!< Sets the screen and the necessary tickers and timeouts for state 2 */

void state_3_screen(); /*!< Sets the screen and the necessary tickers and timeouts for state 3 */

void state_4_screen(); /*!< Sets the screen and the necessary tickers and timeouts for state 4 */

void state_5_screen(); /*!< Sets the screen and the necessary tickers and timeouts for state 5 */

void state_6_screen(); /*!< Sets the screen and the necessary tickers and timeouts for state 6 */

void state_7_screen(); /*!< Sets the screen and the necessary tickers and timeouts for state 7 */

void state_8_screen(); /*!< Sets the screen and the necessary tickers and timeouts for state 8 */

void lcd_border(); /*!< Prints a border to the Nokia 5110 LCD  */

void pin_text_box(); /*!< Prints a box to the Nokia 5110 LCD for the states where a pin is required  */

void screen_progression(); /*!< Calls check pin to see if the entered pin is correct and from there either proceed or go back */

void screen_selection(); /*!< Calls the relevant state_X_screen() depending on the value of g_next_state */
///
/// Screen Animation Functions
///
void setting_animation(); /*!< Prints the animation on the setting screen to the Nokia 5110 LCD */

void timeout_protocol(); /*!< Prints timeout counter when the alarm is triggered */
///
/// Read Distance Functions
///
void get_setting_distance(); /*!< Gets the distance from the SRF02 and then increments the setting_distance_counter */

void get_intruder_distance(); /*!< Gets the distance from the SRF02 and then increments the intruder_distance_counter */

void calculate_setting_distance(); /*!< Averages the past 10 distance readings and stores them in the initial_setting_distance */

void calculate_intruder_distance(); /*!< Averages the past 10 distance readings and compares this value to the initial_setting_distance to detect a potential intruder */
///
/// Timeout Functions
///
void screen_5_transition(); /*!< Called five seconds after the setting screen is called */

void pin_confirm(); /*!< Clears the screen when a pin is entered to tell the user to press button 'C' to confirm the entered pin */

void state_1_transition(); /*!< Sends the user back to state 1 if the device is tampered in the setting state */

void alarm_setting_buzz(); /*!< Turns off buzzer one second after entering state 5 */
///
/// Accelerometer Functions
///
void get_axis_data(); /*!< Gets the accelerometer data from the K64F on_board accelerometer and stores them in acc_X, acc_Y and acc_Z */

void compare_axis_data(); /*!< Compares the accelerometer data from the K64F on-board accelerometer with the stored values of setting_acc_X, setting_acc_Y and setting_acc_Z */ 

void device_tampered_protocol(); /*!< If the device is moved whilst in the setting state this function is called to stop the alarm from setting */
///
/// Pin Functions
///
void enter_pin(); /*!< Prints '*' to the Nokia 5110 LCD when a pin is being entered */

void reset_entered_pin(); /*!< Resets the entered_pin array to have elements with a value of -1 */

void check_pin(); /*!< Compares the entered_pin to the set_pin */

void change_pin(); /*!< Writes entered_pin to the SD card when a pin is entered in state 3 - adapted from 2545_SD_Card Example by Dr. C.A. Evans */

void read_pin(); /*!< Reads pin from the SD card to the set_pin array - adapted from 2545_SD_Card Example by Dr. C.A. Evans */

void delete_file(char filename[]); /*!< Deletes the old pin from the SD card - taken from 2545_SD_Card Example by Dr. C.A. Evans */

/**

*/

#endif