/**
@file ESP.h
@brief Header file containing function prototypes, global variables, ticker objects, structs as well as K64F connections
@author Aleksander Filipiak
@date   May 2016
*/


#include "mbed.h"
#include "N5110.h"



/**
@namespace lcd
@brief N5110 library object
@brief pin connections:VCC,SCE,RST,D/C,MOSI,SCLK,LED
*/
N5110 lcd(PTE26 , PTA0 , PTC4 , PTD0 , PTD2 , PTD1 , PTC3);

#define DIRECTION_TOLERANCE 0.05

/**
@namespace start_button
@brief GPIO input for "start" button
*/
DigitalIn start_button(PTE24);

InterruptIn pause_button(PTE24);
/**
@namespace speed_button
@brief GPIO input for button controlling ship speed
*/
DigitalIn speed_button(PTE25);
/**
@namespace xPot
@brief GPIO input for joystick's x-axis potentiometer
*/
AnalogIn xPot(PTB3);
/**
@namespace yPot
@brief GPIO input for joystick's y-axis potentiometer
*/
AnalogIn yPot(PTB10);

///creates enumerated type for directions: 0 = UP, 1 = DOWN, 2 = LEFT, 3 = RIGHT, 4 = CENTRE and 6 = UNKNOWN
enum DirectionName {
    UP,
    DOWN,
    LEFT,
    RIGHT,
    CENTRE,
    UNKNOWN
};


typedef struct JoyStick Joystick;
/** 
@struct Joystick
@brief Struct for joystick parameters
@param x - current x value
@param x0 - 'centred' x value
@param y - current y value
@param y0 - 'centred' y value
@param direction - direction joystick is set to
@see {DirectionName}
*/
struct JoyStick {
    float x;    
    float x0;   
    float y;    
    float y0;   
    DirectionName direction;  
};
// create struct variable
Joystick joystick;

/** flag for joystick update
@see {updateJoystick}
*/
bool printFlag = 0;

/**
@struct Ship
@brief Struct for ship parameters
@param x - Current x value of centre pixel
@param y - Current y value of centre pixel
@param direction - Direction of flight
@see {DirectionName}
*/
typedef struct Ship Ship;
struct Ship {
    short x;         
    short y;          
    DirectionName direction;  
};
// the ship, controlled by player
Ship ship;

/** 
@struct Turret
@brief Struct for turret parameters
@brief Four turret variables are used: one for each border edge
@param position - Position along relevant axis. Depends on game edge on which the turrent moves long.
@param shoot - Flag raised if missle was fired
@see {shoot}
@param direction - Direction of movement
@see {DirectionName}
*/
typedef struct Turret Turret;
struct Turret {
    short position; 
    bool shoot;    
    DirectionName direction; 
};
// four turrets, one for each side
Turret top_turret;
Turret bottom_turret;
Turret left_turret;
Turret right_turret;

/**
@struct Missle
@brief Struct for missle parameters
@brief Six variables are used, one for each of the six possible acitve missles on the screen 
@param x - Position on x-axis
@param y - Position on y-axis
@param direction - Direction of travel
@see {DirectionName}
@param hit_flag - Flag that missle has hit a target
@see {check_hit}
@param active - Is the missle active
@see {shoot}
*/
typedef struct Missle Missle;
struct Missle {
    short x; 
    short y; 
    DirectionName direction; 
    bool hit_flag; 
    bool active; 
};
// maximum of 6 active missles
Missle missle_one;
Missle missle_two;
Missle missle_three;
Missle missle_four;
Missle missle_five;
Missle missle_six;

typedef struct Collectible Collectible;
/**
@struct Collectible
@brief Struct for collectible parameters
@brief Two variables are used, one for each possible collectible
@param x - x coordinate of collectible once it appears
@param y - y coordinate of collectible once it appears
@param active - Flag for when the collectible has appeared
@see {print_collect}
*/
struct Collectible {
    short x;
    short y;
    bool active; 
};
// max number of 6 active collectibles
Collectible collect_one;
Collectible collect_two;


//  --------------------------------------------------------------------------TICKERS---------------------------------------------------------------------------------------------------

/// timer to regularly read the joystick
Ticker pollJoystick;
/// timer for updating missle position
Ticker missle_mover;
/// timer for checking for missle hits
Ticker hit_checker;
/// timer for turrets' shoot function
Ticker shooter;
/// game timer
Ticker timer;
/// timer for updating ship position
Ticker ship_mover;
/// updates position of turrets
Ticker turret_mover;
/// timer for collectible timeout
Ticker collect_timer;
/// triggers function for creating collectibles
Ticker collect_printer;
///Ticker for updating collectible on screen
Ticker collect_updater;
/// triggers function for checking if collectible has been collected
Ticker collect_checker;

// ---------------------------------------------------------------------------VARIABLES-------------------------------------------------------------------------------------------------

/** int for keeping track of the number of loops of the change_speed function
@see {speed_change} */
int g_ship_counter;

/** variable for time left in game
@see {time_left} */
int g_time_left;

/// keep track of score
short g_score;

/// flag for collectible collected, set in check_collect
volatile bool g_collect_flag;

/// flag for hit detection
volatile bool g_hit_flag;

/// flag for collectible timeout, set in collect_timeout().
volatile bool g_collect_timeout_flag;

/// variable for storing what collectible (collect_one or collect_two) has expired
volatile bool g_what_expired;

/// flag for pause function, set in pause_isr
volatile bool g_pause_flag;


// ---------------------------------------------------------------------------FUNCTION DECLARATIONS-------------------------------------------------------------------------------------

/** interrupt for game pause, triggered by pause_button
@see {pause_button}
*/
void pause_isr();

/** read default positions of the joystick to calibrate later readings
@see {Joystick}
*/
void calibrateJoystick();

/** read and update position of the joystick
@see {Joystick}
*/
void updateJoystick();

/** Isr for following timer, decrements g_time_left every second.
@see {g_time_left}
Prints the time left on the above the game borders. 
*/
void timer_isr ();

/** prints border between four set points
@param x_start - x axis starting point
@param x_end - x axis end point
@param y_start y axis starting point
@param y_end y axis end point
*/
void print_border (int x_start, int x_end, int y_start, int y_end);

/** Setup initial screen: prints four turrets and ship in their default positions.
@see {Ship}
@see {Turret}*/
void setup_game ();

/** Function for changing the speed of the ship based on the value of speed_button.
@see {speed_button}
    Uses a modulus command to either update the ship's position every time the function is called, 
    i.e. when speed_button is pressed and the ship should fly fast, or only every third time the function is calles, 
    when the ship should fly at 1/3 of the fast speed. */
void speed_change();


/** Update ship's position based on joystick controls: clears ship at current position and prints it at the next one.
    The shape which is printed depends on the direction of flight set by the joystick controlls.
    Stores position and direction of flight in the Ship struct. limits flight of ship to within the game boders
    Is contained withing the speed_change function.
@see {change_speed}
@see {Ship}*/
void update_ship ();

/** Updates the position of the four turrets, clears turrets at current position and prints them at their new positions.
    Turrets move along their respective axis until they hit the game screen boundry, then they travel in the opposite direction.
    function is triggered at regular intervals by turret_mover Ticker.
@see {Turret} */   
void update_turrets();

/** At regular intervals, dictated by shoort Ticker, gives each turret a 50% chance to shoot a missle.
    Turret will shoot if any of the 6 possible missles are available. If a missle is available, its starting x/y position and direction of flight
    will be set to that of the middle of the turret, and it will be set to active.*/
void shoot();

/** updates the position of all active missles at regular intervals.
@see {missle_mover}
@see {shoot}
 */
void update_missles ();

/** Checks if missle has hit the ship, triggered by hit_checker Ticker.
    Works by checking if the central pixel of the active missle has is inside the ship's "hit-box": a rectangle around its dimensions.
    hit-box implementation idea sourced from:
@see {https://www.youtube.com/watch?v=POVQBI9icDs} 
@brief also deactivates and clears missles when the reach the game border 
@see {update_missles}*/
void check_hit();

/** Checks if ship has moved through collectible by checking if the centre of the collectible is within the bounds of the ship's hit box.
@see {check_hit}*/
void check_collect (); 

/** every 10 seconds the function sets a flag signaling that the oldest collectible has expired
@see {print_collect}*/
void collect_timeout();

/** re-draws the collectible in case another object has removed it off the screen*/
void update_collect();

/** prints a collectible at random x/y coordinates every 5 seconds. Up to two collectibles can exist at a time. If two collectibles exist already 
the finction will alternate between which one expires when the collec_timeout_flag is set.
@see {collect_timeout}*/
void print_collect(); 

/** decrements the timer every second and prints the time left above the game borders*/
void time_left();