/**
@file   main.h
@brief  Header file containing functions prototypes, defines and global variables.
@brief  Parking Sensor Project (Year 2)
@brief  ELEC 2645
@author Arnau Segura Doncel
@date   March 2016
*/
#ifndef MAIN_H
#define MAIN_H

#include "mbed.h"
#include "N5110.h"
#include "SRF02.h"
#include "TMP102.h"

//Inputs

/**
@namespace SRF02
@brief Ultra sonic distance sensor I2C mode SDA and SCL clock lines (no pull-up resitors required)
*/
SRF02 srf(I2C_SDA,I2C_SCL);

/**
@namespace TMP102
@brief Temperature sensor I2C mode SDA and SCL clock lines (no pull-up resitors required)
*/
TMP102 tmp102(I2C_SDA,I2C_SCL);

/**
@namespace Upper_Button
@brief Interrupt button (top button)
*/
InterruptIn topButton(PTB19);

/**
@namespace Lower_Button
@brief Interrupt button (bottom button)
*/
InterruptIn botButton(PTB18);

/**
@namespace sw2
@brief sw2 on the K64F
*/
InterruptIn sw2(SW2);

/**
@namespace sw3
@brief sw3 on the K64F
*/
InterruptIn sw3(SW3);

/**
@namespace Potentiometer
@brief An analogue input
*/
AnalogIn pot(PTB10);

//Outputs

/**
@namespace LCD
@brief LCD screen outputs
*/
N5110 lcd(PTE26,PTA0,PTC4,PTD0,PTD2,PTD1,PTC3);

/**
@namespace Screens_LED
@brief onboard LED on the nokia N5110 screen
*/
PwmOut screenLED(PTC3);

/**
@namespace Buzzer
@brief PWM controlled buzzer that serves as sound alert
*/
PwmOut buzzer(PTA2);

/**
@namespace Warning_LED
@brief LED that will warn the user
*/
DigitalOut LED(PTC2);

/**
@namespace red_led
@brief red LED on the K64F
*/
DigitalOut r_led(LED_RED);

/**
@namespace green_led
@brief green LED on the K64F
*/
DigitalOut g_led(LED_GREEN);

/**
@namespace blue_led
@brief blue LED on the K64F
*/
DigitalOut b_led(LED_BLUE);

/**
@namespace serial
@brief output serial for USB communication (debugging)
*/
Serial pc(USBTX,USBRX);

//Rest of objects
/**
@namespace sensorTicker
@brief Sensor ticker that will periodically call for distance and temperature measurements
*/
Ticker sensorTicker;

/**
@namespace buttonsDebounce
@brief Button Debounce ticker that will periodically call everytime a button has been pressed and will prevent multiple inputs from being recognised
*/
Ticker buttonsDebounce;

/**
@namespace potTicker
@brief Potetiometer ticker that will periodically call to check the position of the potentiometer
*/
Ticker potTicker;

//functions
/**
method that will get the average of 10 distance measurements
@returns Average of 10 distance measurements
*/
float getDistance();

/**
Method that draws the car on the screen
*/
void drawCar();

/**
Method that lets the user know the car is really close to an object
*/
void reallyCloseDistance();

/**
Method that lets the user know the car is close to an object
*/
void closeDistance();

/**
Method that lets the user know the car is almost close to an object
*/
void mediuDistance();

/**
Method that lets the user know the car far from an object
*/
void farDistance();

/**
Method that prints the bars to show the user the obstacles distance
*/
void printBars();

/**
Method that prints the distance and the temperature on the screen
*/
void numericMode();

/**
Initialises serial communication
*/
void init_serial();

/**
Initialises all the K64F's on board peripherals
*/
void init_K64F();

/**
Sets the buzzers duty cycle to 0
*/
void buzzerOFF();

/**
Sets the buzzers duty cycle to 50%
*/
void buzzerON();

/** @name Interrupt Service Routine Functions
* Sets the functions of the ISR
*/
///@{
/**
Time triggered interrupt method
*/
void timer_isr();

/**
Time triggered interrupt method
*/
void timer1_isr();

/**
Time triggered interrupt method
*/
void timer2_isr();

/**
Event triggered interrupt method
*/
void botButton_isr();

/**
Event triggered interrupt method
*/
void topButton_isr();
///@}

/** @name Menu related functions
* All the functions that set up the menus of the project
*/
///@{
/**
Scans the second bank of 6 and inverts all the pixels
*/
void MenuLine1();

/**
Scans the second bank of 6 and inverts some of the pixels
*/
void MenuLine1Short();

/**
Scans the third bank of 6 and inverts all the pixels
*/
void MenuLine2();

/**
Scans the third bank of 6 and inverts some of the pixels
*/
void MenuLine2Short();

/**
Scans the fourth bank of 6 and inverts all the pixels
*/
void MenuLine3();

/**
Scans the fourth bank of 6 and inverts some of the pixels
*/
void MenuLine3Short();

/**
Scans the fifth bank of 6 and inverts all the pixels
*/
void MenuLine4();

/**
Scans the fifth bank of 6 and inverts some of the pixels
*/
void MenuLine4Short();

/**
Scans the sixth bank of 6 and inverts all the pixels
*/
void MenuLine5();

/**
Scans the sixth bank of 6 and inverts some of the pixels
*/
void MenuLine5Short();

/**
Sets up the intro screen.
Displays the intro message
*/
void introScreen();

/**
Sets up the main menu.
Normalises potentiometer values to the highlighted bank of text
in order to navigate the menus more easily
*/
void mainMenu();

/**
Simple function that will send the user back to the main menu whenever the bottom button is pressed
*/
void goToMainMenu();

/**
Simple function that will send the user to a certain screen depending on what state it is
*/
void goToSelectedOption();

/**
Sets up the peripherals menu. The peripherals are the LED and the buzzer
*/
void peripheralsMenu();

/**
Sets up the modes menu. The user will be able to use the parking sensor 
or a numerical value sensor that will indicate both temperature and distance.
*/
void modesMenu();

/**
Sets up the sound menu. Either turn sound off or on
*/
void soundMenu();

/**
Sets up the LED menu. The led will either turn on at
critical distances from an object or it won't
*/
void LEDMenu();

/**
Sets up the brightness menu. The backlight led will go from a range of
0% to 100% brightness.
*/
void brightnessMenu();

/**
Sets up the day/night menu. The lcd will change from normal mode to inverse mode
when the user desires so
*/
void nightMenu();
///@}

//Variables
float distance; /*!< @brief Float that stores the average distance measurement from the getDistance function*/

float dist; /*!< @brief Float that stores the average distance measurement to be used in the modes functions*/

float T; /*!< @brief Float that stores the temperature measurement to be used in the modes functions*/

int emptyVariable; /*!< @brief Integer that serves as an empty variable to store the distance average measurements*/

int realDistance; /*!< @brief Integer that stores the distance measurements in cm from the distance sensor*/

int temp; /*!< @brief Integer that stores the floored temperature measurement to be used in the parking mode function [printBars()]*/

int current_screen; /*!< @brief Integer that serves as an empty variable to store the current screen state*/

int selected_option; /*!< @brief Integer that serves as an empty variable to store the selected option once an interrupt has occurred*/

int led_state; /*!< @brief Integer that serves as an empty variable to store the LED state (either on or off)*/

int buzzer_state; /*!< @brief Integer that serves as an empty variable to store the buzzer state (either on or off)*/

int circle_position_SOUND = 27; /*!< @brief Integer that stores the initial y value for the circles in the sound menu*/

int circle_position_LED = 27; /*!< @brief Integer that stores the initial y value for the circles in the LED menu*/

int circle_position_brightness = 43; /*!< @brief Integer that stores the initial y value for the circles in the brightness menu*/

int circle_position_NIGHT = 27; /*!< @brief Integer that stores the initial y value for the circles in the night menu*/

int length1; /*!< @brief Integer that stores the length of the distance string to be printed in the numeric mode*/

int length2; /*!< @brief Integer that stores the length of the temperature string to be printed in the numeric mode*/

/** @name Constant integers
* Constant integers used as unchangeable states for menu navigation and states of LED and buzzer
*/
///@{

const int menu_state = 0; /*!< @brief Constant integer that signals the menu state*/

const int peripherals_state = 1; /*!< @brief Constant integer that signals the peripherals state*/

const int brightness_state = 2; /*!< @brief Constant integer that signals the brightness state*/

const int sound_state = 3; /*!< @brief Constant integer that signals the sound menu state*/

const int ledMenu_state = 4; /*!< @brief Constant integer that signals the LED menu state*/

const int nightMenu_state = 5; /*!< @brief Constant integer that signals the night menu state*/

const int modes_state = 6; /*!< @brief Constant integer that signals the modes state state*/

const int parkingMode_state = 7; /*!< @brief Constant integer that signals the parking mode state*/

const int numericMode_state = 8; /*!< @brief Constant integer that signals the numeric mode state*/

const int LEDON_state = 9; /*!< @brief Constant integer that signals the LED is ON state*/

const int LEDOFF_state = 10; /*!< @brief Constant integer that signals the LED is OFF state*/

const int buzzerON_state = 11; /*!< @brief Constant integer that signals the buzzer is ON state*/

const int buzzerOFF_state = 12; /*!< @brief Constant integer that signals the buzzer is OFF state*/
///@}

char buffer[14]; /*!< @brief Char that stores an array of characters to print the distance in the numeric mode*/

char buffer2[14]; /*!< @brief Char that stores an array of characters to print the temperature in the numeric mode and parking mode*/

//Flags
/** @name Interrupt Service Routine Flags
* Set flags in ISR
*/
///@{
volatile int g_timer_flag = 0; /*!< @brief Volatile integer that serves as a time triggered ISR for the sensor ticker*/

volatile int g_timer1_flag = 0; /*!< @brief Volatile integer that serves as a time triggered ISR for the button debouncer ticker*/

volatile int g_timer2_flag = 0; /*!< @brief Volatile integer that serves as a time triggered ISR for the potentiometer reading ticker*/

volatile int g_topButton_flag = 0; /*!< @brief Volatile integer that serves as an event triggered ISR for when the top button is pressed*/

volatile int g_botButton_flag = 0; /*!< @brief Volatile integer that serves as an event triggered ISR for when the bottom button is pressed*/
///@}

#endif
