/**
@file main.h
@brief Header file containing functions prototypes, defines and global variables for menu and pointer
@brief Menu is construct using finit state mechine. 
@brief The input for finit state mechine is the positer position. 
@brief Revision 1.0. 
@author JIANWEI CHEN
@date   May 2016
*/
#ifndef MAIN_H
#define MAIN_H

#include "mbed.h"
//        VCC, SCE, RST, D/C, MOSI,SCLK, LED
N5110 lcd(PTE26,PTA0,PTC4,PTD0,PTD2,PTD1,PTC3);
//                MOSI, MISO, SCK, CS
SDFileSystem sd(PTE3, PTE1, PTE2, PTE4, "sd");

FILE *fp; /*!< File pointer */

DigitalOut red_led(PTA1);
DigitalOut green_led(PTC2);
InterruptIn button(PTB18);
PwmOut buzzer (PTA2);

////////////////////////////////////////////////////////
// Variables
////////////////////////////////////////////////////////
/**
Create ticker object for menu
@brief ticker to control the speed of updating the menu
*/
Ticker menuTimer;

volatile int g_menuTimer_flag = 0; /*!< Flag for menu ticker */
volatile int g_button_flag = 0; /*!< Flag for button interrupt */

/**State of menu
@brief Variable control the state of menu
@note state 0 - main menu
@note state 1 - game level choosing menu
@note state 2 - show highest score menu
@note state 3 - game with easy level
@note state 4 - game with hard level
*/
int state=0;

int pointer_position=0;/*!< Variable control the position of pointer */

/**struct for state
*/
struct State {
    int nextState[3];  // array of next states
};
typedef const struct State STyp;

STyp fsm[5] = {
    {1,2,0}, // states chosen according to inputs(pointer position)
    {3,4,0},
    {0,0,0},
    {0,0,0},
    {0,0,0}
};

//////////////////////////////////////////////////////////////////
// Functions
//////////////////////////////////////////////////////////////////

/**
Menu isr function
*/
void menuTimer_isr();

/**
Initialise the program and lcd, mbed initial settings
*/
void init();

/**
Button interrupt isr function
*/
void button_isr();

/**Drawing the pointer at (x,y)
@brief Drawing the pointer image at (x,y) according to inputs
@param x - x co-ordinate of lcd
@param y - y co-ordinate of lcd
@param fill - fill = 0 white, fill = 1 black
*/
void pointer_image(int x,int y,int fill);//0-white 1-black

/** Move the pointer
@brief Function to control the pointer according to joystick
*/
void pointer();


/////////////////////////////////////////////////
/// Function Definitions
/////////////////////////////////////////////////
void menuTimer_isr()
{
    g_menuTimer_flag = 1;
}


void button_isr()
{
    g_button_flag=1;
}


void pointer()
{
    updateJoystick(); // get joystick direction
    if (joystick.direction==UP) {
        if(pointer_position==0) { // pointer at the top
            pointer_position=0;
        } else {
            pointer_position--;
        }
    } else if(joystick.direction==DOWN) {
        if(pointer_position==1&&(state!=1)) { //only game level choosing menu have 3 pointer positions
            pointer_position=1;
        } else if (pointer_position==2&&(state==1)) {
            pointer_position=2;
        } else {
            pointer_position++;
        }
    }

    if(pointer_position==0) {
        pointer_image(10,27,1); // draw pointer
        pointer_image(10,35,0); // clear pointer at other positions
        pointer_image(10,43,0);
    } else if(pointer_position==1) {
        pointer_image(10,27,0);
        pointer_image(10,35,1);
        pointer_image(10,43,0);
    } else if(pointer_position==2) {
        pointer_image(10,35,0);
        pointer_image(10,43,1);
    }

}


void pointer_image(int x,int y,int fill)
{
    if(fill==0) {// draw white pointer
        lcd.drawLine(x,y,x-4,y+2,0);
        lcd.drawLine(x,y,x-4,y-2,0);
    } else { // draw black pointer
        lcd.drawLine(x,y,x-4,y-2,1);
        lcd.drawLine(x,y,x-4,y+2,1);
    }
}


void init(){
    wait(2.0);  // short delay for power to settle
    // initialise sd card
    fp = fopen("/sd/topscore.txt", "r");
    fclose(fp);  // ensure you close the file after reading
    lcd.init(); // initialise lcd
    lcd.normalMode();      // normal colour mode
    lcd.setBrightness(1.0); // put LED backlight on 100%
    
    lcd.printString("Calibrating",0,0);
    lcd.printString("Joystick........",0,1);
    lcd.printString("PLEASE DO NOT",0,3);
    lcd.printString("MOVE JOYSTICK.....",0,4);
    lcd.refresh();
    calibrateJoystick();
    wait(4.0);
    
    lcd.clear();
    lcd.refresh();
    button.mode(PullDown);
    red_led=0;
    green_led=0;
    button.fall(&button_isr);
    menuTimer.attach(&menuTimer_isr,0.1); // attach menu ticker
    buzzer.period(1.0/4000.0);  // set buzzer period to 4 kKz
}
#endif