/**
@file main.cpp
@brief Main file containing the code for the game.
@brief Revision 1.0.2
@author Jack A. Townsend
@date June 2016
*/
#include "mbed.h"
#include "N5110.h"
#include "time.h"
#define SENS 0.4
#define XCALI 0.5
#define YCALI 0.5

/**
Allows transmission of information over USB via serial for debugging purposes
*/
Serial pc(USBTX, USBRX);

/**
Sets the display pins for the N5110 LCD
*/
N5110 lcd(PTE26,PTA0,PTC4,PTD0,PTD2,PTD1,PTC3);

/**
@namespace ticker1
@brief Ticker to allow for repeated interrupts within the code
@namespace ticker2
@brief Ticker to allow for repeated interrupts within the code
@namespace ticker3
@brief Ticker to allow for repeated interrupts within the code
*/
Ticker ticker1;
Ticker ticker2;
Ticker ticker3;

/**
@namespace Button1
@brief Interrupt for Button 1
@namespace Button2
@brief Interrupt for Button 2
@namespace Button3
@brief Interrupt for Button 3
@namespace Button4
@brief Interrupt for Button 4
@namespace JoyButton
@brief Interrupt for the Joystick Button
*/
InterruptIn Button1(PTB18);
InterruptIn Button2(PTB19);
InterruptIn Button3(PTC1);
InterruptIn Button4(PTC8);
InterruptIn JoyButton(PTB11);

/**
@namespace Pot
@brief Analog input for the Potentiometer
@namespace XJoy
@brief Analog input for the Joystick X-Axis
@namespace YJoy
@brief Analog input for the Joystick Y-Axis
*/
AnalogIn Pot(PTB10);
AnalogIn XJoy(PTB2);
AnalogIn YJoy(PTB3);

/**
@namespace rled
@brief GPIO output for the red LED
@namespace yled
@brief GPIO output for the yellow LED
@namespace gled
@brief GPIO output for the green LED
*/
DigitalOut rled(PTC10);
DigitalOut yled(PTC11);
DigitalOut gled(PTA1);

/**
@namespace flash
@brief PWM output for the flashing red LED
@namespace buzz
@brief PWM output for the piezo buzzer
*/
PwmOut flash(PTC2);
PwmOut Buzz(PTA2);

/**
@namespace seed
@brief Holds seed data for map generation
*/
int seed[10][12] = {
    {13, 15, 1, 9, 2, 3, 0, 14, 7, 6, 12, 5},
    {10, 0, 15, 3, 7, 2, 11, 12, 9, 8, 14, 5},
    {8, 12, 15, 9, 7, 13, 14, 3, 5, 1, 4, 10},
    {15, 14, 0, 2, 10, 9, 1, 7, 11, 8, 5, 6},
    {11, 7, 13, 0, 15, 4, 12, 10, 8, 2, 14, 9},
    {7, 8, 3, 12, 9, 11, 2, 5, 15, 13, 1, 6},
    {7, 9, 6, 12, 3, 4, 10, 13, 1, 11, 15, 5},
    {0, 5, 1, 2, 7, 11, 14, 6, 4, 15, 8, 10},
    {14, 7, 10, 1, 0, 4, 2, 8, 11, 15, 13, 9},
    {5, 6, 12, 2, 9, 14, 8, 15, 13, 1, 10, 6},
};


int mapdata [16][46][60] = {0}; /*!< Array for Map data*/

/**
@namespace objdata
@brief Holds sprite data for map generation
*/
int objdata [5][12][12] = {{
        ///Station sprite
        {0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0},
        {0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0},
        {0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0},
        {0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 1},
        {0, 0, 0, 1, 1, 1, 0, 1, 1, 1, 1, 1},
        {0, 0, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1},
        {1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 0, 0},
        {0, 0, 1, 1, 0, 1, 0, 0, 1, 1, 0, 0},
        {0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 0},
        {0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0},
        {0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0},
        {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}
    },

    ///Asteroid cluster sprite
    {   {0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0},
        {0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0},
        {1, 1, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0},
        {0, 1, 1, 1, 1, 0, 0, 0, 1, 1, 1, 0},
        {0, 0, 1, 1, 0, 0, 0, 0, 1, 0, 1, 1},
        {1, 0, 0, 0, 1, 0, 0, 1, 1, 1, 0, 1},
        {0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1},
        {0, 0, 1, 0, 0, 1, 0, 1, 1, 1, 1, 1},
        {0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0},
        {0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0},
        {0, 0, 1, 1, 0, 1, 0, 1, 0, 0, 0, 0},
        {0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0}
    },

    ///Planet sprite
    {   {0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0},
        {0, 0, 1, 1, 0, 1, 0, 0, 1, 1, 0, 0},
        {0, 1, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0},
        {0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 1, 0},
        {1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 1},
        {1, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 1},
        {1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1, 1},
        {1, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 1},
        {0, 1, 0, 0, 1, 1, 0, 0, 1, 0, 1, 0},
        {0, 1, 0, 0, 1, 0, 1, 1, 0, 1, 1, 0},
        {0, 0, 1, 1, 0, 1, 1, 0, 1, 1, 0, 0},
        {0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0}
    },

    ///Gas cloud sprite
    {   {0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0},
        {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
        {0, 0, 1, 0, 0, 0, 1, 0, 0, 1, 0, 0},
        {0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0},
        {0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1},
        {1, 0, 0, 1, 1, 1, 0, 1, 0, 0, 0, 0},
        {0, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 1},
        {0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0},
        {0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 0, 0},
        {0, 1, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0},
        {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0},
        {0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0}
    },

    ///Cargo icon
    {   {0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 0},
        {0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 0},
        {0, 0, 0, 1, 1, 1, 1, 1, 1, 0, 1, 0},
        {0, 0, 1, 1, 1, 1, 1, 1, 0, 0, 1, 0},
        {0, 1, 1, 1, 1, 1, 1, 0, 0, 1, 1, 0},
        {0, 1, 0, 1, 1, 0, 1, 0, 1, 0, 1, 0},
        {0, 1, 1, 0, 0, 1, 1, 1, 0, 0, 1, 0},
        {0, 1, 1, 0, 0, 0, 1, 0, 0, 1, 0, 0},
        {0, 1, 1, 0, 0, 1, 1, 0, 1, 0, 0, 0},
        {0, 1, 0, 1, 1, 0, 1, 1, 0, 0, 0, 0},
        {1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 1},
        {1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1}
    },
};
/**
@namespace ship
@brief Holds ship sprites in various rotations
*/
int ship [4][4][4] = {
    {   {0, 1, 1, 0},
        {0, 1, 1, 0},
        {1, 0, 0, 1},
        {1, 1, 1, 1}
    },

    {   {1, 1, 0, 0},
        {1, 0, 1, 1},
        {1, 0, 1, 1},
        {1, 1, 0, 0}
    },

    {   {1, 1, 1, 1},
        {1, 0, 0, 1},
        {0, 1, 1, 0},
        {0, 1, 1, 0}
    },

    {   {0, 0, 1, 1},
        {1, 1, 0, 1},
        {1, 1, 0, 1},
        {0, 0, 1, 1}
    },
};
int ZPos = 0; /*!< Z-position of player in the arrays */
int XPos = 0; /*!< X-position of player in the arrays */
int YPos = 0; /*!< Y-position of player in the arrays */
int v = 0; /*!< Generic variable */
int i = 0; /*!< Generic variable */
int x = 0; /*!< Generic variable */
int y = 0; /*!< Generic variable */
int pixel = 0; /*!< Pixel variable used in drawing loops */
int s = 0; /*!< Generic variable */
int b = 0; /*!< Generic variable */
int joystate = 3; /*!< Joystick position */
int motion = 0; /*!< Motion detection result */
int clear = 1; /*!< Variable for ensuring ship pacement in free space */
int start = 1; /*!< Record of starting screen */
int cargo = 0; /*!< Cargo amount */ 
int life = 3; /*!< Life amount */
int g = 0; /*!< Generic variable */
double cash = 0; /*!< Cash amount */
int cost = 0; /*!< Repair cost amount */


volatile int butx_flag[4] = {0}; /*!<ISR flag for all main buttons */
volatile int butj_flag = 0; /*!<ISR flag for the joystick button */
volatile int refresh_flag = 0; /*!< ISR flag for screen refreshing */


void but1_isr(); /** Unset Button one flag */
void but2_isr(); /** Unset Button two flag */
void but3_isr(); /** Unset Button three flag */
void but4_isr(); /** Unset Button four flag */
void butj_isr(); /** Unset Joystick Button flag */
void refresh_isr(); /** Unset Refresh flag */



/**
Generates a new map based on the selected seed 
*/
void MapGen();
/**
Used in conjunction with MapGen, draws objects from the sprite data into the map data 
*/
void drawobj();
/**
Based on player location, draws the data from the map array onto the screen itself 
*/
void ScreenDraw(); 
/**
Runs the main game environment 
*/
void run1();
/**
Gets the positions of the joystick axes and the potentiometer, updates the appropriate variables 
*/
void ain(); 
/**
Updates the screen and handles ship rotation and movement detection/implementation 
*/
void refresh(); 
/**
Repositions the player 
*/
void reposition(); 
/**
Implements damage to the player 
*/
void Damage(); 
/**
Collision detection 
*/
void collision_check(); 
//defined in the code
int UI(int); 
/**
Restarts map generation if a crash is detected 
*/
void crash(); 
/**
Starts the repair minigame 
*/
void repair(); 
/**
Returns to the main game 
*/
void Return();
//defined in the code 
int reset(int);






/**
Initialises the display and interrupts, then launches the game
*/
int main()
{

    pc.printf("Main\n \n");
    lcd.init(); //initialise display
    srand(time(NULL));  //Set up rand function
    JoyButton.rise(&butj_isr);
    Button1.rise(&but1_isr);
    Button2.rise(&but2_isr);
    Button3.rise(&but3_isr);
    Button4.rise(&but4_isr);
    JoyButton.mode(PullDown);
    Button1.mode(PullDown);
    Button2.mode(PullDown);
    Button3.mode(PullDown);
    Button4.mode(PullDown);
    lcd.printString("Jack",29,2);
    lcd.printString("Townsend",17,3);
    lcd.printString("2016 Leeds",12,4);
    wait(3);
    lcd.clear();
    pc.printf("Display initialised\n \n");
    pc.printf("Rand funct set\n \n");
    UI(2);
}
/**
@brief Draws a user interface based on what is required by the game
@param t - Interface number
@Returns The interface that will be implemented
*/
int UI(int t)
{
    switch(t) {
        case 0:
            s = 4;
            for (x = 0; x < 12; x++) {
                for (y = 0; y < 12; y++) {
                    pixel = objdata [s][y][x];
                    if(pixel == 0) {
                        lcd.clearPixel(x+67,y+3);
                    } else {
                        lcd.setPixel(x+67,y+3);
                    }
                }
            }
            lcd.drawRect(67,16,11,28,0);
            break;
        case 1:
            ticker2.detach();
            ticker1.detach();
            lcd.inverseMode();
            lcd.clear();
            joystate = 3;
            lcd.drawRect(0,0,83,47,0);
            lcd.drawRect(2,2,79,43,0);
            wait(0.5);
            lcd.printString("STATION:",5,1);
            wait(1);
            lcd.printString("Sell Cargo",5,3);
            wait(0.5);
            lcd.printString("Repair",5,4);
            wait(0.5);
            x = 0;
            y = 0;
            ticker1.attach(&ain, 0.125);
            while(x == 0) {
                lcd.drawRect(66,23,10,17,2);
                wait(0.2);
                switch(joystate) {
                    case 3:
                        lcd.printChar('<',66,3);
                        y = 3;
                        break;
                    case 1:
                        lcd.printChar('<',66,4);
                        y = 4;
                        break;
                    case 2:
                        x = 1;
                        break;
                    case 0:
                        x = 1;
                        break;
                }
                wait(0.2);
            }
            ticker1.detach();
            switch(y) {
                case 3:
                    cash = cash + cargo;
                    cargo = 0;
                    Return();
                    break;
                case 4:
                    cost = 3 - life;
                    if(cash < 50*cost) {
                        lcd.clear();
                        lcd.printString("Can't afford!",5,4);
                        wait(2);
                        Return();
                    } else {
                        cash = cash - 50*cost;
                        life = 3;
                        lcd.clear();
                        lcd.printString("Repaired!",5,4);
                        wait(2);
                        Return();
                    }
            }
            break;
        case 2:
            lcd.inverseMode();
            lcd.clear();
            joystate = 3;
            lcd.drawRect(0,0,83,47,0);
            lcd.drawRect(2,2,79,43,0);
            wait(0.2);
            lcd.printString("SpacE",27,1);
            wait(0.4);
            lcd.printString("-EXPLORER-",12,2);
            wait(2);
            lcd.printString("Start",5,3);
            wait(0.5);
            lcd.printString("Help",5,4);
            x = 0;
            y = 0;
            ticker1.attach(&ain, 0.125);
            while(x == 0) {
                lcd.drawRect(38,23,20,17,2);
                wait(0.2);
                switch(joystate) {
                    case 3:
                        lcd.printChar('<',40,3);
                        y = 3;
                        break;
                    case 1:
                        lcd.printChar('<',40,4);
                        y = 4;
                        break;
                    case 2:
                        x = 1;
                        break;
                    case 0:
                        x = 1;
                        break;
                }
                wait(0.2);
            }
            ticker1.detach();
            wait(0.2);
            b = rand()%10;
            UI(y);
            break;
        case 3:
            lcd.clear();
            lcd.normalMode();
            lcd.printString("Loading...",4,10);
            i=0;
            ticker3.attach(&crash, 2.5);
            i=1;
            MapGen();
            pc.printf("\nMap generated\n \n");
            ScreenDraw();
            pc.printf("Loading complete!\n");
            break;
        case 4:
            lcd.clear();
            lcd.printString("The aim is to",1,2);
            wait(0.5);
            lcd.printString("mine asteroids",1,3);
            wait(0.5);
            lcd.printString("to earn points",1,4);
            wait(3);
            lcd.clear();
            lcd.printString("Your ship is",1,2);
            wait(0.5);
            lcd.printString("controlled w/",1,3);
            wait(0.5);
            lcd.printString("the joystick",1,4);
            wait(3);
            lcd.clear();
            lcd.printString("Pressing the",1,2);
            wait(0.5);
            lcd.printString("stick fires",1,3);
            wait(0.5);
            lcd.printString("your laser",1,4);
            wait(3);
            lcd.clear();
            lcd.printString("You will need",1,2);
            wait(0.5);
            lcd.printString("to search to",1,3);
            wait(0.5);
            lcd.printString("find asteroids",1,4);
            wait(3);
            lcd.clear();
            lcd.printString("Luckily, you",1,2);
            wait(0.5);
            lcd.printString("can fly any",1,3);
            wait(0.5);
            lcd.printString("distance...",1,4);
            wait(3);
            lcd.clear();
            lcd.printString("in any",1,2);
            wait(0.5);
            lcd.printString("direction!",1,3);
            wait(2);
            lcd.printString("However...",1,4);
            wait(2);
            lcd.clear();
            lcd.printString("You MUST",1,2);
            wait(0.5);
            lcd.printString("not fly into",1,3);
            wait(0.5);
            lcd.printString("any objects!",1,4);
            wait(3);
            lcd.clear();
            lcd.printString("If you are",1,2);
            wait(0.5);
            lcd.printString("on the same",1,3);
            wait(0.5);
            lcd.printString("screen as...",1,4);
            wait(3);
            lcd.clear();
            lcd.printString("a station,",1,2);
            wait(0.5);
            lcd.printString("you can dock",1,3);
            wait(0.5);
            lcd.printString("w/ button 4",1,4);
            wait(3);
            lcd.clear();
            lcd.printString("You can sell",1,2);
            wait(0.5);
            lcd.printString("ore and get",1,3);
            wait(0.5);
            lcd.printString("repairs",1,4);
            wait(3);
            lcd.clear();
            lcd.printString("On the right",1,2);
            wait(0.5);
            lcd.printString("will be your",1,3);
            wait(0.5);
            lcd.printString("cargo bar",1,4);
            wait(3);
            lcd.clear();
            lcd.printString("It shows you",1,2);
            wait(0.5);
            lcd.printString("how full the",1,3);
            wait(0.5);
            lcd.printString("ship is",1,4);
            wait(3);
            lcd.clear();
            lcd.printString("Now get out",1,2);
            wait(0.5);
            lcd.printString("there and",1,3);
            wait(0.5);
            lcd.printString("EXPLORE!",1,4);
            wait(5);
            UI(2);
            break;
        case 5:
            lcd.clear();
            lcd.refresh();
            lcd.printString("BOOM!",29,2);
            wait(1);
            lcd.printString("Game over",20,4);
            wait(4);
            UI(2);


    }
}
void but1_isr()
{
    butx_flag[3] = 1;
    pc.printf("4\n");
}
void but2_isr()
{
    butx_flag[0] = 1;
    pc.printf("1\n");
}
void but3_isr()
{
    butx_flag[1] = 1;
    pc.printf("2\n");
}
void but4_isr()
{
    butx_flag[2] = 1;
    pc.printf("3\n");
}

void Return()
{
    XPos = 24;
    YPos = 20;
    lcd.normalMode();
    lcd.clear();
    start = 1;
    ScreenDraw();
}

void crash()
{
    lcd.clear();
    srand(b*20);
    UI(3);
}


void butj_isr()
{
    if(cargo < 53) {
        butj_flag = 1;
    } else {
        butj_flag = 0;
    }
}
/**
@brief Resets the selected flag
@param s - Flag selection
@returns a cleared flag
*/
int reset(int s)
{
    for(i=0; i<4; i++) {
        butx_flag[i] = s;
    }
}

void repair()
{
    lcd.drawRect(0,0,60,48,2);  //draws a blank rectangle to clear a specific area of screen
    lcd.printString("DAMAGE",10,2);
    lcd.printString("CRITICAL!",5,4);
    wait(2);
    reset(1);
    lcd.clear();
    lcd.printString("SYSTEMS:",2,0);
    i=rand()%3+1;   //generates a random number of faults (up to three)
    for(y=0; y<i; y++) { //for these faults,
        x=rand()%4+1; //generate a random fault number (out of 4 possible)
        butx_flag[x-1] = 0; //set the appropriate flag to zero, depending on fault chosen
        switch(x) { //displays the fault
            case 1:
                lcd.printString("!Reactor temp!",2,y+1);
                break;
            case 2:
                lcd.printString("!Coolant leak!",2,y+1);
                break;
            case 3:
                lcd.printString("!CO2 high!",2,y+1);
                break;
            case 4:
                lcd.printString("!Hatch seal!",2,y+1);
                break;
        }
    }
    for(i=0; i<4; i++) { //loops until all flags are set to 1 by the user
        if(butx_flag[i] != 1) {
            i--;
        }
        wait(0.1);
        pc.printf("Flags: %d %d %d %d\n",butx_flag[0], butx_flag[1], butx_flag[2], butx_flag[3]); //flag status for debugging
    }


}

void collision_check()
{
    switch(joystate) {
        case 0:
            for(y=-1; y<5; y++) {
                if(lcd.getPixel(XPos-1,YPos+y) != 0) {
                    life--;
                    Damage();
                }
            }
            break;
        case 1:
            for(x=-1; x<5; x++) {
                if(lcd.getPixel(XPos+x,YPos+4) != 0) {
                    life--;
                    Damage();
                }
            }
            break;
        case 2:
            for(y=-1; y<5; y++) {
                if(lcd.getPixel(XPos+4,YPos+y) != 0) {
                    life--;
                    Damage();
                }
            }
            break;
        case 3:
            for(x=-1; x<5; x++) {
                if(lcd.getPixel(XPos+x,YPos-1) != 0) {
                    life--;
                    Damage();
                }
            }
            break;
    }
}


void Damage()
{
    //Prevent tickers from interfering with the function
    ticker1.detach();
    ticker2.detach();
    wait(0.5); //Allows user to react
    if(life == 2) { //depending on the amount of life left, takes one life away per collision
        gled = 1;
    }
    if(life == 1) {
        yled = 1;
    }
    if(life == 0) { //when life becomes 0, the emergency repair minigame starts
        rled = 1;
        g++;
        repair();
        lcd.randomiseBuffer(); //generates static screens to simulate a software crash
        lcd.refresh();
        wait(0.2);
        lcd.randomiseBuffer();
        lcd.refresh();
        wait(0.1);
        lcd.clear();
        lcd.printString("Rebooting...",2,0);
        wait(0.2);
        lcd.randomiseBuffer();
        lcd.refresh();
        wait(0.1);
        lcd.clear();
        lcd.printString("Rebooting...",2,0);
        XPos = 24; //resets the player character position (avoids further collisions
        YPos = 20;
        wait(1.5);
        lcd.clear();
        reset(0);
        switch(g) {  //if g<2, the player is rewarded with one life.  If g == 2, the player's ship explodes.
            case 0:
            case 1:
                life++;
                rled = 0;
                ScreenDraw();
                break;
            case 2:
                UI(5);
                break;
        }




    }

    ticker1.attach(&ain, 0.075);  //tickers for the game are re-attached
    ticker2.attach(&refresh_isr, 0.125);

}
void refresh()
{
    lcd.refresh();
    pc.printf("tick\n");


    if(motion == 1) {
        switch(joystate) {
            case 0:
                XPos--;
                break;
            case 1:
                YPos++;
                break;
            case 2:
                XPos++;
                break;
            case 3:
                YPos--;
                break;
        }
    }
    if(butj_flag == 0) {
        collision_check();
    }

    switch(motion) {
        case 0:
            break;
        case 1:
            for(y=-1; y<5; y++) {
                for(x=-1; x<5; x++) {
                    lcd.clearPixel(XPos+x,YPos+y);
                }
            }
            for(y=0; y<4; y++) {
                for(x=0; x<4; x++) {
                    pixel = ship [joystate][x][y];
                    if(pixel == 0) {
                        lcd.clearPixel(XPos+x,YPos+y);
                    } else {
                        lcd.setPixel(XPos+x,YPos+y);
                    }
                }

            }

            break;
    }

}

void ain()
{
    if(XJoy.read() > (1+SENS) * XCALI) {
        joystate = 0;
        motion = 1;
    } else if(XJoy.read() < (1-SENS) * XCALI) {
        joystate = 2;
        motion = 1;
    } else if(YJoy.read() > (1+SENS) * YCALI) {
        joystate = 3;
        motion = 1;
    } else if(YJoy.read() < (1-SENS) * YCALI) {
        joystate = 1;
        motion = 1;
    } else {
        motion = 0;
    }

}
void refresh_isr()
{
    refresh_flag = 1;
}
void run1()
{
    pc.printf("Screen %d \n", ZPos);
    if(start == 1) {
        for(y=-1; y<5; y++) {
            for(x=-1; x<5; x++) {
                pixel = lcd.getPixel(XPos+x,YPos+y);
                if(pixel == 0) {
                } else {
                    reposition();
                }
            }
        }

    } else {
    }
    start = 0;
    for(y=0; y<4; y++) {
        for(x=0; x<4; x++) {
            pixel = ship [joystate][x][y];
            if(pixel == 0) {
                lcd.clearPixel(XPos+x,YPos+y);
            } else {
                lcd.setPixel(XPos+x,YPos+y);
            }
        }

    }
    UI(0);
    lcd.refresh();
    ticker3.detach();
    ticker1.attach(&ain, 0.075);
    ticker2.attach(&refresh_isr, 0.125);
    while(1) {
        if(refresh_flag == 1) {
            refresh_flag--;
            refresh();
        }
        if(ZPos == seed[b][11]) {
            if(butx_flag[3]) {
                butx_flag[3] = 0;
                wait(0.5);
                UI(1);
            }
        }
        if(XPos > 55) {
            ZPos = ZPos++;
            XPos = 6;
            ticker1.detach();
            ticker2.detach();
            ScreenDraw();
        } else if(XPos < 4) {
            ZPos = ZPos--;
            XPos = 54;
            ticker1.detach();
            ticker2.detach();
            ScreenDraw();
        } else if (YPos > 41) {
            ZPos = ZPos + 4;
            YPos = 4;
            ticker1.detach();
            ticker2.detach();
            ScreenDraw();
        } else if(YPos < 4) {
            ZPos = ZPos - 4;
            YPos = 41;
            ticker1.detach();
            ticker2.detach();
            ScreenDraw();
        }
        if(butj_flag == 1) {
            butj_flag = 0;
            wait(0.05);
            switch(joystate) {
                case 0:
                    for(x=2; x<12; x++) {
                        if(lcd.getPixel(XPos-x,YPos+2)!= 0) {
                            cargo++;
                        }
                        lcd.setPixel(XPos-x,YPos+2);
                        mapdata[ZPos][XPos-x][YPos+2] = 0;
                    }
                    lcd.refresh();
                    wait(0.075);
                    for(x=2; x<12; x++) {
                        lcd.clearPixel(XPos-x,YPos+2);
                    }
                    lcd.refresh();
                    break;
                case 2:
                    for(x=5; x<15; x++) {
                        if(lcd.getPixel(XPos+x,YPos+1)!= 0) {
                            cargo++;
                        }
                        lcd.setPixel(XPos+x,YPos+1);
                        mapdata[ZPos][XPos+x][YPos+1] = 0;
                    }
                    lcd.refresh();
                    wait(0.075);
                    for(x=5; x<15; x++) {
                        lcd.clearPixel(XPos+x,YPos+1);
                    }
                    lcd.refresh();
                    break;
                case 1:
                    for(y=5; y<15; y++) {
                        if(lcd.getPixel(XPos+2,YPos+y)!= 0) {
                            cargo++;
                        }
                        lcd.setPixel(XPos+2,YPos+y);
                        mapdata[ZPos][XPos+2][YPos+y] = 0;
                    }
                    lcd.refresh();
                    wait(0.075);
                    for(y=5; y<15; y++) {
                        lcd.clearPixel(XPos+2,YPos+y);
                    }
                    lcd.refresh();
                    break;
                case 3:
                    for(y=2; y<12; y++) {
                        if(lcd.getPixel(XPos+1,YPos-y)!= 0) {
                            cargo++;
                        }
                        lcd.setPixel(XPos+1,YPos-y);
                        mapdata[ZPos][XPos+1][YPos-y] = 0;
                    }
                    lcd.refresh();
                    wait(0.075);
                    for(y=2; y<12; y++) {
                        lcd.clearPixel(XPos+1,YPos-y);
                    }
                    lcd.refresh();
                    break;

            }
            for(y=0; y<0.5f*cargo; y++) {
                for(x=0; x<10; x++) {
                    lcd.setPixel(x+68,44-y);
                }
                pc.printf("Cargo: %d \n", cargo);
            }

        }
    }
}
void drawobj()
{
    pc.printf("Drawing object at:\n \n");
    pc.printf("x %1d y %1d \n", XPos, YPos);
    pc.printf("on screen %1d \n \n \n", ZPos);
    for (y = 0; y < 12; y++) {
        for (x = 0; x < 12; x++) {
            mapdata [ZPos][XPos+x][YPos+y] = objdata [s][x][y];
        }
    }
    pc.printf("Draw Complete, v = %d \n", v);
}
void ScreenDraw()
{
    volatile int butx_flag[4] = {0};
    pc.printf("Drawing screen\n \n");
    if(ZPos<0 | ZPos>15) {
        ZPos = rand()%16;
    }
    lcd.drawRect(0,0,83,47,0);
    for (y = 1; y < 47; y++) {

        for (x = 1; x < 61; x++) {

            pixel = mapdata [ZPos][y-1][x-1];
            if(pixel == 0) {

                lcd.clearPixel(x,y);
            } else {

                lcd.setPixel(x,y);
            }
        }
    }
    for (y = 0; y < 48; y++) {
        lcd.setPixel(61,y);
        lcd.setPixel(62,y);
    }
    pc.printf("\n");
    pc.printf("Buffer complete\n");
    lcd.refresh();
    pc.printf("Refreshed\n \n");
    run1();

}
void reposition()
{
    XPos = rand()%37 + 6;
    YPos = rand()%24 + 6;
    start = 1;
}

void MapGen()
{
    pc.printf("Generating map\n");
    //Array is already declared
    //Generate 15 background stars on each screen in random locations
    for(ZPos=0; ZPos<16; ZPos++) {
        for(v=0; v<10; v++) {
            XPos = rand()%46;
            YPos = rand()%60;
            mapdata[ZPos][XPos][YPos] = 1;
        }

    }
    //pc.printf("160 Debris Generated\n");
    s = 2;
    //Generate 9 gas clouds on the map in random locations
    for(v=0; v<9; v++) {
        ZPos = rand()%16;
        XPos = rand()%37 + 6;
        YPos = rand()%24 + 6;
        drawobj();

    }
    pc.printf("9 Gas Clouds Generated\n");
    s = 3;
    //Generate 6 planets on the map in seed locations
    for(v=0; v<6; v++) {
        ZPos = seed[b][v];
        XPos = rand()%37 + 6;
        YPos = rand()%24 + 6;
        drawobj();

    }
    pc.printf("6 Planets Generated\n");
    s = 1;
    //Generate 5 asteroid clusters on the map in seed locations
    for(v=0; v<5; v++) {
        ZPos = seed[b][v+6];
        XPos = rand()%37 + 6;
        YPos = rand()%24 + 6;
        drawobj();

    }
    pc.printf("5 Asteroid Clusters Generated\n");

    //Generate one station on the map in a seed location
    s = 0;
    ZPos = seed[b][11];
    XPos = rand()%37 + 6;
    YPos = rand()%24 + 6;
    pc.printf("1 Station Generated\n");
    drawobj();
    //places the character
    XPos = 24;
    YPos = 20;
}

