/**
@file main.h
@brief Header file containing functions prototypes, defines and global variables for the snake game
@brief Revision 1.0.
@author Meurig Phillips
@date   April 2016
*/

#ifndef MAIN_H
#define MAIN_H

#include "mbed.h"
#include "beep.h"
#include "N5110.h"
#include "SDFileSystem.h"
#include "Joystick.h"
#include "fsmMenu.h"

/**
@namespace lcd
@brief GPIO for LCD screen
*/
N5110 lcd (PTD3, PTA0 , PTC4 , PTD0 , PTD2 , PTD1 , PTC3);
/**
@namespace sd
@brief GPIO for SD card holder
*/
SDFileSystem sd(PTE3, PTE1, PTE2, PTE4, "sd");
/**
@namespace greenLed
@brief GPIO output for green LED
*/
DigitalOut greenLed(PTC2);
/**
@namespace redLed
@brief GPIO output for red LED
*/
DigitalOut redLed(PTA2);
/**
@namespace pot
@brief GPIO input for the POT
*/
AnalogIn pot(PTB10);
/**
@namespace buzzer
@brief GPIO output for buzzer
*/
Beep buzzer(PTA1);
/**
@namespace RB
@brief GPIO input for Right Button
*/
InterruptIn RB(PTE24);
/**
@namespace LB
@brief GPIO input for Left Button
*/
InterruptIn LB(PTE25);
/**
@namespace gameTicker
@brief ticker to apply logic and update screen
*/
Ticker gameTicker;

//Enums
/*! Enum containing current direction of snake */
enum CurrentDirection {
    up,
    down,
    left,
    right,
    centre,
};
CurrentDirection currentDirection = centre; //initialise direction

/*! Enum containing different game modes */

enum GameType {
    classicMode,
    infiniteMode,
    hardMode,
};
GameType gameType = classicMode; //initialise

//Volatile Integers for ISR
volatile int game_timer_flag = 0; /*!< flag for game timer isr */
volatile int rb_flag = 0; /*!< flag for right button isr */
volatile int lb_flag = 0; /*!< flag for left button isr */

//Integers
int randomX =  rand() % 83 + 1; /*!< random number in the range of 1 to 83 assigned to randomX */
int randomY =  rand() % 47 + 1; /*!< random number in the range of 1 to 47 assigned to randomY */  
int randomXoddEven = randomX%2; /*!< distinguish whether randomX is odd or even */
int randomYoddEven = randomY%2; /*!< distinguish whether randomY is odd or even */
int snakeTailX[100]; /*!< array for X coordinate of each snake segment */
int snakeTailY[100]; /*!< array for Y coordinate of each snake segment */
int snakeTailLength = 3; /*!< length of snake, intialised to 3 */
int score = 0; /*!< score for current round */
int top_score = 0; /*!< top score read and write from the SD card */
int fruitValue = 10; /*!< value of the fruit */
int i = 40; /*!< x origin of snake head, intialised at 41 */
int j = 22; /*!< y origin of snake head, intialised at 23 */
int prev_i; /*!< integer to store previous value of x/i */
int prev_j; /*!< integer to store previous value of y/j */
int prev2_i; /*!< integer to store previous, previous value of x/i */
int prev2_j; /*!< integer to store previous, previous value of y/j */
int pauseCount; /*!< counts how many times the player has paused */

// Booleans
bool gamePlaying = false; /*!< bool to store whether the game is in play or not */

// Function proto types
//ISR's
/** @name ISR Functions
 * Functions to raise flags
 */
///@{
void timer_isr(); /*!< Sets the gameTicker flag when called */
void rb_isr(); /*!< Sets the Right Button flag when called */
void lb_isr(); /*!< Sets the Left Button flag when called */
///@}

//Other functions
/**
Displays new fruit
*/
void generateFood();
/**
Generates new random coordinates for the fruit
*/
void newFruitXY();
/**
Receives input form joystick and updates the current direction
*/
void moveSnake();
/**
Restricted boundaries
*/
void hardWall();
/**
Map with obstacles in the way
*/
void specialMap();
/**
Infinite boundaries
*/
void wrapAround();
/**
Calculates score
*/
void scoreCalculation();
/**
Displayes scores when player dies
*/
void gameOver();
/**
Initilaises the snake so it's moving from left to right to begin with
*/
void initSnakeTail();
/**
Splash screen for the intro of the game
*/
void snakeIntro();
/**
Game paused function
*/
void gamePaused();
/**
Contains all logic for the snake game and displays accordingly
*/
void gameLogic();
/**
Main loop for the game
*/
void mainGame();
/**
Main menu loop
*/
void mainMenu();
/**
Init interrupts peripherals
*/
void initInterrupts();


#endif