/**
@file main.h
@brief Header File containing Defines, Function Prototypes and Global Variables.
@brief Revision 1.0.
@author Jack David Berriman - 200836574
@date Summer 2016
*/

#ifndef MAIN_H // Define Guard - Prevents Multiple Definitions of Same File.
#define MAIN_H


// ------------------------- INCLUDES ------------------------- //

#include "mbed.h"
#include "N5110.h" // Include Library for Nokia N5110 LCD Screen.
#include "SRF02.h" // Include Library for SRF02 Distance Sensor.
#include "TMP102.h" // Include Library for TMP102 Temperature Sensor.
#include "SoftPWM.h" // Include Library for Software PWM.


// ------------------------- DEFINES ------------------------- //

/** Pixel Per Degree = 46/35... 46 Pixels in 'Thermometer', Max 35º = 1.314285714.
*/
#define PIXEL_PER_DEGREE 1.314285714

/** Pixels Per CM = 72/250... 72 Pixels in Box, Max 250cm = 0.288.
*/
#define PIXEL_PER_CM 0.288


// ------------------------- CLASS REFERENCES ------------------------- //

/**
@namespace screen
@brief Object to Class for Nokia N5110 Library.
*/
N5110 screen(PTE26 , PTA0 , PTC4 , PTD0 , PTD2 , PTD1 , PTC3);
//           VCC,    SCE,   RST,   D/C,   MOSI,  SCLK,   LED

/**
@namespace srf02
@brief Object to Class for SRF02 Distance Sensor.
*/
SRF02 srf02(I2C_SDA,I2C_SCL);

/**
@namespace tmp102
@brief Object to Class for TMP102 Temperature Sensor.
*/
TMP102 tmp102(I2C_SDA,I2C_SCL);

/**
@namespace pc
@brief Object to Class for Serial Library
*/
Serial pc(USBTX,USBRX); // UART Connection for PC - Used for Debugging (Displays Output on 'CoolTerm.exe'.


// ------------------------- TICKERS ------------------------- //

/**
@namespace systemTicker
@brief Object for a Ticker which keeps the whole system in time.
*/
Ticker systemTicker;


// ------------------------- SYSTEM INPUTS ------------------------- //

/**
@namespace controller
@brief Potentiometer connected to AnalogIn Pin PTB10 (Used to Select Menu Items).
*/
AnalogIn controller(PTB10); // Potentiometer

/**
@namespace backward
@brief Push Button connected to InterruptIn Pin PTB19 (Used to Return to Main Menu).
*/
InterruptIn backward(PTB19); // Backward Button

/**
@namespace forward
@brief Push Button connected to InterruptIn Pin PTB18 (Various Uses - Menu Selection & Toggling Options...).
*/
InterruptIn forward(PTB18); // Forward Button


// ------------------------- SYSTEM OUTPUTS ------------------------- //

/**
@namespace piezo
@brief Piezo Buzzer connected to PwmOut Pin PTA2 (Used w/ Distance Sensor for Audible Warning).
*/
PwmOut piezo(PTA2); // Piezo (PWM)

/**
@namespace red_led
@brief Red LED connected to DigitalOut Pin PTC9 (Used w/ Distance and Temperature Sensor for Visual Warning).
*/
DigitalOut red_led(PTC9); // Red LED (Digital)

/**
@namespace yellow_led
@brief Yellow LED connected to DigitalOut Pin PTC0 (Used w/ Distance and Temperature Sensor for Visual Warning).
*/
DigitalOut yellow_led(PTC0); // Yellow LED (Digital)

/**
@namespace green_led
@brief Green LED connected to DigitalOut Pin PTC7 (Used w/ Distance and Temperature Sensor for Visual Warning).
*/
DigitalOut green_led(PTC7); // Green LED (Digital)


// ------------------------- SOFT PWM ------------------------- //

// Third Party Library for Generating Software PWM - https://developer.mbed.org/users/komaida424/code/SoftPWM/

/**
@namespace redPWM
@brief Red LED connected to DigitalOut Pin PTC9 (BUT used with 'SoftPWM' so it can flash periodically).
*/
SoftPWM redPWM(PTC9);

/**
@namespace yellowPWM
@brief Yelow LED connected to DigitalOut Pin PTC0 (BUT used with 'SoftPWM' so it can flash periodically).
*/
SoftPWM yellowPWM(PTC0);

/**
@namespace greenPWM
@brief Green LED connected to DigitalOut Pin PTC7 (BUT used with 'SoftPWM' so it can flash periodically).
*/
SoftPWM greenPWM(PTC7);


// ------------------------- K64F ON-BOARD PERIPHERALS ------------------------- //

/**
@namespace r_led
@brief Object for the K64F On-Board Red LED - Never used but necessary to turn it off.
*/
DigitalOut r_led(LED_RED);

/**
@namespace g_led
@brief Object for the K64F On-Board Green LED - Never used but necessary to turn it off.
*/
DigitalOut g_led(LED_GREEN);

/**
@namespace b_led
@brief Object for the K64F On-Board Blue LED - Never used but necessary to turn it off.
*/
DigitalOut b_led(LED_BLUE);


// ------------------------- ISR FUNCTIONS (INTERRUPT SERVICE ROUTINE) ------------------------- //

/**
@brief ISR Function for the 'systemTicker' Flag (ISR = Interrupt Service Routine).
*/
void system_isr();

/**
@brief ISR Function for the 'forward' Button InterruptIn (ISR = Interrupt Service Routine).
*/
void forward_isr();

/**
@brief ISR Function for the 'backward' Button InterruptIn (ISR = Interrupt Service Routine).
*/
void backward_isr();


// ------------------------- VOID/BOOL/INT FUNCTIONS ------------------------- //

/**
@brief Function clears the buffer - Any text or set pixels on the screen.
*/
void clearAll(); // Clears Buffer

/**
@brief Initialises the Nokia Screen by setting various parameters (i.e Button Pull-Ups/Downs and RGB LEDs).
*/
void init_K64F(); // Initialise K64F

/**
@brief Initialises the Piezo Buzzer by setting various parameters (i.e Initial Period and Duty Cycle).
*/
void init_buzzer(); // Initialise Buzzer

/**
@brief Initialises all other aspects surrounding the programme.
*/
void init_program(); // Initialise Program

/**
@brief Contains strings which display an introductory message on the screen before the programme begins.
*/
void welcomeScreen(); // Displays Start-Up Splash Screen

/**
@brief Function for displaying text and navigating around the Main Menu (Sensors/Options).
*/
void mainMenu(); // Function for Main Menu (Sensors/Options)

/**
@brief Function for displaying text and navigating around the Sensor Menu (Distance/Temperature).
*/
void sensorMenu(); // Function for Sensors Menu (Distance/Temperature)

/**
@brief Function for displaying text and navigating around the Options Menu (LEDs/Sound/Colour).
*/
void optionsMenu(); // Function for Options Menu (LEDs/Sound/Colour)

/**
@brief Function for displaying the output from the distance sensor numerically and graphically.
*/
void distanceDisplay(); // Function for Distance Display

/**
@brief Function for displaying the output from the temperature sensor numerically and graphically.
*/
void temperatureDisplay(); // Function for Temperature Screen

/**
@brief Returns the programme back to the Main Menu from any screen when the 'backward' Button is pressed.
*/
void returnToMenu(); // Returns To Main Menu (When Pressed)

/**
@brief Prints the Position of the Cursor - Used in the Main and Sensor Menus Only.
@brief The function must be used in conjuction with 'getCursorPosition'.
@param cursor - The 'Bank' Number (0-5) is written to it
*/
void printCursorPositionMain(int cursor); // Print the Position of Cursor in Main Menu & Sensor Menu

/**
@brief Prints the Position of the Cursor - Used in the OptionsOnly.
@brief The function must be used in conjuction with 'getCursorPosition'.
@param cursor - The 'Bank' Number (0-5) is written to it
@param stateOfLEDTrigger - Check LED Trigger for 1 or 0
@param stateOfSoundTrigger - Check Sound Trigger for 1 or 0
*/
void printCursorPositionOptions(int cursor, int stateOfLEDTrigger, int stateOfSoundTrigger); // Print the Position of Cursor in Options Menu

/**
@brief Reads value of Potentiometer ('controller') and assigns it to a Bank (0-5) through Multiplication.
@brief Ensures that cursor is always set to a Bank - Anything greater than 5 is set to Bank 5.
*/
int getCursorPosition(); // Gets Position of Cursor from Potentiometer

/**
@brief Uses Software PWM to cause the Red/Yellow/Green LEDs to flash at a specified period.
@brief Used with the Distance Sensor when the distance value is below 15 cm.
@returns Bank - Value of Bank (0-5) 
*/
void LEDs_Flashing(); // Flashes LEDs Using SoftPWM

/**
@brief Uses Software PWM to cause the Red/Yellow/Green LEDs to stop flashing.
@brief Called in order to explicitly tell the K64F to stop using SoftPWM.
*/
void cancel_LEDs_Flashing(); // Stops LEDs Flashing Using SoftPWM

/**
@brief Function for turning ON/OFF LEDs depending on the distance from the SRF02 Sensor.
*/
void LEDs_Distance(int d); // Function for Turning LEDs ON/OFF Depending on Distance

/**
@brief Function for turning ON/OFF LEDs depending on the temperature from the TMP102 Sensor.
*/
void LEDs_Temperature(int t); // Function for Turning LEDs ON/OFF Depending on Temperature

/**
@brief Turns all LEDs ON.
*/
void allLEDOn(); // Turns ON All LEDs

/**
@brief Turns all LEDs OFF.
*/
void allLEDOff(); // Turns OFF All LEDs

/**
@brief Turns ON the Piezo and its outputs a constant noise.
*/
void buzzerOn(); // Turns Buzzer ON

/**
@brief Turns ON the Piezo and uses a period that simulates being 'far away' - Less frequent bleeps.
*/
void buzzerFar(); // 'Beeps' Buzzer Periodically ('Far' Distance)

/**
@brief Turns ON the Piezo and uses a period that simulates being a 'safe, normal' distance - Moderately frequent bleeps.
*/
void buzzerNormal(); // 'Beeps' Buzzer Periodically ('Normal' Distance)

/**
@brief Turns ON the Piezo and uses a period that simulates being 'close' to an object - Highly freqeuent bleeps.
*/
void buzzerClose(); // 'Beeps' Buzzer Periodically ('Close' Distance)

/**
@brief After the 'forward' Button is pressed - Checks the status of a 'Trigger' (triggerLEDs).
@brief Depending on Trigger status, it prevents or enables the LEDs from illuminating in the programme.
@returns 'stateOfLED' - A 1 or 0 which is used to either fill or unfill the Cursor.
*/
bool toggleLEDs(); // Toggles LEDs (Options Menu)

/**
@brief After the 'forward' Button is pressed - Checks the status of a 'Trigger' (triggerSound).
@brief Depending on Trigger status, it prevents or enables the Piezo from sounding in the programme.
@returns 'stateOfSound' - A 1 or 0 which is used to either fill or unfill the Cursor.
*/
bool toggleSound(); // Toggles Sound (Options Menu)

/**
@brief After the 'forward' Button is pressed - Checks the status of a 'Trigger' (triggerColour).
@brief Depending on Trigger status, it changes the colour of the screen: Black-on-White or White-on-Black
*/
void toggleColour(); // Toggles Colour - Black-on-White or White-on-Black (Options Menu)


// ------------------------- STRUCTS ------------------------- //

/**
@struct AverageValues
@brief Takes the average over 10 reading from each sensor
@brief Precautionary measure to reduce the effect of random errors and improves accuracy.
*/
struct AverageValues { // Struct For Calculating Average Values (Temp/Dist Sensor)
    int averageTemperature;
    int averageDistance;
};


// ------------------------- GLOBAL VARIABLES ------------------------- // 

/**
@brief Each Screen Number denotes a different navigation menu.
@brief Initially set to zero because 'MainMenu' = Screen Number = 0.
*/
int screenNumber = 0; // Sets Initial Screen Number to 0 (Main Menu)

/**
@brief Converts value of cursor into a pixel number (0-48) - Main Menu or Sensor Menu Only.
@brief Conversion is passed into a function for drawing the option 'selector'.
*/
int y_axis_rect_main; // Y-Position of Screen Cursor (Main Menu & Sensor Menu)

/**
@brief Converts value of cursor into a pixel number (0-48) - Options Menu Only.
@brief Conversion is passed into a function for drawing the option 'selector'.
*/
int y_axis_rect_options; // // Y-Position of Screen Cursor (Options Menu)


// ------------------------- GLOBAL FLAGS ------------------------- //

/**
@brief Sets the Flag for 'systemTicker' initially to zero.
*/
volatile int g_system_flag = 0; // Flag for 'systemTicker'

/**
@brief Sets the Flag for 'forward' Button Interrupt initially to zero.
*/
volatile int g_forward_flag = 0; // Flag for Forward Button Interrupt

/**
@brief Sets the Flag for 'backward' Button Interrupt initially to zero.
*/
volatile int g_backward_flag = 0; // Flag for Backward Button Interrupt


// ------------------------- VOLATILE & NON-VOLATILE BOOLS ------------------------- //

/**
@brief Sets the status of the variable from the 'ToggleLED()' function.
@brief Used for filling or unfilling the option 'selector'.
*/
bool stateOfLED = 0; // Initial State of LED = 0

/**
@brief Sets the status of the variable from the 'ToggleSound()' function.
@brief Used for filling or unfilling the option 'selector'.
*/
bool stateOfSound = 0; // Initial State of Sound = 0

/**
@brief Trigger is used for enable/disable LED option.
@brief If trigger equals 0 = disable LEDs.
@brief If trigger equals 1 = enable LEDs.
*/
volatile bool triggerLEDs = 0; // Trigger Set to 0 for LED

/**
@brief Trigger is used for enable/disable Sound option.
@brief If trigger equals 0 = disable Piezo.
@brief If trigger equals 1 = enable Piezo.
*/
volatile bool triggerSound = 0; // Trigger Set to 0 for Sound

/**
@brief Trigger is used for enable/disable Colour option.
@brief If trigger equals 0 = Black Text/Pixels on White Background.
@brief If trigger equals 1 = White Text/Pixels on Black Background.
*/
volatile bool triggerColour = 0; // Trigger Set to 0 for Colour

#endif