/**
@file main.h
@brief Revision 1.0.
@author Robin Milward Cooney
@date   May 2015
*/

//--------------------------------------INCLUSIONS---------------------------------------------------

#include "mbed.h"
#include "N5110.h"
#include "math.h"
#include "stdint.h"
#include "SDFileSystem.h"
#include "gameCharacters.h"

//---------------------------------------------------------------------------------------------------

//-------------------------------------DEFINITIONS---------------------------------------------------

#define joystickTolerance 0.05f

//---------------------------------------------------------------------------------------------------

//--------------------------------------NAMESPACES---------------------------------------------------

/**
@namespace lcd
@brief Output pins for lcd screen pin order: VCC,SCE,RST,D/C,MOSI,SCLK,LED
*/
N5110 lcd(PTE26,PTA0,PTC4,PTD0,PTD2,PTD1,PTC3);

/**
@namespace led
@brief LED bus output, each pinout represents a LED on a LED bar, from bottom to top
*/
BusOut led(PTC8,PTC9,PTC0,PTC7,PTC5);

/**
@namespace sd
@brief SD card output, the pins are in the following order: MOSI, MISO, SCK, CS
*/
SDFileSystem sd(PTE3, PTE1, PTE2, PTE4, "sd");

/**
@namespace red_led
@brief Red LED output on mbed, used for error messages and debugging when reading and or writing on SD card
*/
DigitalOut red_led(LED_RED);

/**
@namespace green_led
@brief Green LED output on mbed, used for success messages when reading and or writing on SD card
*/
DigitalOut green_led(LED_GREEN);

/**
@namespace swJoy
@brief Switch on potentiomiter. DigitalIn and to be polled using a ticker
*/
DigitalIn swJoy(PTB18);                                 //define potentiomiter switch

/**
@namespace xPot
@brief AnalogIn that correspods the horizontal movement of the joystick
*/
AnalogIn xPot(PTB2);

/**
@namespace yPot
@brief AnalogIn that correspods the virtical movement of the joystick
*/
AnalogIn yPot(PTB3);

/**
@namespace b_A
@brief Additional pushbutton (PullDown). DigitalIn and to be polled using a ticker
*/
DigitalIn b_A(PTE24);

/**
@namespace b_A
@brief Additional pushbutton (PullDown). DigitalIn and to be polled using a ticker
*/
DigitalIn b_B(PTE25);

/**
@namespace press_bA
@brief IterruptIn corresponding to the pushbutton A (PullDown). To be used when polling using a ticker is inconvinient or inefficient
*/
InterruptIn press_b_A(PTE24);

/**
@namespace press_bA
@brief IterruptIn corresponding to the pushbutton B (PullDown). To be used when polling using a ticker is inconvinient or inefficient
*/
InterruptIn press_b_B(PTE25);

/**
@namespace PWM
@brief Pulse width modulation output conected to buzzer to contol the music
*/
PwmOut PWM(PTC11);


/**
@namespace Ticker_Menu
@brief Ticker used for polling the joysick and the digital inputs while the menu funtions are running
*/
Ticker Ticker_Menu;

/**
@namespace Ticker_Game
@brief Ticker used for polling the joysick and the digital inputs while the game funtion is running
*/
Ticker Ticker_Game;

/**
@namespace Ticker_ds
@brief Ticker used to create a more energy efficient wait() funtion
*/
Ticker Ticker_ds;

/**
@namespace tOut
@brief Timeout used to implement music. It turns the buzzer off at the correct moment
*/
Timeout tOut;

/**
@namespace Highscores
@brief File that stores an array of the top 5 highscores
*/
FILE *Highscores;

/**
@namespace Progress
@brief File that stores the progress (i.e unlocked chapeters) of the Story Mode
*/
FILE *Progress;

//--------------------------------------------------------------------------------------

//---------------------------------VARIABLES--------------------------------------------

int i=0;                   /*!< Loop counter for the game() funtion */
int recks_movement=2;      /*!< Global variable that deffines the direction of all mobs with respect to Recks (main character)*/
int g_jump=36;             /*!< Global variable that deffines the vertical possition of Recks and the bullet during the jumping action*/
int jumpUp;                /*!< Global variable that flags depending on the direction of Recks while jumping. 0 when going up, 1 when going down*/
int accel=0;               /*!< Global variable that changes every loop during the jump to increase or decrease the velociry of Recks during the jump. Providing real time physics*/
int hound_jump=0;          /*!< Global variable that changes every loop during the jump to increase or decrease the velociry of the hound during the jump. Providing real time physics*/
int fall=37;               /*!< Variable that deffines the vertical possition of Recks when he is in Quick sand, decreases at a constant velocity*/
int bullet=9;              /*!< Global variable that deffines the horizontal possition of the bullet*/
int bullet_height;         /*!< Global variable that deffines the vertical possition of the bullet*/
int menu_select;            /*!< Global variable, it the output of the fsm that deffines what item on the menus have been selected*/
int menuState=0;            /*!< Global variable, it is the current state of the fsm*/
double lives=4;                 /*!< Global variable that indicates the number of lives Recks has*/
int score=0;                /*!< Global variable inticating the overall number of points achieved duning the game, it is the variable saved to the highscores[] array after.
                            Score=h_movement/10+killscore-shield_score*/
int kill_score;             /*!< Global variable inticating the number of points achieved duning by destroying the mobsr*/
int shield_score;           /*!< Global variable counting the ammount of times the shield has been used and for every time it decreases the overall score by 10 points*/
int shield_counter=0;       /*!< Global variable that counts the number of loops the shield has been "On" for*/
int h_movement=0;           /*!< Global variable that counts the distance travelled by Recks, and is used
                                 to contol the horizontal possition of the clouds. Diffrent to recks_movement in order so the
                                 forgrond can move faster than than the background giving it a sense of perspective. Also used
                                 to calcualte the score.*/
int ammo=24;                /*!< Global variable indicating the number of bullets left to use, decreases when a bullet is shot, increasis when ammo is picked up*/
int g_shoot_loop=0;             /*!< Global variable that prevents a bullet beeing shot while there is already a bullet on screen, when the bullet is onscreen it is >0 otherwhise it is 0*/
int fire_on_screen=0;           /*!< Global variable that prevents another fireball beeing shot while there is already a fire ball on screen, when the fire ball is onscreen it is >0 otherwhise it is 0*/
int story_mode_flag=0;          /*!< Global variable that flags when the player is in the story mode, rather than in the minigame */
int lose_lives_delay_flag=0;    /*!< Global variable that flags when the player loses a life and resets to 0 after 1 second. This means that there is a recovery time after losing a life*/
int lives_delay_loop=0;         /*!< Global variable indicating the number of times the lose_lives_delay_flag statement has iterated, after a certain numeer of itterations it resets the
                                    flag and the value to 0. Mening the revovery time is over*/
int difficulty=2;               /*!< Global variable that dictates the probability of certain flags swtiching, i.e it it's low then less enemies appear therefore the game is easier*/
int story_progress=1;           /*!< Global variable that indicates the story mode progress, it increases after each chapter then autosaves to the Progress file on SD card*/
int g_story_mode_win=0;         /*!< Global variable that indicates whether the final chapter is complete*/
int bear_lives=0;               /*!< Global vairable that indicates the number of bullets taken by the bear/3*/
int t_rex_lives=0;              /*!< Global vairable that indicates the number of bullets taken by the T Rex/3*/
int g_music_count=1;            /*!< Global vairable that indicates the location of the note in the song array*/
int g_top_scores[6];            /*!< Global array that stores the 6 top scores, only the 5 highest are printed*/
float brightness=0.5;           /*!< Global vairable that indicated the brighntness of the lcd screen*/
int random_num;                 /*!< Global vairable, random number created in the generate_random_number() funtion*/
int g_g1;                       /*!< Global vaiable that is is 1 when the players selects revolver*/
int g_g2;                       /*!< Global vaiable that is is 1 when the players selects rifle*/

//******************MOVEMENT GLOBAL VARIABLES*******************
// These global variables indicate the horizontal (or vertical) movement of all the different interactive elements (mobs, pick ups, speed boosts...)
//If it's horizontal movement it is indicated by either "element name"_movement or "element name"_hMovement the vertical movement is indicaded by "element name"_vMovement
int rat_movement=95;        /*!< Global variable indicating the horizontal movement of the rat*/
int hound_hMovement=98;     /*!< Global variable indicating the horizontal movement of the hound*/
int hound_vMovement=40;     /*!< Global variable indicating the vertical movement of the hound*/
int bear_movement=100;      /*!< Global variable indicating the horizontal movement of the bear*/
int bird_hMovement=95;      /*!< Global variable indicating the horizontal movement of the bird*/
int bird_vMovement=20;      /*!< Global variable indicating the vertical movement of the bird*/
int cactus_movement=110;    /*!< Global variable indicating the horizontal movement of the cactus*/
int t_rex_movement=120;     /*!< Global variable indicating the horizontal movement of the T Rex*/
int quick_sand_movement=85; /*!< Global variable indicating the horizontal movement of the Quick sand*/
int fire_ball_hMovement=t_rex_movement-6; /*!< Global variable indicating the horizontal movement of the fire ball made by the T Rex*/
int fire_ball_vMovement=25;               /*!< Global variable indicating the vertical movement of the fire ball made by the T Rex*/
int heart_movement=90;      /*!< Global variable indicating the horizontal movement of the fire ball made by the heart pick up*/
int ammo_movement=100;      /*!< Global variable indicating the horizontal movement of the fire ball made by the ammo pick up*/
int speed_boost_movement=130;   /*!< Global variable indicating the horizontal movement of the fire ball made by the speed boost pick up*/
//**************************************************************

//*****FLAGS FOR ACCTIONS AND PRINTING INTERACTIVE ELEMENTS*****
//These falgs are randomly activated (more or less frequently depending on the difficulty). For interactive elements when the flag is "On" (=1) the
//element is printed, the flag is then cleared when the element either moves offscreen or is destroyed or picked up. For the action flags (jump, shoot
//shield) the flag is cleared when the action is over.
int jump_flag=0;            /*!< Global variable that flags (=1) when the jump button is pressed and unflags(=0) when the jump has finished*/
int shoot_flag=0;           /*!< Global variable that flags (=1) when the shoot button is pressed and unflags(=0) when the bulllet has left the screen*/
int shield_flag=0;          /*!< Global variable that flags (=1) when the shield is activated (joystick pointed upwards)and unflags(=0) after 1 second*/
int print_rat_flag=0;       /*!< Global variable that flags (=1) randomly and prints the rat onscreen. It unflags(=0) when the rat leaves the screen*/
int print_hound_flag=0;     /*!< Global variable that flags (=1) randomly and prints the hound onscreen. It unflags(=0) when the hound leaves the screen or is destroyed*/
int hound_jump_flag=0;      /*!< Global variable that flags (=1) randomly to make the hound jump. It unflags(=0) when the jump is over*/
int print_bear_flag=0;      /*!< Global variable that flags (=1) randomly and prints the bear onscreen. It unflags(=0) when the bear leaves the screen or is destroyed*/
int print_bird_flag=0;      /*!< Global variable that flags (=1) randomly and prints the bird onscreen. It unflags(=0) when the bird leaves the screen or is destroyed*/
int print_heart_flag=0;     /*!< Global variable that flags (=1) randomly and prints the heart pickup onscreen. It unflags(=0) when the pickup leaves the screen or is picked up*/
int print_ammo_flag=0;       /*!< Global variable that flags (=1) randomly and prints the ammo pickup onscreen. It unflags(=0) when the pickup leaves the screen or is picked up*/
int print_speed_boost_flag=0;    /*!< Global variable that flags (=1) randomly and prints the speed boost pickup onscreen. It unflags(=0) when the pickup leaves the screen or is picked up*/
int print_cactus_flag=0;      /*!< Global variable that flags (=1) randomly and prints the cactus onscreen. It unflags(=0) when the cactus leaves the screen*/
int print_t_rex_flag=0;        /*!< Global variable that flags (=1) randomly and prints the T Rex onscreen. It unflags(=0) when the T Rex leaves the screen or is destroyed*/
int print_fire_ball_flag=0;     /*!< Global variable that flags (=1) randomly when the print_t_rex_flag is flagged and prints the T Rex onscreen. It unflags(=0) when the fire ball leaves the screen*/
int print_quick_sand_flag=0;    /*!< Global variable that flags (=1) randomly and prints the quicksand onscreen. It unflags(=0) when the quicksand leaves the screen*/
//****************************************************************

//*******************FLAGS FOR TICKERS AND INTERRUPTS*************
volatile int g_Ticker_Menu_flag=0;  /*!<Global variable to flag the menu ticker*/
volatile int g_Ticker_Game_flag=0;  /*!<Global variable to flag the game ticker*/
volatile int g_Ticker_Music_flag=0; /*!<Global variable to flag the music ticker*/
volatile int g_Ticker_ds_flag=0;    /*!<Global variable to flag the ticker for the ticker wait funtion*/
volatile int g_press_b_A_flag=0;    /*!<Global variable to flag the Interrupt of button A*/
volatile int g_press_b_B_flag=0;    /*!<Global variable to flag the Interrupt of button B*/
//*****************************************************************

//***********************FINITE STATE MACHINE**********************
struct menuState {      /*!<Struct declaring the fsm */
    int menu_select;
    int nextState[5];
};
typedef const struct menuState STyp;

STyp fsm_main_menu[5] = {
    {0,{0,1,4,0,0}},
    {8,{1,2,0,1,1}},
    {16,{2,3,1,2,2}},
    {24,{3,4,2,3,3}},
    {32,{4,0,3,4,4}}
};

STyp fsm_settings_menu[5] = {
    {0,{0,1,2,0,0}},
    {8,{1,2,0,1,1}},
    {16,{2,3,1,2,2}},
    {24,{3,0,2,3,3}},
    {32,{4,4,4,4,4}}
};
//*****************************************************************

enum joystickDirection {    //enum type for every neccessary direction of the joystick
    CENTRE,                 //when the joystick isn't moved
    DOWN,
    UP,
    LEFT,
    RIGHT,
};

typedef struct JoyStick Joystick;       //struct for Joystick
struct JoyStick {
    float x;    // current x value
    float x0;   // 'centred' x value
    float y;    // current y value
    float y0;   // 'centred' y value
    int swJoy; // button state (assume pull-down used, so 1 = pressed, 0 = unpressed)
    joystickDirection direction;  // current direction
};

// create struct variable
Joystick joystick;



//------------------------------------------------------------------------------------------------------------------

//---------------------------------------------FUNTIONS-------------------------------------------------------------

/// Initializes all game values so that when the game ends everything returns to it's normal state
void initialize_values();

/**Waits for t/10 seconds, a more energy efficient but same funtionality as wait()
@param t - time in 0.1*seconds*/
void ticker_wait(int t);

///Generates a random munber between 0 and 9999 using the clock on the mbed as a source for the srand() funtion
void generate_random_number();

///Displayes the current number of lives on the led bar
void led_bar();

///Plays the next note form song1 array on the piezo buzzer
void play_music();

//*******************TICKERS, INTERUPTS AND TIMEOUTS****************
///Funtion that sets g_Ticker_Menu_flag=1
void Ticker_Menu_isr();

///Funtion that sets g_Ticker_Game_flag=1
void Ticker_Game_isr();

///Funtion that sets g_Ticker_ds_flag=1
void Ticker_ds_isr();

///Funtion that sets PWM.period=0.0, so that the tune stops when it times-out
void music_tOut();

///Funtion that sets g_press_b_A_flag=1
void press_b_A_isr();

///Funtion that sets g_press_b_B_flag=1
void press_b_B_isr();

//*******************************************************************

//******************************JOYSTICK*****************************
/**Funtion that reads the joystick position when it is still and form those readings it allocates
values of xPot and yPot that are equicalent to the directions UP, DOWN, LEFT, RIGHT, CENTRE*/
void calibrateJoystick();

/**Reads the current values of xPot and yPot and detemins the position of the joysick*/
void updateJoystick();

//*******************************************************************

//*******************SD CARD & FILE MANAGEMENT***********************
/**Reads the Pregress file on the SD card*/
void readSD_progress();

/**Writes the current game progress onto the Progress file on the SD card*/
void writeSD_progress();

/**Reads the Highscores file on the SD card and prints the 5 highest numbers in the Highscores array on the lcd screen*/
void readSD_and_print_top_score();

/**Writes the topscore on the 6th position in the Highscores array (so it doesn't overwrite any values that are already stored)*/
void writeSD();

///Sorts the values and orders them in descending order
void sort_top_scores();

/**Funtion to delete a file
@param filename[] - Name of file that is to be deleted*/
void delete_file(char filename[]);

///Delete the Highscores file
void Delete_Highscores();

//*****************************************************************

//************************PRINTING ON LCD**************************
///Prints the name of the game for 3 seconds when it's turned on
void intro();

///Prints the credits on the LCD
void Credits();

///Prints the floor on the LCD
void ground();

///Prints the g_heart array on the LCD
void print_heart();

///Prints the g_ammo_pickUP on the LCD
void print_ammo_pickUp();

///Prints the g_speed_boost on the LCD
void print_speed_boost();

///Prints Recks (main character), the g_recks_still_gun array on the LCD
void print_recks_still_gun();

///Prints the g_recks_moving_gun array on the LCD
void print_recks_moving_gun();

///Prints the g_recks_crouch_gun array on the LCD
void print_recks_crouch_gun();

///Prints the g_recks_shield array on the LCD
void print_recks_shield();

///Prints the g_recks_jump_gun array on the LCD
void print_recks_jump_gun();

///Prints the g_recks_falling array on the LCD
void print_recks_falling();

///Prints the g_mob_rat_p1 (position 1) array on the LCD
void print_mob_rat_p1();

///Prints the g_mob_rat_p2 (position 2) array on the LCD
void print_mob_rat_p2();

///Prints the g_mob_hound_p1 (position 1) array on the LCD
void print_mob_hound_p1();

///Prints the g_mob_hound_p2 (position 2) array on the LCD
void print_mob_hound_p2();

///Prints the g_mob_hound_dead array on the LCD
void print_mob_hound_dead();

///Prints the g_mob_bear_p1 (position 1) array on the LCD
void print_mob_bear_p1();

///Prints the g_mob_bear_p2 (position 2) array on the LCD
void print_mob_bear_p2();

///Prints the g_mob_bear_dead array on the LCD
void print_mob_bear_dead();

///Prints the g_mob_bird_p1 (position 1) array on the LCD
void print_mob_bird_p1();

///Prints the g_mob_bird_p2 (position 2) array on the LCD
void print_mob_bird_p2();

///Prints the g_mob_bear_dead array on the LCD
void print_mob_bird_dead();

///Prints the cactus array on the LCD
void print_cactus();

///Prints the g_t_rex array on the LCD
void print_t_rex();

///Prints the g_t_rex_moving array on the LCD
void print_t_rex_moving();

///Prints the g_t_rex_attack array on the LCD
void print_t_rex_attack();

///Prints the g_fire_ball_p1 (position 1) array on the LCD
void print_fire_ball_p1();

///Prints the g_fire_ball_p2 (position 2) array on the LCD
void print_fire_ball_p2();

///Prints the g_clouds array on the LCD
void print_clouds();

/**Prints the g_clouds array on the LCD. It prints it multiple times next to the
corresponding chapter. They are either locked or unlocked depending on the 
story_progress value*/
void print_locks();

/// Prints a line of lengh 16 one pixel above the ground
void print_quick_sand();

/// Prints animation of Recks sinking in quick sand
void falling_animation();

///Prints "SCORE" and the current score value on the top left of the LCD screen
void print_score();

///Pritns "AMMO" and the remaining ammo on the top right of the LCD screen
void print_ammo();
//***************************************************************

//******************************MENU'S***************************
/**Funtion that pritns the main menu options and allows the user to select 
one of the options by pressing the button A. */
void main_menu();

/**Funtion that pritns the story mode menu options and allows the user to select 
one of the chapters by pressing the button A or returning to the main menu by
pressing the button B. */
void Story_Mode();

/**Funtion that pritns the settings menu options and allows the user to select 
one of the options by pressing the button A or returning to the main menu by
pressing the button B. */
void Settings();

/**Funtion that pritns the brightness menu options and allows the user to select 
one of the different brightnesses by pressing the button A or returning to the 
main menu by pressing the button B. */
void Brightness();

/**Funtion that pritns the difficulty menu options and allows the user to select 
one of the different difficulties by pressing the button A or returning to the 
main menu by pressing the button B. */
void Difficulty();

/**Funtion that allows the user to view the highscores and returning to the main 
menu by pressing the button B. */
void Leaderboard();

/**Funtion that pritns the guns options and allows the user to select 
one of the different guns by pressing the button A or returning to the 
main menu by pressing the button B. */
void guns();
//****************************************************************

//***************************GAME PHYSICS*************************
/**Funtion that contols the motion, physics and interations of Recks with the other
 interactive elements*/
void Recks();

/**Funtion that contols the motion, physics and interations of the rat with Recks*/
void rat();

/**Funtion that contols the motion, physics and interations of the hound with Recks*/
void hound();

/**Funtion that contols the motion, physics and interations of the bear with Recks*/
void bear();

/**Funtion that contols the motion, physics and interations of the bird with Recks*/
void bird();

/**Funtion that contols the motion, physics and interations of the cactus with Recks*/
void cactus();

/**Funtion that contols the motion, physics and interations of the T Rex with Recks*/
void t_rex();

/**Funtion that contols the motion, physics and interations of the fire ball with Recks*/
void fire_ball();

/**Funtion that contols the motion, physics and interations of the quick sand with Recks*/
void quick_sand();

/**Funtion that contols the motion, physics and interations of the heart pickup with Recks*/
void heart();

/**Funtion that contols the motion, physics and interations of the ammo pickup with Recks*/
void pickUp_ammo();

/**Funtion that contols the motion, physics and interations of the speed boost with Recks*/
void speed_boost();

/**Funtion that turns on the shield*/
void shield();
//****************************************************************

//***************************GAME MODES***************************
/**This funtion sets flags for printing interactive elements, depending on the difficulty
leavel selected*/
void set_difficulty();

/**Funtion that runs the game*/
void Game();

/**Funtion that runs the game without the sory mode*/
void Minigame();

/**Funtion that prints sequential text introducing the game, and the game funtions*/
void Tutorial();

/**Funtion that prints sequential text introducing the story and then runs the game
on a easy difficulty setting*/
void Chapter1();

/**Funtion that prints sequential text expanding the story and then runs the game
on a increased difficulty setting*/
void Chapter2();

/**Funtion that prints sequential text expanding the story and then runs the game
on a increased difficulty setting*/
void Chapter3();

/**Funtion that prints sequential text expanding the story and then runs the game
with a final boss fight*/
void Chapter4();

/**Funtion that runs when the lives global variable is 0. Printing a game over message
@returns 0 or 1. 0 returns back to the main menu and 1 re-plays the game after initializing values
*/
int Game_over();

/**Funtion that stops the game and powers down the LED's until the joystick button is pressed*/
void Pause();
//*****************************************************************



