/**
@file main.h
@brief Header file containing function prototypes, defines and global variables
@author Avinash Patel
@date April 2016
*/

#ifndef MAIN_H
#define MAIN_H

#include "mbed.h"
#include "N5110.h"
#include "Joystick.h"
#include "SDFileSystem.h"
#include <algorithm>
#include <fstream>

//Direction invaders are travelling
#define LEFT 0
#define RIGHT 1

/**
@namespace r_led
@brief K64F Red LED
*/
DigitalOut r_led(LED_RED);
/**
@namespace g_led
@brief K64F Green LED
*/
DigitalOut g_led(LED_GREEN);
/**
@namespace b_led
@brief K64F Blue LED
*/
DigitalOut b_led(LED_BLUE);

/**
@namespace sw2
@brief Connects SW2 on the K64F as an Interrupt
*/
InterruptIn sw2(SW2);
/**
@namespace sw3
@brief Connects SW3 on the K64F as an Interrupt
*/
InterruptIn sw3(SW3);

/**
@namespace sd
@brief Configures the sd card file system
*/
SDFileSystem sd(PTE3, PTE1, PTE2, PTE4, "sd"); // MOSI, MISO, SCK, CS

/**
@namespace pc
@brief Serial output to USB for debugging
*/
Serial pc(USBTX,USBRX);

/**
@namespace Joystick
@brief Creates the joystick object
@brief X potentiometer, Y potentiometer, joystick button
*/
Joystick joystick(PTB3, PTB2, PTB11);

/**
@namespace shoot_button
@brief Connects the shoot button as an Interrupt
*/
InterruptIn shoot_button(PTB18);

/**
@namespace buzzer
@brief Configures the buzzer as a PwmOut
*/
PwmOut buzzer(PTA2);

/**
@namespace lcd
@brief Interfaces the N5110 Screen with the mbed
<<library /users/eencae/code/N5110/>>
*/
//         VCC,    SCE,   RST,   D/C,   MOSI,  SCLK,   LED
N5110 lcd (PTE26 , PTA0 , PTC4 , PTD0 , PTD2 , PTD1 , PTC3);

/**
@namespace update_screen
@brief Ticker object which will be used to update the screen every 0.05s
*/
Ticker update_screen;
/**
@namespace move_joystick
@brief Ticker object which will query the status of the joystick 
@brief Fires every 0.05s
*/
Ticker move_joystick;
/**
@namespace move_cannon_missile
@brief Ticker object which will move the cannon's missile up the screen
@brief Is only attached when a missile is on the screen
@brief Fires every 0.05s
*/
Ticker move_cannon_missile;
/**
@namespace move_enemies
@brief Ticker object which moves the enemies across and down the screen
@brief Inital firing speed is 1s - Decreses as no of invaders alive decreses up to 0.1s 
*/
Ticker move_enemies;
/**
@namespace move_invader_normal_missile
@brief Array of tickers which move the invaders missiles down the screen
@brief 1 Ticker per missile
@brief Only attached when the ticker's missile is visible
@brief Fires every 0.1s
*/
Ticker move_invader_normal_missile[2];
/**
@namespace move_ufo
@brief Ticker object which moves the UFO across the screen
@brief Is only attached when the UFO is on the screen
@brief Fires every 0.1s
*/
Ticker move_ufo;
/**
@namespace fire_buzzer
@brief Ticker object used to change buzzer output
@brief Fires half the period in which move enemies does
*/
Ticker fire_buzzer;

/**
@namespace joystick_cursor_regulator
@brief Timeout object which is used to stop the menu cursor jumping
@brief Fires 0.1s after being attached
*/
Timeout joystick_cursor_regulator; //Stops the cursor from jumping
/**
@namespace shoot_button_debounce
@brief Timeout object which is used to reduce noisy input from the shoot button
@brief Fires 0.125s after being attached
*/
Timeout shoot_button_debounce; //Stops the cannon from firing when transitioning from a menu to the game
/**
@namespace lp_wait
@brief Timeout object to perform a low power wait
@brief Fires at the users request
*/
Timeout lp_wait;
/**
@namespace cannon_hit
@brief Timeout object to keep cannon invunerable after it's been hit
@brief Fires 1s after being attached
*/
Timeout cannon_hit;

int screen_buffer[84][48]; /*!< Screen Buffer. Each object on the screen has a different integer assigned to it */
const int empty_pixel = 0 ; /*!< Empty Pixels */
const int cannon_pixel = 1;  /*!< Pixel number for cannon */
const int first_barrier_pixel = 2; /*!< Pixel number of barrier no 1 */
const int first_large_invader_pixel = 5; /*!< Pixel number of large invader no 1 */
const int first_medium_invader_pixel = 10; /*!< Pixel number of medium invader no 1 */
const int first_small_invader_pixel = 15; /*!< Pixel number of large invader no 1 */
const int ufo_pixel = 20; /*!< Pixel number for UFO */
const int cannon_missile_pixel = 21; /*!< Pixel number for the cannon's missile */
const int invader_normal_missile_pixel = 22; /*!< Pixel number for the invader's normal missile */

//Booleans
//Invader related bools
bool invaders_in_state2 = true; /*!< Viewing state the invaders are in */
bool invader_direction = RIGHT; /*!< Direction the invaders are travalling */
//Cannon missile bool
bool cannon_missile_on_screen = false; /*!< Stores whether the cannon has a missile on screen */
//UFO bool
bool ufo_on_screen = false; /*!< Stores whether the UFO is on the screen */
bool ufo_direction = RIGHT; /*!< Stores the direction of the UFO */
bool ufo_bonus = false; /*!< Player has activated UFO bonus scoring */
//Buzzer bool
bool is_muted = true; /*!< Whether game is muted */
bool play_sound = true; /*!< Whether the buzzer is in a sound playing state */

//Integers
int score = 0; /*!< Current score*/
int fsm_state = 0; /*!< Menu FSM state */
int brightness_state = 4; /*!< Brightness FSM state */
int buzzer_state = 0; /*!< Buzzer FSM state */
int cursor_y_pos = 0; /*!< Cursor position */
int shot_count; /*!< No of missiles player has fired for ufo bonus */
//Cannon related integers
int number_of_lives = 3; /*!< Number of lives remaining */
int cannon_xpos = 15; /*!< X position of the cannon */ 
const int cannon_ypos = 43; /*!< Y position of the cannon */
//Cannon missile related integers
int cannon_missile_x_pos = 28; /*!< X position of the cannon's missile */
int cannon_missile_y_pos = 40; /*!< Y position of the cannon's missile */
//Invader related integers
int no_of_alive_invaders = 15; /*!< Number of alive invaders */
int down_count = 0; /*!< Times invaders have gone down */
//Invader missile related integers
int invader_strong_missile_x_pos;
int invader_strong_missile_y_pos;
//UFO integers
int ufo_x_pos; /*!< X position of the UFO */
const int ufo_y_pos = 1; /*!< Y position of the UFO */

//Floats
float ticker_period = 1; /*!< Time period the move enemies ticker refreshes */

//Enums
/*! Enum containing invader status */
enum Status {
    dead, /*!< Invader is dead - no longer on screen */ 
    dying, /*!< Invader is dying - has been hit with cannon missile but still on the screen */
    alive /*!< Invader is alive */
};
/*! Enum containing invader type */
enum Invader {
    small, /*!< Small invader */
    medium, /*!< Medium invader */
    large,  /*!< Large invader */
    none /*!< No invader */
};
/*! Enum containing game status */
enum GameState {
    menu, /*!< Menu state */
    game, /*!< Game state */
    paused, /*!< Pause game state */
    save, /*!< Save game state */
    load, /*!< Load game state */
    scores, /*!< High scores state */
    settings, /*!< Setting screen state */
    over /*!< Game over state */
};
GameState game_state = menu; /*!< Holds what state the game is currently in */

//Bit-maps
/*! Bitmap of cannon */
const bool cannon_bitmap[5][9] = {
    {0, 0, 0, 0, 1, 0, 0, 0, 0},
    {0, 0, 0, 1, 1, 1, 0, 0, 0},
    {0, 1, 1, 1, 1, 1, 1, 1, 0},
    {1, 1, 1, 1, 1, 1, 1, 1, 1},
    {1, 1, 1, 1, 1, 1, 1, 1, 1}
};
//Bitmaps for small invaders
/*! Bitmap of 1st state small invader */ 
const bool small_invader_bitmap_1[6][8] = { //First state
    {0, 0, 0, 1, 1, 0, 0, 0},
    {0, 1, 1, 1, 1, 1, 1, 0},
    {1, 1, 0, 1, 1, 0, 1, 1},
    {1, 1, 1, 1, 1, 1, 1, 1},
    {0, 1, 0, 1, 1, 0, 1, 0},
    {1, 0, 1, 0, 0, 1, 0, 1}
};
/*! Bitmap of 2nd state small invader */ 
const bool small_invader_bitmap_2[6][8] = { //Second state
    {0, 0, 0, 1, 1, 0, 0, 0},
    {0, 1, 1, 1, 1, 1, 1, 0},
    {1, 1, 0, 1, 1, 0, 1, 1},
    {1, 1, 1, 1, 1, 1, 1, 1},
    {1, 0, 0, 0, 0, 0, 0, 1},
    {0, 1, 0, 0, 0, 0, 1, 0}
};
//Bitmaps for medium invaders
/*! Bitmap of 1st state medium invader */ 
const bool medium_invader_bitmap_1[6][10] = { //First state
    {1, 0, 0, 1, 0, 0, 1, 0, 0, 1},
    {1, 0, 1, 1, 1, 1, 1, 1, 0, 1},
    {1, 1, 1, 0, 1, 1, 0, 1, 1, 1},
    {0, 1, 1, 1, 1, 1, 1, 1, 1, 0},
    {0, 0, 1, 0, 0, 0, 0, 1, 0, 0},
    {0, 1, 0, 0, 0, 0, 0, 0, 1, 0}
};
/*! Bitmap of 2nd state medium invader */ 
const bool medium_invader_bitmap_2[6][10] = { //Second state
    {0, 0, 0, 1, 0, 0, 1, 0, 0, 0},
    {0, 0, 1, 1, 1, 1, 1, 1, 0, 0},
    {0, 1, 1, 0, 1, 1, 0, 1, 1, 0},
    {1, 1, 1, 1, 1, 1, 1, 1, 1, 1},
    {1, 0, 1, 0, 0, 0, 0, 1, 0, 1},
    {0, 0, 0, 1, 1, 1, 1, 0, 0, 0}
};
//Bitmaps for large invaders
/*! Bitmap of 1st state large invader */ 
const bool large_invader_bitmap_1[6][12] = { //First state
    {0, 0, 0, 1, 1, 1, 1, 1, 1, 0, 0, 0},
    {0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0},
    {1, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 1},
    {1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1},
    {0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0},
    {0, 0, 1, 1, 0, 0, 0, 0, 1, 1, 0, 0}
};
/*! Bitmap of 2nd state large invader */ 
const bool large_invader_bitmap_2[6][12] = { //Second state
    {0, 0, 0, 1, 1, 1, 1, 1, 1, 0, 0, 0},
    {0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0},
    {1, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 1},
    {1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1},
    {0, 0, 1, 1, 0, 1, 1, 0, 1, 1, 0, 0},
    {1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1}
};
/*! Bitmap of invaders normal missile */
const bool invader_normal_missile_bitmap[4][3] = {
    {0, 1, 0},
    {1, 1, 1},
    {0, 1, 0},
    {0, 1, 0}
};
/*! Bitmap of barriers */
const bool barrier_bitmap[8][14] = {
    {0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0},
    {0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0},
    {1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1},
    {1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1},
    {1, 1, 1, 1, 1, 1, 0, 0, 1, 1, 1, 1, 1, 1},
    {1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1},
    {1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1},
    {1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1}
};
/*! Damage bitmap applied when cannon missile hits barrier */
const bool barrier_cannon_missile_damage_bitmap[3][3] = {
    {0, 1, 0},
    {1, 0, 0},
    {1, 0, 1}
};
/*! Damage bitmap applied when normal invader missile hits barrier */
const bool barrier_invader_normal_missile_damage_bitmap[2][3] = {
    {1, 0, 1},
    {0, 1, 0}
};
/*! UFO bitmap */
const bool ufo_bitmap[5][14] = {
    {0, 0, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 0, 0},
    {0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0},
    {0, 1, 1, 0, 1, 0, 1, 1, 0, 1, 0, 1, 1, 0},
    {1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1},
    {0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0}
};

/**
@struct Invaders
@brief Holds data for each invader
*/
struct Invaders {
    int x_pos; /*!< X position of invader */
    int y_pos; /*!< Y position of invader */
    enum Status status; /*!< Status of invader */
}; 
Invaders small_invader[5]; /*!< Declares 5 Small Invaders */
Invaders medium_invader[5]; /*!< Declares 5 Medium Invaders */
Invaders large_invader[5]; /*!< Declairs 5 Large Invaders */
/**
@struct InvaderNormalMissiles
@brief Hold data for normal invader missiles
*/
struct InvaderNormalMissiles {
    int x_pos; /*!< X position of missile */
    int y_pos; /*!< Y position of missile */
    bool fired; /*!< Whether missile is on the screen */
};
InvaderNormalMissiles invader_normal_missile[2]; /*!< Declares 2 normal invaders */
/**
@struct Barriers
@brief Holds barrier data
*/
struct Barriers {
    int x_pos; /*!< X position of barrier */
    int y_pos; /*!< Y position of barrier */
    bool before_bitmap[8][14]; /*!< State of the displayed barrier */
    bool after_bitmap[8][14]; /*!< State of barrier to be displayed */
};
Barriers barrier[3]; /*!< Declares 3 barriers */
/**
@struct HighScores
@brief Holds leaderboard data
*/
struct HighScores {
    char name[3];
    int score; 
};
HighScores high_scores[3]; /*!< Holds data for 5 scores */
/**
@struct FSMMenus
@brief Hold the transition data for menus
*/
struct FSMMenus {
    const GameState output; /*!< Output of the current state */
    /**
    @var next_state
    @brief Stores the next state of the FSM
    @brief Index 0 is when joystick is down
    @brief Index 1 is when joystick is up
    */
    const int next_state[2];
};
/*! 4 state FSM for main menu */
FSMMenus fsm_main_menu[4] = {
    {game, {1, 3}},
    {load, {2, 0}},
    {scores, {3, 1}},
    {settings, {0, 2}}
};
/*! 3 state FSM for pause menu */
FSMMenus fsm_paused[3] = {
    {game, {1, 2}},
    {save, {2, 0}},
    {menu, {0, 1}}
};

/*! 4 state fsm for settings menu */
FSMMenus fsm_settings[4] = {
    {settings, {1, 3}},
    {settings, {2, 0}},
    {settings, {3, 1}},
    {settings, {0, 2}}
};
/**
@struct FSMBrightness
@brief Holds the brightness transition data
*/
struct FSMBrightness {
    float output;
    int next_state;
};
/*! 5 state fsm for brightness storage */
FSMBrightness fsm_brightness[5] = {
    {0.00, 1},
    {0.25, 2},
    {0.50, 3},
    {0.75, 4},
    {1.00, 0}
};
/**
@struct FSMBuzzer
@brief Holds the transition infomation for the buzzer
*/
struct FSMBuzzer {
    const float frequency;
    const int next_state;
};
/*! 4 state FSM holding buzzer frequencies */
FSMBuzzer fsm_buzzer[] = {
    {235.0, 1},
    {235.0, 2},
    {196.0, 3},
    {275.0, 0}
};

/** @name ISR Flags
 * Flags set in ISR's
 */
///@{
volatile bool g_update_screen_flag = true; /*!< Update screen flag */
volatile bool g_move_joystick_flag = true; /*!< Move joystick flag */
volatile bool g_move_enemies_flag = true; /*!< Move enemies flag */
volatile bool g_shoot_pressed_flag = false; /*!< Shoot pressed flag */
volatile bool g_move_cannon_missile_flag = false; /*!< Move cannon missile flag */
volatile bool g_move_invader_normal_missile_flag[2] = {false, false}; /*!< Move invader normal missile flags */
volatile bool g_cannon_hit_flag = false; /*!< Cannon hit flag */
volatile bool g_joystick_cursor_regulator_flag = false; /*!< Joystick regulator flag */
volatile bool g_shoot_button_debounce_flag = false; /*!< Shoot button debounce flag */
volatile bool g_move_ufo_flag = false; /*!< Move UFO flag */
volatile bool g_fire_buzzer_flag = true; /*!< Fire Buzzer flag */
volatile bool g_lp_wait_flag = false; /*!< LP Wait flag */
///@}

//Function prototypes
//ISR's
/** @name ISR Functions
 * Functions to set ISR values
 */
///@{
void update_screen_isr(); /*!< Sets the update screen flag when called */
void move_joystick_isr(); /*!< Sets the move joystick flag when called */
void move_enemies_isr(); /*!< Sets the move enemies flag when called */
void shoot_pressed_isr(); /*!< Sets the shoot pressed flag when called */
void move_cannon_missile_isr(); /*!< Sets the move cannon missile flag when called */
void cannon_hit_isr(); /*!< Clears the cannon hit flag when called */
void joystick_cursor_regulator_isr(); /*!< Sets the joystick regulator flag when called */
void shoot_button_debounce_isr(); /*!< Sets the shoot button debounce flag when called */
//Function pointer to move invader normal missile isr
void (*move_invader_normal_missile_isr[2])(); /*!< Function pointer to help attach move invader normal missile isr's */
void move_invader_normal_missile_0_isr(); /*!< Sets the move invader normal missile [0] flag when called */
void move_invader_normal_missile_1_isr(); /*!< Sets the move invader normal missile [1] flag when called */
void move_ufo_isr(); /*!< Sets the move ufo flag when called */
void fire_buzzer_isr(); /*!< Sets the fire buzzer flag when called */
void lp_wait_isr(); /*!< Clear the lp wait flag when called */
void shoot_pressed_rise_isr(); /*!< Attaches shoot button debounce timeout */
///@}
//Other functions
/** Hangs - flashing an LED
*/
void error();
/** Sets up serial port to 115200 baud
*/
void init_serial();
/** Set-up the on-board LEDs and switches
*/
void init_K64F();
/** Configures the shoot button
@brief Sets the shoot button to PullUp mode. 
@brief Sets the falling edge of the swich to call the shoot_pressed_isr
*/
void init_shoot();
/** Seeds the random number generator with noise from an analog in pin
*/
void init_rng();
/** Clears the screen buffer and runs invader and barrier initalisation functions
@brief Sets the flags so invaders show up straight away. 
@brief Clears the missile fired flags
*/
void InitaliseGame();
/** Contains game logic
@brief Calls screen refresh, cannon movement, barrier drawing, invader movement, missile movement, pause and game over transition functions
@brief Only calls functions when ISR flags are set
*/
void Game();
/** Loops through the screen buffer
@brief If pixels are set in the screen buffer set them on the lcd
@brief Otherwise clear them
*/
void UpdateScreen();
/** Clears the cannon from the buffer
@brief Increments or decrements the x coordinate of the cannon depending on joystick position - Capped between 0 and 75
@brief Redraws the cannon onto the buffer
*/
void MoveCannon();
/** @fn Sets the position and status of the small invaders
*/
void InitSmallInvaders();
/** @fn Cycles through small invaders
@brief If their status is not equal to dead clear them from the buffer
*/
void ClearSmallInvaders();
/** Cycles through the small invader current bitmap and sets the pixels it occupys in the buffer to empty
@param invader_no - Small invader object number between 0-4
*/
void ClearSingleSmallInvader(int invader_no);
/** Cycles through small invaders
@brief If their status is equal to alive set them in the buffer
*/
void DrawSmallInvaders();
/** Cycles through the small invader current bitmap and sets the pixels it occupys in the buffer to first_small_invader_pixel + invader_no
@param invader_no - Small invader object number between 0-4
*/
void DrawSingleSmallInvader(int invader_no);
/** Sets the position and status of the medium invaders
*/
void InitMediumInvaders();
/** Cycles through medium invaders
@brief If their status is not equal to dead clear them from the buffer
*/
void ClearMediumInvaders();
/** Cycles through the medium invader current bitmap and sets the pixels it occupys in the buffer to empty
@param invader_no - Medium invader object number between 0-4
*/
void ClearSingleMediumInvader(int invader_no);
/** Cycles through medium invaders
@brief If their status is equal to alive set them in the buffer
*/
void DrawMediumInvaders();
/** Cycles through the medium invader current bitmap and sets the pixels it occupys in the buffer to first_medium_invader_pixel + invader_no
@param invader_no - Medium invader object number between 0-4
*/
void DrawSingleMediumInvader(int invader_no);
/** Sets the position and status of the large invaders
*/
void InitLargeInvaders();
/** Cycles through large invaders
@brief If their status is not equal to dead clear them from the buffer
*/
void ClearLargeInvaders();
/** Cycles through the large invader current bitmap and sets the pixels it occupys in the buffer to empty
@param invader_no - Large invader object number between 0-4
*/
void ClearSingleLargeInvader(int invader_no);
/** Cycles through large invaders
@brief If their status is equal to alive set them in the buffer
*/
void DrawLargeInvaders();
/** Cycles through the large invader current bitmap and sets the pixels it occupys in the buffer to first_large_invader_pixel + invader_no
@param invader_no - Large invader object number between 0-4
*/
void DrawSingleLargeInvader(int invader_no);
/** Sets the position and loads the bitmap into the barrier objects
*/
void InitBarriers();
/** Clears the barrier with the before bitmap and redraw it from the after bitmap which has the damage applied
@brief Copies the after bitmap to the before bitmap
*/
void DrawBarriers();
/** If the invader direction is right, calculate the right limit and move the invaders right
@brief Otherwise calculate the left limit and move the invaders left
*/
void MoveInvaderXPositions();
/** Checks the status off the invaders per column, starting from the left
@brief If they're alive return the row number
@returns The left-most column invaders are alive
*/
int CalculateInvaderLeftLimit();
/** Checks the status off the invaders per column, starting from the right
@brief If they're alive return the row number
@returns The right-most column invaders are alive
*/
int CalculateInvaderRightLimit();
/** If there is still room for the invaders to move, move them 2 to the left
@brief Otherwise move them down
@param limit - Large invader object to check for space against
*/
void MoveInvadersLeft(int limit);
/** If there is still room for the invaders to move, move them 2 to the right
@brief Otherwise move them down
@param limit - Large invader object to check for space against
*/
void MoveInvadersRight(int limit);
/** Calculates the lowest alive invader
@brief If there is still space for the invader to move, move them 3 pixels down
@brief Otherwise end the game
*/
void MoveInvaderYPositions(bool new_direction);
/** Checks each row of invaders to find the lowest row which has an alive invader
@returns The lowest row which has an alive invader
*/
Invader CalculateInvaderYLimit();
/** Queries the invaders on their status in the specified column
@param column - The column to query
@returns The lowest invader alive in that column
*/
Invader LowestInvaderInColumn(int column);
/** Sets the cannon fired flag to true
@brief Configures the start coordinates
@brief Attaches the move cannon missile ticker to fire every 0.05s
*/
void FireCannonMissile();
/** If the missile is not beyond the bounds of the screen, clear it, increment the shot and run collision detection
@brief Otherwise clear the last 3 pixels of the missile, mark it as no longer on the screen and detach the ticker
*/
void MoveCannonMissile();
/** If the missile hits an invader it will clear it and increment the score or if it hits a barrier apply the appropiate damage to it
@brief Otherwise push the missile to the screen buffer
*/
void CollisionDetectionCannonMissile();
/** Finds the invader number the missile hits, sets the hit invader to dying, decrements the no_of_alive_invaders
@brief Stops the missile from travalling up the screen
@param first_pixel - Start no of invader group hit
@param row - What part of the missile hit the invader
@param invader[5] - Struct of the invader group which was hit
@returns The invader number which was hit
*/
int CannonMissileHitInvader(int first_pixel, int row, struct Invaders (&invader)[5]);
/** Calculates where to start drawing the damage bitmap over the barrier bitmap and performs the operation
@param row - Which part of the missile hit the barrier
*/
void CannonMissileHitBarrier(int row);
/** Random chance of 1/18, decrementing to a 1/3 depending on no of alive invaders, to fire a missile
@brief Only fires if they're are less than 2 on the screen
*/
void AttemptToFireInvaderNormalMissiles();
/** Finds the centre point of the chosen invader and fires the missile
@brief Attaches the ticker to move the missile every 0.05s
@param missile_no - Which missile will be fired
@param source - Type of invader which missile will be fired from (Used to calculate start position)
@param invader - Invader which missile will orginiate from
*/
void FireNormalInvaderMissile(int missile_no, Invader source, const struct Invaders (&invader));
/** Clears the missile from the screen buffer
@brief Increments the missile up the screen and runs collision detection if within screen bounds, otherwise set fired flag as false and detach the ticker
@param missile_no - Which missile is being moved
*/
void MoveInvaderNormalMissile(int missile_no);
/** Checks the bottom centre point for a collision. If it hasn't push the missile to the screen buffer
@param missile_no - Which missile is being checked
*/
void CollisionDetectionInvaderNormalMissile(int missile_no);
/** Decrements the number of lives and pauses the game for 2 seconds
*/
void InvaderNormalMissileHitCannon();
/** Calculates where to start drawing the damage bitmap over the barrier bitmap and performs the operation
@param missile - Missile which hit the barrier 
*/
void InvaderNormalMissileHitBarrier(const struct InvaderNormalMissiles (&missile));
/** Detaches game related tickers
*/
void DetachTickers();
/** Attaches game related tickers
*/
void AttachTickers();
/** Prints the pause screen and draws the selection cursor
*/
void PauseScreen();
/** Prints the text on the pause screen
*/
void PrintPauseScreen();
/** Clears the cursor and sets the next state of the supplied fsm
@brief Next state depends on joystick value
@param fsm - Finite State Machine for the current menu
*/
void MoveCursor(const struct FSMMenus *fsm);
/** Prints the menu screen and draws the selection cursor
*/
void MenuScreen();
/** Prints menu screen text
*/
void PrintMenuScreen();
/** Attempts a 1/8 chance to spawn a UFO
*/
void AttemptToSpawnUFO();
/** Clears the UFO
*/
void ClearUFO();
/** Draws the UFO
@brief If the coordinates extend past the screen the ticker is detached and the UFO is set to not on the screen
*/
void DrawUFO();
/** Predicate function to sort high scores struct
@param lhs - Object to be sorted
@param rhs - Object to be sorted
@returns lhs.score > rhs.score
*/
bool high_scores_sorter(const struct HighScores (&lhs), const struct HighScores (&rhs));
/** Prints the score screen
*/
void ScoreScreen();
/** Prints scores to screen
*/
void PrintHighScores();
/** Prints the game over screen
@brief Name on the left, scores on the right
*/
void GameOverScreen();
/** Asks the user to input 3 charactors for name
@param name[3] - Name to go with high score
*/
void GetNameForScore(int (&name)[3]);
/** Saves the high score structs to file
@param name[3] - Name to go with high score
*/
void SaveHighScore(const int (&name)[3]);
/** Saves the status of the game to the sd card
@brief File is called game_save.dat
*/
void SaveGameData();
/** Loads the status of the game to the sd card
@brief File is called game_save.dat
*/
void LoadGameData();
/**Loads High scores
*/
void LoadHighScores();
/** Prints the settings screen
*/
void SettingsScreen();
/** Prints static text on screen
*/
void PrintSettingsText();
/** Prints changing options on screen
*/
void PrintSettingsOptions();
/** Changes requested value on settings screen
*/
void EditOption();
/** Deletes game data
*/
void DeleteGameData();
/** Attaches a timeout then sets the MCU to sleep until it fires
@param t - Time to sleep
*/
void LPWait(float t);
/** Prints a splash screen and uses LPWait to wait for 3 seconds
*/
void SplashScreen();
/** Sets the buzzer frequency and activates or deactivates it
*/
void PlayBuzzer();

#endif