/**
@file main.h
@brief Header file containing function prototypes, defines and global variables
@author Max Hamilton
@date May 2015
*/
#include "mbed.h"
#include "N5110.h"
#include "tones.h"

#define DIRECTION_TOLERANCE 0.05    // tolerance of joystick direction
#define PLAYERRADIUS 2              // size of player ball

/** 
@namespace lcd
@brief Creates N5110 object for Nokia LCD screen
@brief Pin connections are VCC, SCE, RST, D/C, MOSI, SCLK, LED
*/
N5110 lcd (PTD3 , PTA0 , PTC4 , PTD0 , PTD2 , PTD1 , PTC3); // VCC, SCE, RST, D/C, MOSI, SCLK, LED

/** 
@namespace xPot
@brief AnalogIn to control the joystick's x-position
*/
AnalogIn xPot(PTB11);

/** 
@namespace yPot
@brief AnalogIn to control the joysticks's y-position
*/
AnalogIn yPot(PTB10);

/** 
@namespace button
@brief InterruptIn for joystick button
*/
InterruptIn button (PTB3);

/** 
@namespace buzzer
@brief PwmOut to control buzzer
*/
PwmOut buzzer(PTA2);

/** 
@namespace backlight
@brief AnalogIn object for 10k potentiometer to control screen brightness
*/
AnalogIn backlight(PTB2);

/** 
@namespace pollJoystick
@brief Ticker object to read the joystick position at regular intervals
*/
Ticker pollJoystick;

/** 
@namespace gameTicker
@brief Ticker object to regularly call game functions
*/
Ticker gameTicker;

/**
@namespace serial
@brief USB serial connection for debugging
*/
Serial serial(USBTX,USBRX);


///Create enumerated type (0,1,2,3 etc. for direction)
enum DirectionName {
    UP,
    DOWN,
    LEFT,
    RIGHT,
    UPRIGHT,
    UPLEFT,
    DOWNRIGHT,
    DOWNLEFT,
    CENTRE,
    UNKNOWN
};

typedef struct JoyStick Joystick;
/**
@struct Joystick
@brief Struct for joystick parameters/inputs
@param x - current x value
@param x0 - calibrated centre x value
@param y - current y value
@param y0 - calibrated centre y value
@param direction - direction joystick is set to
@see {DirectionName}
*/
struct JoyStick {
    float x;                    /// current x value
    float x0;                   /// 'centred' x value
    float y;                    /// current y value
    float y0;                   /// 'centred' y value
    int button;                 /// button state (assume pull-down used, so 1 = pressed, 0 = unpressed)
    DirectionName direction;    /// current direction
};

// creates 'joystick' instance of struct
Joystick joystick;

typedef struct Wall Wall;
/**
@struct Wall
@brief Struct for storing properties of moving walls
@param x - Current x value of wall (relative to the centre of wall gap)
@param y - Current y value of wall (relative to the centre of wall gap)
@param random - Randomly generated integer to determine when the wall starts moving
@param cooldown - Variable to stop wall reappearing directly after if disappears
@param moveFlag - flag to determine if wall is moving
@param genFlag - flag to determine if wall is generated
*/
struct Wall {
    int x;                    
    int y;                  
    int random;              
    int cooldown;              
    volatile bool moveFlag;    
    volatile bool genFlag;     
};

// Creates instances of structs
Wall leftwall;              // left-moving wall
Wall rightwall;             // right-moving wall
Wall downwall;              // down-moving wall
Wall upwall;                // up-moving wall

typedef const struct State STyp;
/**
@struct STyp
@brief Struct for game menu parameters
@param output - output of the menu state
@param next_state - array containing next state depending on joystick direction
*/

// Creates instance of struct
struct State {
    int output;                 
    int next_state[3];          
};

// array containing menu state outputs and next state possibilites
STyp fsm[4] = { 
    {1,{0,1,0}},
    {2,{1,2,0}},
    {3,{2,3,1}},
    {4,{3,3,2}}
};

typedef struct Highscore Highscore;
/**
@struct State
@brief Struct player highscores
@param one - Highest player score
@param two - Second highest player score
@param three - Three highest player score
*/
struct Highscore {
    int one;                    
    int two;               
    int three;             
};

// Creates instance of struct
Highscore hiscore;

// - - - - - - - - - - - - - - - - - - - - - - - - Variables  - - - - - - - - - - - - - - - - - - - - - - - - 

/**
@brief Player x-coordinate
*/
int i;                     

/**
@brief Player y-coordinate
*/
int j;

/**
@brief Int to keep track of how many time gameplay ticker funtion has been called
*/                     
int counter = 0;

/**
@brief Int to keep track of player score
*/                  
int score = 0;

/**
@brief how many uses of invincibility the player has left
*/  
int saves = 5;

/**
@brief Int to count how long the player stays invincible.
@brief Increases with counter variable, and resets when invincibility runs out
*/ 
int saves_cool = 0;

/**
@brief Flag to determine if player is invincible or not.
*/ 
volatile bool mortal = 1;

/**
@brief Flag for gameTicker isr.
*/ 
volatile int g_timer_flag = 0; 

/**
@brief Flag to determine if game is running
*/ 
volatile int game_running = 0;      

/**
@brief Flag for button isr.
*/ 
volatile int g_button_flag = 0;

/**
@brief Int to determine what menu item to go to depending on joystick direction
*/ 
int menu_direction;

/**
@brief Output of the menu state the player has selected
*/ 
int state;
/**
@brief Int to determine which menu state the player is on
*/ 
int menu_item;

// - - - - - - - - - - - - - - - - - - - - - - - - Function Prototypes - - - - - - - - - - - - - - - - - - - - - - - - 

/** 
@brief Calibrates josytick and gets centred value
*/
void calibrateJoystick();

/** 
@brief Reads direction joystick is moved in
*/
void updateJoystick();

/** 
@brief Initialise display
*/
void initDisplay();

/** 
@brief Initialises the gameplay variables for the start of a new game
*/
void initGame();

/** 
@brief Initialises serial and sets baud rate for data transfer
*/
void initSerial();

/** 
@brief Initialises highscores
*/
void initHiscores();

/** 
@brief Interrupt for game loop, triggered by gameTicker
*/
void gameTicker_isr();

/** 
@brief Button interrupt, triggered by joystick button
*/
void button_isr();

/** 
@brief Reads direction of the joystick and moves the player accordingly
*/
void moveBall();

/** 
@brief Moves the walls across the game screen
*/
void moveWall();

/** 
@brief Ignores wall collisions if player has triggered invincibility.
@brief Can only be used 5 times.
@brief Disables the 'checkWallCollisions()' function as long as the player is invincible.
*/
void invincible();

/** 
@brief Checks if the player has hit one of the walls.
@brief Sets the 'game_running' = 0 to end the game if collision occurs.
@brief Bypassed if 'mortal' variable is false.
*/
void checkWallCollision();

/** 
@brief Checks if the player has hit the game border.
@brief Stops the player moving off the screen.
*/
void checkBorderCollision();

/** 
@brief Redraws the screen with the new positions of the player and walls.
@brief Also redraws borders and any remianing invincibility indicators.
*/
void updateScreen();

/**
@brief Prints game instructions when Help is selected in the menu.
@brief Move through instructions by pressing the joystick.
*/
void printHelp();

/** 
@brief Shows the top three highscores when Scores is selected in the menu.
*/
void printScores();

/**
@brief Sets new highscore if one has been achieved
@brief If the latest score is set as a highscore, previous highscores are shifted down accordingly.
*/
void calculateHighscores();

/** 
@brief Makes the screen flash briefly.
@brief IMPORTANT: Do not put in the ISR as it contains a 'wait()' function.
*/
void flash();

/** 
@brief Plays a note of specific frequency and duration in seconds.
@brief If the preproccesor directive MUTE is defined as 1, the code instead waits the defined time.
@brief IMPORTANT: Do not put in the ISR as it contains a 'wait()' function.
@param freq - The note frequency (N.B. program uses definitions from the 'tone.h' file).
@param time - The time the note plays for in seconds.
*/
void playNote(int freq, float time);    

/**
@brief Draws the game border.
*/
void draw_border();                     

/**
@brief Runs a countdown before the start of the game.
*/
void countdown();

/**
@brief Displays the opening title and animation for the game.
*/
void introScreen();

/** 
@brief Moves to selected part of the program from menu (Easy Start, Hard Start, Help, Scores).
*/
void proceed();

/** 
@brief Dettaches tickers from current speed and reattaches them with a new speed
@param speed - speed to re-attach tickers with (rate per second = 1/speed)
*/                 
void setTickers(int time);  

/** 
@brief Congratulate the player if they have got a new highscore
*/
void congratulate();    

/** 
@brief Creates and navigates the game menu
*/
void menu();

/**
@brief Starts the gameplay
*/
void playGame();

/**
@brief Displays player score and current highscore after game ends
*/
void resultsScreen();
