/**
@file intro.h
@brief Introduction 
@author James Bruce
*/

#ifndef INTRO_H
#define INTRO_H
#include "mbed.h"
#include "N5110.h"
#include <bitset>

/**
@namespace update_screen
@brief defining the tickers used that update the screen
*/
Ticker update_screen, update_screen2;
/**
defining the flag for updating screen1
*/
volatile bool g_update_screen = 0; 
/**
defining the flag for updating screen2
*/
volatile bool g_update_screen_2 = 0;
/**
defining a new, bigger buffer for the movable text 
*/
int introBuffer[168][96];

/**
function to initialise the mbed, on board switches and LEDs
*/
void init_K64F();

/**
function that prints strings to buffer in bit form; 
this avoids banks so that it can be moved easily pixel by pixel 
*/
void PrintStringToBuffer(const char * str, int x, int y);

/**
function to send the new buffer to the lcd
*/
void SendBufferToLCD();

/**
function to initialise the serial connections from mbed to PC
*/
void init_serial();

/**
function to flip the flag for updating the screen 
*/
void update_screen_isr();

/**
function to print the first section of the text 
*/
void printFirstText();

/**
function to animate the menu sliding on to the screen 
*/
void animateToMenu();

/**
function to flip the falg for updating the new screen 
uses a different rate than the other
*/
void update_screen2_isr();

/**
function that runs the correct sequence for the animated intro 
*/
void performAnimatedIntro();


//*****************************************************************************************************//

// END OF GLOBAL VARIABLES & FUNCTIONS //

//*****************************************************************************************************//



// FUNCTION TO PERFORM THE ORDER OF ANIMATION 
void performAnimatedIntro()
{
    update_screen.attach(&update_screen_isr, 0.05);
    update_screen2.attach(&update_screen2_isr, 0.05);

    printFirstText();
    animateToMenu();
}

// FUNCTION TO FLIP FLAG OF THE TICKER
void update_screen_isr()
{
    g_update_screen = 1;
}

// FUNCTION TO FLIP FLAG OF THE TICKER
void update_screen2_isr()
{
    g_update_screen_2 = 1;
}

// FUNCTION TO SET SERIAL BAUD RATE FOR DEBUGGING
void init_serial()
{
    // set to highest baud - ensure terminal software matches
    pc.baud(115200);
}

// FUNCTION THAT PRINTS STRING TO BUFFER
void PrintStringToBuffer(const char * str, int x, int y)
{
    // initialise the number of chars inside a string
    int char_no = 0;

    /// while the string is incrementing to the end char
    while (*str) {
        // sets the font position relative to the ASCII table
        int font_pos = (*str - 32)*5;
        /// loops through the number of columns in a char 
        for (int i = 0; i < 5; i++) {
            int x_pixel = x + i + (char_no*6); // set to correct value for char length of 6 pixels

            /// converts to the string to bitset form (8 bits per char)
            std::bitset<8> bit_representation (font5x7[font_pos + i]);
            /// loops through the bits and sets them for each char passed in the string
            for (int bit = 0; bit < 8; bit++) {
                if (bit_representation.test(bit)) { 
                    introBuffer[x_pixel][y + bit] = 1;
                }
            }
        }

        // increase char no. & increase str pointer
        char_no++;
        str++;
    }
}

// FUNTION THAT SENDS BUFFER TO THE LCD SCREEN 
void SendBufferToLCD()
{
    /// loop through the bounds of the screen and set the relative pixels to the buffer
    for (int col = 0; col < 84; col++) {
        for (int row = 0; row < 48; row++) {
            if (introBuffer[col][row]) {
                lcd.setPixel(col, row); // set pixel to buffer
            }
        }
    }
}

// FUNCTION THAT PRINTS THE FIRST ANIMATED TEXT 
void printFirstText()
{
    /// preset the text to animate upwards 
    char text[] = "Sensor";
    char text1[] = "Project By";
    char text2[] = "James Bruce";
    /// define their starting Y position 
    int y_pos = 50; //Starting y position
    int y_pos1 = 58; //Starting y position
    int y_pos2 = 66; //Starting y position

    /// decrease the Y bit of the text until it reaches 5 
    while (y_pos > 5) { // while loop 
        if (g_update_screen) {
            g_update_screen = 0;

            // forces clear of the screen and buffer
            for (int col = 0; col < 168; col++) {
                for (int row = 0; row < 96; row++) {
                    introBuffer[col][row] = 0;
                }
            }
            
            lcd.clear();
            // print the string at the new Y position 
            PrintStringToBuffer(text, 3, y_pos);
            PrintStringToBuffer(text1, 3, y_pos1);
            PrintStringToBuffer(text2, 3, y_pos2);
            SendBufferToLCD(); // send to the LCD

            lcd.refresh();

            /// decrease the positions of Y by 1 bit each time
            y_pos--; 
            y_pos1--;
            y_pos2--;
        }
    }
    wait(1); // short delay so the user can read the text before next animation 
}

// FUNCTION THAT PRINTS THE ANIMATED INTRO OF THE MENU 
void animateToMenu()
{
    /// preset the menu text 
    char text[] = "MENU";
    char text1[] = "SETTINGS";
    char text2[] = "GRAPH PLOT";
    char text3[] = "NUMERICAL";
    char text4[] = "RADAR MODE";
    /// re-define the previous text 
    char oldText[] = "Sensor";
    char oldText1[] = "Project By";
    char oldText2[] = "James Bruce";
    /// define the X positions for the menu text 
    int x_pos = 83+30; //Starting x position
    int x_pos1 = 83+17; //Starting x position
    int x_pos2 = 83+12; //Starting x position
    int x_pos3 = 83+16; //Starting x position
    int x_pos4 = 83+12; //Starting x position
    int x = 3; // old x position 

    /// decreases the X position of the menu and the Y position of the previous text 
    while (x_pos1 > 17) { //Stops when x = 0
        if (g_update_screen_2) {
            g_update_screen_2 = 0;

            // forces clear of the screen and buffer
            for (int col = 0; col < 168; col++) {
                for (int row = 0; row < 96; row++) {
                    introBuffer[col][row] = 0;
                }
            }
            
            lcd.clear();
            // prints the previous text back to its position 
            PrintStringToBuffer(oldText, 3, x+2);
            PrintStringToBuffer(oldText1, 3, x+10);
            PrintStringToBuffer(oldText2, 3, x+18);
            // prints the new text to the buffer
            PrintStringToBuffer(text, x_pos, 0);
            PrintStringToBuffer(text1, x_pos1, 16);
            PrintStringToBuffer(text2, x_pos2, 24);
            PrintStringToBuffer(text3, x_pos3, 32);
            PrintStringToBuffer(text4, x_pos4, 40);
            SendBufferToLCD();

            lcd.refresh();
            /// decreases all the X positions of the menu and increases the X of the old texts
            x_pos--; //Decrements the y coordinate
            x_pos1--;
            x_pos2--;
            x_pos3--;
            x_pos4--;
            x++;
            // dont allow the old text to wrap round the top and distorting the screen 
            if (x > 60) {
                x = 60;   
            }
        }
    }
}

//*****************************************************************************************************//

// END OF FUNCTION IMPLEMENTATIONS //

//*****************************************************************************************************//


#endif





