/**
@file UserSettings.h
@brief Header file containing member functions and variables
@author James Erringham-Bruce
*/

#ifndef USERSETTINGS_H
#define USERSETTINGS_H

#include "mbed.h"           // mbed library
#include "DataController.h" // handling the common data
#include <string>           // string library 

// creating the class used in plotting the graph
class UserSettings
{
    // defining the public functions
public:

    /**
    function is used to print the settings on the screen in order for the user to interact with and chose the option
    within the function, it controls the scrolling for the screen
    */
    void printScreen();
    /**
    function allows the user to scroll up and down on the screen to view all the options
    @param bank
    */
    void scrollUp_Down(int bank);
    /**
    function to return the value of the bank to invert
    @returns bank - to invert colour
    */
    int setBank(); 


    // defining the private functions
private:

};

//*****************************************************************************************************//

// END OF CLASS //

//*****************************************************************************************************//

#endif

/**
@struct settingsStrings
struct that holds the array of chars to state whether the LED & Buzzer are ON or OFF
*/
struct settingsStrings {
    char LED_State[5]; /*!<  3 chars to hold either "on" or "off", 5 used to give space for printing text */
    char buzzer_State[5]; /*!<  3 chars to hold either "on" or "off" 5 used to give space for printing text */
};

settingsStrings setAndSaveSettings();

char buffer2[14]; /*!<character length #2 as two strings are printed at once*/

char OFF[] = "OFF"; /*!< copy this string when off... */
char ON[] = "ON"; /*!< copy this string when on... */

volatile bool LED_Toggle = 0; /*!< determines whether LED is logic high(1) or logic low (0)*/
volatile bool buzzer_Toggle = 0; /*!< determines whether buzzer is logic high (1) or logic low (0)*/
volatile bool brightness_Toggle = 0; /*!< determines whether the brightness is being changed */

/**
function to allow the text that is off the screen to be printed when the user scrolls down the screen 
*/
void printText(settingsStrings state);

/**
function that allows the main menu to be returned if the interrupt button has been pressed
*/
void goBackToMenu();

/**
function to allow the brightness of the LCD screen to be changed using PWM
@param br
*/
void changeBrightness(float br);
/**
@namespace settingsStrings
@brief allows passage of the struct to be used in other functions
*/
settingsStrings setString;

int brightness = 50; /*!< intialises the brightness at start up */

//*****************************************************************************************************//

// END OF GLOBAL VARIABLES //

//*****************************************************************************************************//

// SAVE THE STRINGS TO THE SETTINGS SCREEN AND REWRITE THE CHANGED DATA TO THE SYSTEM...
settingsStrings setAndSaveSettings()
{
    /// checks position of the potentiometer 
    /// if the settings flag has been activated
    /// flip the toggle of the 
    if (selectorPot < 0.15f) { // for the LED toggle states
        if (selector_flag) {
            LED_Toggle = !LED_Toggle;
            if (LED_Toggle) {
                strcpy(setString.LED_State,ON);
                LED = 1;
            } else {
                strcpy(setString.LED_State,OFF);
                LED = 0;
            }
            selector_flag = 0; // put the flag back to zero ready for another use
        }
    }
    if (selectorPot >= 0.3f && selectorPot < 0.45f) { // for the Buzzer toggle states
        if (selector_flag) {
            buzzer_Toggle = !buzzer_Toggle;
            if (buzzer_Toggle) {
                strcpy(setString.buzzer_State,ON);
            } else {
                strcpy(setString.buzzer_State,OFF);
            }
            selector_flag = 0; // put the flag back to zero ready for another use
        }
    }
    if (selectorPot >= 0.7f ) { // hold while brightness is changed
        if (selector_flag) {
            do {
                brightness = selectorPot*100; // setting brightness as a percentage 
                changeBrightness(selectorPot*2);
                int length = sprintf(buffer,"%d %% ",brightness); // defining max length of chars 84 ÷ 6 = 14
                if ( length <= 14 ) { // if string will fit on display
                    lcd.printString(buffer, 35, 5);
                }
                if (back_flag) {
                    selector_flag = 0;
                    isInMenu = 0;
                    isInSettings = 1;
                } // if back, return to settings not the main menu!!
                lcd.printString("Adjust Your",9,2);
                lcd.printString("Brightness",13,3);
            } while (selector_flag);
        }
        sprintf(buffer,"%d %% ",brightness); //ensure brightness is still printed after loop
        lcd.printString(buffer, 35, 5);
    }
    return setString;
}

// FUNCTION TO PRINT THE SETTINGS SCREEN ACCORDING TO THE CHANGED PARAMETERS
void UserSettings::printScreen()
{
    /// initially set the buzzer & LED to off
    isInSettings = 1;
    unsigned int y_scroller = 0;

    settingsStrings newStrings; // call the struct
    selector_flag = 0;
    // COPY THE CHARS OFF TO THE EMPTY STATES
    strcpy(setString.LED_State,OFF); // set initial state as off
    strcpy(setString.buzzer_State,OFF); // set initial state as off
    
    while (isInSettings) {
        
        goBackToMenu();
        
        if (settings_timer_flag) {
            settings_timer_flag = 0;

            lcd.clear();
            unsigned int bank = setBank();
            // normalising the scroller icons pixel values
            /// set the scrolling ican and move down when the screen scrolls
            if ( selectorPot > 0.45f ) {
                y_scroller = ((selectorPot*25)*2) - 10; // 40 is the amount of free pixels the scrolling icon can move
            }
            // doubled the normalised value to compensate for the loss of pot range due to brightness..
            // ..taking up most of the value
            if (y_scroller > 48) {// ensuring the scroller does not exceed out of bounds
                y_scroller  = 48;
            }
            // print new strings obtained
            printText(newStrings);
            lcd.drawLine(82,y_scroller,82,y_scroller+8,1);// scrolling icon
            lcd.drawLine(83,y_scroller,83,y_scroller+8,1);// scrolling icon

            lcd.refresh();
        }
        sleep(); // sleep after ticker has run
    }
}

// FUNCTION TO RETURN THE BANK VALUE OF THE CURSOR
int UserSettings::setBank()
{
    int currentBank = 3;

    if ( selectorPot < 0.15f ) {
        currentBank = 3;
    }
    if ( selectorPot >= 0.15f && selectorPot < 0.3f ) {
        currentBank = 4;
    }
    if ( selectorPot >= 0.3f && selectorPot < 0.45f ) {
        currentBank = 5;
    }
    if ( selectorPot >= 0.45f && selectorPot < 0.7f ) {
        currentBank = 6; // exceeds screens length, scrolling algorithm must be implemented
    }
    if ( selectorPot >= 0.7f ) {
        currentBank = 7; // exceeds screens length, scrolling algorithm must be implemented
    }

    int cursorBank =  currentBank; // creating a stopping point for the cursor
    if ( currentBank > 5 ) {
        cursorBank = 5;
    }
    lcd.printChar('<',70, cursorBank);
    return currentBank;
}

// FUNCTION TO OVERRIDE, WHEN THE BANK IS > 5 PUSH TEXT BACK TO A BANK ON SCREEN
void printText(settingsStrings state)
{
    /// pass in the new values of the buzzer and LED state from the struct 
    state = setAndSaveSettings(); // assign new values

    // Print text as an array so that it can be manipulated to act as if the
    // text is scrolliing bank by bank
    /// print the screen as an array that only displays parts of that array
    const char screenText[8][14] = {
        {"   SETTINGS"},
        {             }, // empty slots
        {"LED:"       },
        {             }, // empty slots
        {"BUZZER:"    },
        {             }, // empty slots
        {"BRIGHTNESS:"},
        {             }, // empty slots
    };
    int length, length2;
    unsigned int textRow = 0;
    /// print the states as a string even when scrolling is implemented
    if ( selectorPot < 0.45f ) {
        textRow = 5;
        length = sprintf(buffer,"%s",state.LED_State); // defining max length of chars 84 ÷ 6 = 14
        if ( length <= 14 ) { // if string will fit on display
            lcd.printString(buffer, 20, 3);
        }
        length2 = sprintf(buffer2,"%s",state.buzzer_State); // defining max length of chars 84 ÷ 6 = 14
        if ( length2 <= 14 ) { // if string will fit on display
            lcd.printString(buffer2, 20, 5);
        }

    } else if ( selectorPot >= 0.45f && selectorPot < 0.7f ) {
        textRow = 6;
        length = sprintf(buffer,"%s",state.LED_State); // defining max length of chars 84 ÷ 6 = 14
        if ( length <= 14 ) { // if string will fit on display
            lcd.printString(buffer, 20, 2);
        }
        length2 = sprintf(buffer2,"%s",state.buzzer_State); // defining max length of chars 84 ÷ 6 = 14
        if ( length2 <= 14 ) { // if string will fit on display
            lcd.printString(buffer2, 20, 4);
        }

    } else if ( selectorPot >= 0.7f) {
        textRow = 7;
        length = sprintf(buffer,"%s",state.LED_State); // defining max length of chars 84 ÷ 6 = 14
        if ( length <= 14 ) { // if string will fit on display
            lcd.printString(buffer, 20, 1);
        }
        length2 = sprintf(buffer2,"%s",state.buzzer_State); // defining max length of chars 84 ÷ 6 = 14
        if ( length2 <= 14 ) { // if string will fit on display
            lcd.printString(buffer2, 20, 3);
        }
    }
    lcd.printString(screenText[textRow - 5],2,0);
    lcd.printString(screenText[textRow - 4],2,1);
    lcd.printString(screenText[textRow - 3],2,2);
    lcd.printString(screenText[textRow - 2],2,3);
    lcd.printString(screenText[textRow - 1],2,4);
    lcd.printString(screenText[textRow],2,5);
}

// FUNCTION TO CHANGE THE BRIGHTNESS OF THE LCD BACKLIGHT
void changeBrightness(float br)
{
    /// ensure the brightness value never goes out of range 0-1
    if(br>=1) {
        br = 1;
    } else if(br <= 0) {
        br = 0;
    }
    /// write the value to the PWM pin defined as 'backLight'
    backLight.write(br);
}


//*****************************************************************************************************//

// END OF FUNCTION IMPLEMENTATIONS //

//*****************************************************************************************************//





