/**
@file   DataController.h
@brief  Header file containing common data.
@author James Erringham-Bruce
@date   March 2015
*/

#ifndef DATACONTROLLER_H
#define DATACONTROLLER_H 

#include "mbed.h"
#include "N5110.h"
#include "SRF02.h"

#define nx 84
#define ny 48


/**
@namespace LCD 5110
@brief outputs displaying the pin connections for the screen to mbed 
*/
N5110 lcd (PTE26 , PTA0 , PTC4 , PTD0 , PTD2 , PTD1 , PTC5);
/**
@namespace PC
@brief serial input and output lines for printing data
*/
Serial pc(USBTX,USBRX);
/**
@namespace DistanceSensor
@brief I2CC pins to SRF02 Sensor
*/
SRF02 ds(PTE25, PTE24);
/**
@namespace r_led
@brief digital output for the on board red LED
*/
DigitalOut r_led(LED_RED);

/**
@namespace g_led
@brief digital output for the on board green LED
*/
DigitalOut g_led(LED_GREEN);

/**
@namespace b_led
@brief digital output for the on board blue LED
*/
DigitalOut b_led(LED_BLUE);

/**
@namespace LED
@brief digital output for warning LED
*/
DigitalOut LED(PTC2);

/**
@namespace backlight
@brief defining the LED for 5110 LCD screen
*/
PwmOut backLight(PTC3);
/**
@namespace Buzzer
@brief definig the pin for the buzzer
*/
PwmOut buzzerAlert(PTA2);
/**
@namespace backButton
@brief defining the button to go back as an action 
*/
InterruptIn backButton(PTB19);
/**
@namespace interruptSelector
@brief defining the pin for buttons
*/
InterruptIn interruptSelector(PTB18);
/**
@namespace Potentiometer
@brief defining the pins for the potentiometer
*/
AnalogIn selectorPot(PTB2);
/**
@namespace sw2
@brief defining the on board switch 2
*/
InterruptIn sw2(SW2);
/**
@namespace sw3
@brief defining the on board sw3
*/
InterruptIn sw3(SW3);
/**
@namespace Ticker
@brief defining the tickers used
*/
Ticker ticker;

/**
@namespace Menu
@brief defining the tickers used
*/
Ticker menuTicker;

/**
@namespace averageTicker
@brief defining the tickers used
*/
Ticker ticker2;

/**
@namespace settingsTicker
@brief defining the tickers used
*/
Ticker settingsTicker;

/**
function to display an error warning message on screen to the user as a visual display when they are out of bounds
*/
void readingErrors();
/**
function to initialise the mbed with the on board LED's and switches
*/
void init_K64F();
/**
function that displays the menu to the user for the user to interact with and select the various options displayed on screen
@param int average - average reading of distance
*/
void mainMenu();
/**
function to return back to the main menu at the trigger of an interrupt
*/
void goBackToMenu();
/**
function to manage the flag of the ticker
*/
void timer_isr();
/**
function to manage the flag of the ticker
*/
void timer_isr2();
/**
function to enable the user to go back to previous screens once they are within different settings screens
*/
void backButton_isr();
/**
function to enable the user to go back to previous screens once they are within different settings screens
*/
void forwardButton_isr();
/**
function to manage flag of the ticker
*/
void selector_isr();
/**
function to manage the flag of the ticker
*/
void settings_timer_isr();
/**
function to manage the flag of the ticker
*/
void menu_isr();
/**
function for initialising the serial port for the pc
*/
void init_serial(); 
/**
function to initialise the serial connection, LCD screen, tickers and the mbed with the on board switches & LEDs
*/
void init_all();



int getAverageReadingInch(); /*!< function that returns int 'getAverageReadingInch' */

char buffer[14]; /*!< buffer length for character length to print text on screen*/

volatile int g_timer_flag = 0; /*!< to ensure the compiler does not stick an object with permanent logic 1 so we can change the logic
of this object to high or low when needed by the user */

volatile int g_timer_flag2 = 0;/*!< to ensure the compiler does not stick an object with permanent logic 1 so we can change the logic
of this object to high or low when needed by the user */

volatile int settings_timer_flag = 0; /*!< to ensure the compiler does not stick an object with permanent logic 1 so we can change the logic
of this object to high or low when needed by the user */

volatile int menu_flag = 0; /*!< to ensure the compiler does not stick an object with permanent logic 1 so we can change the logic
of this object to high or low when needed by the user */

volatile int selector_flag = 0; /*!< to ensure the compiler does not stick an object with permanent logic 1 so we can change the logic
of this object to high or low when needed by the user */

volatile int back_flag = 0; /*!< to ensure the compiler does not stick an object with permanent logic 1 so we can change the logic
of this object to high or low when needed by the user */

volatile bool isInMenu = 0; /*!< to ensure the compiler does not stick an object with permanent logic 1 so we can change the logic
of this object to high or low when needed by the user */

volatile bool isInNumerical = 0; /*!< to ensure the compiler does not stick an object with permanent logic 1 so we can change the logic
of this object to high or low when needed by the user */

volatile bool isInGraph = 0; /*!< to ensure the compiler does not stick an object with permanent logic 1 so we can change the logic
of this object to high or low when needed by the user */

volatile bool isInRadar = 0; /*!< to ensure the compiler does not stick an object with permanent logic 1 so we can change the logic
of this object to high or low when needed by the user */

volatile bool isInSettings = 0; /*!< to ensure the compiler does not stick an object with permanent logic 1 so we can change the logic
of this object to high or low when needed by the user */

//*****************************************************************************************************//

// END OF GLOBAL VARIABLES & FUNCTIONS //

//*****************************************************************************************************//


// FUNCTION TO RETURN AVERAGE VALUE IN CM
int getAverageReadingCm()
{
    int sum = 0;
    /// algorithm to implement ouput of average reading after 10 readings
    for( int j = 0; j < 10; j++) {
        int i = ds.getDistanceCm(); /// function that keeps saving sensor values as int i
        sum += i; // increase the sum 10 times
    }
    float avg = sum / 10; // sum ÷ number of samples  = average formula
    return avg; // return the average value
}

// FUNCTION TO RETURN AVERAGE VALUE IN INCH
int getAverageReadingInch()
{
    int sum = 0;
    /// algorithm to implement ouput of average reading after 10 readings
    for( int j = 0; j < 10; j++) {
        int i = ds.getDistanceInch(); /// function that keeps saving sensor values as int i
        sum += i; // increase the sum 10 times
    }
    float avg = sum / 10; // sum ÷ number of samples  = average formula
    return avg; // return the average value
}

// FUNCTION TO RETURN BACK TO THE MAIN MENU ON PRESS
void backButton_isr()
{
    back_flag = 1; /// flip the flag to go back 
    /// then reset all screen flags that control the while loops
    isInSettings = 0;
    isInRadar = 0;
    isInGraph = 0;
    isInNumerical = 0;
    isInMenu = 1;
}

// FUNCTION TO FLIP THE FLAG OF THE SETTINGS TICKER
void selector_isr()
{
    selector_flag = 1; /// flip the flag of the settings ticker 
}

// FUNCTION TO GO BACK TO MAIN MENU 
void goBackToMenu() 
{
   /// if the back flag is activated then reset the flag and go back to menu 
   if (back_flag) {
       back_flag = 0;
       
    }    
}

//*****************************************************************************************************//

// END OF FUNCTION IMPLEMENTATIONS //

//*****************************************************************************************************//







#endif 