/**
@file Graph.h
@brief Header file containing member functions and variables
@author James Erringham-Bruce
*/

#ifndef GRAPH_H
#define GRAPH_H
#include "mbed.h"           // mbed library
#include "DataController.h"
#include <deque>            // std::deque


// creating the class used in plotting the graph
class Graph
{
    // defining the public functions and variables
public:
    /**
    function is used to plot a graph on the LCD screen as a visual display to the user so the user can compare previous data 
    if they so desire in order to come to a conclusion about the distances measured or an average
    */
    void plotGraph();
    /**
    function is used to draw the axis on the graph in order for the data that is being sent from the sensor can be plotted on the 
    graph accurately to allow the user further experiment if they wish
    */
    void drawGraphAxis();
    /**
    function allows the axis label for the y axis to be labelled horizontally in order to increase presentation towards the user 
    */
    void drawBitmap();
    /**
    function which uses the points that have been plotted on the graph and draws a line to improve the visual display for the user 
    */
    void drawLines();
    /**
    function to allow the points that have been plotted to shift along the x axis in the negative direction to allow the new points 
    to be plotted and allow the user to see the updated measurements taken from the sensor as the user moves away or closer to the 
    set distance 
    */
    void drawLinesBackwards();


    // defining the private functions and variables
private:

};

//*****************************************************************************************************//

// END OF CLASS //

//*****************************************************************************************************//

// members initialies outside of the class//

unsigned int i = 0; /*!< used global variable for iterations */

unsigned int x = 20; /*!< variable used for the x axis on the graph allowing us to plot points for visual display to the user*/

int points[7] = {10 ,20 , 30 ,40 ,50 ,60, 70}; /*!< array of 'x' points to plot on the graph to use for visual display to the user */

// 
const bool verticalWord_bitmap[46][7] = {
    { 0,0,0,1,1,0,0 },
    { 0,0,1,0,1,0,1 },
    { 0,0,1,0,1,0,1 },
    { 0,0,1,0,1,0,1 },
    { 0,0,0,1,1,1,0 },
    { 0,0,0,0,0,0,0 },
    { 0,0,0,0,0,1,0 },
    { 0,0,1,0,0,0,1 },
    { 0,0,1,0,0,0,1 },
    { 0,0,1,0,0,0,1 },
    { 0,0,0,1,1,1,0 },
    { 0,0,0,0,0,0,0 },
    { 0,0,0,1,1,1,1 },
    { 0,0,1,0,0,0,0 },
    { 0,0,1,0,0,0,0 },
    { 0,0,0,1,0,0,0 },
    { 0,0,1,1,1,1,1 },
    { 0,0,0,0,0,0,0 },
    { 0,0,0,1,1,1,1 },
    { 0,0,1,0,1,0,1 },
    { 0,0,1,0,1,0,1 },
    { 0,0,1,0,1,0,1 },
    { 0,0,0,0,0,1,0 },
    { 0,0,0,0,0,0,0 },
    { 0,0,0,0,0,1,0 },
    { 0,0,0,0,0,0,1 },
    { 0,0,1,0,0,0,1 },
    { 1,1,1,1,1,1,0 },
    { 0,0,1,0,0,0,0 },
    { 0,0,0,0,0,0,0 },
    { 0,0,1,0,0,1,0 },
    { 0,0,1,0,1,0,1 },
    { 0,0,1,0,1,0,1 },
    { 0,0,1,0,1,0,1 },
    { 0,0,0,1,0,0,1 },
    { 0,0,0,0,0,0,0 },
    { 0,0,0,0,0,0,1 },
    { 1,0,1,1,1,1,1 },
    { 0,0,1,0,0,0,1 },
    { 0,0,0,0,0,0,0 },
    { 1,1,1,1,1,1,1 },
    { 0,0,0,1,0,0,1 },
    { 0,0,1,0,0,0,1 },
    { 0,0,1,0,0,0,1 },
    { 0,0,0,1,1,1,0 },
    { 0,0,0,0,0,0,0 }
};  /*!< bitmap of the vertical word 'distance' displayed on the graph to increase aesthetic performance towards the user*/

#endif

/** @namespace DISTANCE_READINGS_DEQUE
@brief double ended queue for storing graph points 
*/
std::deque<int> DISTANCE_READINGS_DEQUE (6); 

//*****************************************************************************************************//

// END OF GLOBAL VARIABLES & FUNCTIONS //

//*****************************************************************************************************//

// FUNCTION TO PLOT GRAPH OF DISTANCE VS. TIME
void Graph::plotGraph()
{
    isInGraph = 1;
    /// initiate the deque to all 0 values to avoid crashing when plotting a point that doesnt exist 
    DISTANCE_READINGS_DEQUE.assign(6,0);
    lcd.clear();
    while (isInGraph) {
        goBackToMenu();
        /// draw the graph axis on the screen 
        drawGraphAxis(); // draw the axis before the timer fires ( drawing the axis before the 5 second delay )
        // normailise the distance range between amount of free pixels available
        int average = 40 - getAverageReadingCm()/6.25;
        // start the timer flag
        if (g_timer_flag) {
            g_timer_flag = 0; // reset the flag
            // redrawing the graph axis ( crucial because the screen is cleared )
            lcd.clear();
            drawGraphAxis();
            /// pushing in a new average value to the stack ( 6 elements in total, until if statement says otherwise [i] )
            DISTANCE_READINGS_DEQUE.push_front(average);
            // increasing i by 1 each timer fired
            if(i<6) {
                // drawing lines between the points from left hand side
                drawLines();
                i++;
            } else {
                drawLinesBackwards();
            }
            // increase the x axis each timer fired( moving the plotted point along the graph )
            x += 10;
        }
        sleep(); // sleep before next interrupt
    }
}

// FUNCTION TO DRAW LINES BETWEEN POINTS ON THE GRAPH
void Graph::drawLines()
{
    /// draws the lines from point to point in the forward direction
    /// forward direction because the graph is filling up from the start
    if (i != 0) {
        for (int x = 0; x < i; x++) {
            lcd.drawLine(points[x],DISTANCE_READINGS_DEQUE.at(i-x),points[x+1],DISTANCE_READINGS_DEQUE.at(i-x-1),1);
            lcd.refresh();      
        } 
    }
}

// FUNCTION TO DRAW LINES BETWEEN POINTS ON THE GRAPH
void Graph::drawLinesBackwards()
{
    /// draws the lines from point to point in the backward direction 
    /// backwards because when it is filled, the graph needs to be shifted backwards every new point made
    for(int x = 0; x < 6 ; x++) {
        lcd.drawLine(points[6-x],DISTANCE_READINGS_DEQUE.at(x),points[6-x-1],DISTANCE_READINGS_DEQUE.at(x+1),1);
        lcd.refresh();
    }
}

// DRAWING THE GRAPH AXIS AND LABELS
void Graph::drawGraphAxis()
{
    /// draw the Y axis and the X axis
    /// print the names of the axis
    lcd.drawLine(9,6,9,39,1);        // y-axis line
    lcd.drawLine(9,39,80,39,1);      // x-axis line
    drawBitmap();                    // drawing the bitmap of the vertical word
    lcd.printString("time(s)",20,5); // printing the x axis
}

// FUNCTION TO DRAW THE BITMAP OF THE VERTICAL WORD
void Graph::drawBitmap()
{
    /// draws the bitmap of the vertical word 'distance'
    /// used a bitmap because horizontal text is not possible 
    for (int column = 0; column < 7; column++) { // loops through the columns
        for (int row = 0; row < 46; row++) { // loops through the rows
            if (verticalWord_bitmap[row][column]) { // if a bit is found true ( active 1 )
                lcd.setPixel(column, 2 + row); // set the pixel at that location on the screen
            }
        }
    }
}

//*****************************************************************************************************//

// END OF FUNCTION IMPLEMENTATIONS //

//*****************************************************************************************************//





