//Works at speed 2.0, KPS 2.0e-2, KD 1.0e-4

//Has tolerance check, Steering doesn't
//Values from Simulation
//#define KPS 0.0896852742245504f
//#define KD 0.072870569295611f
//#define KPS 0.03f

#include "mbed.h"
#include <iostream>


#define TI 0.001f

#define SCALAR 0.53f
#define MINM 0.0f
#define MAXM 0.53f
#define KPM 0.15f //0.1414f
#define KI 19.7408f

#define KPS 2.0E-2f //Original 2.0e-2
#define KD 1.0e-4f
#define SET 0.0f
#define MINS 0.05f
#define MAXS 0.1f
#define BIAS 0.075f
#define TOL 0.02f
#define STEER_FREQ 0.02f //50 Hz
#define STEERUPDATEPERIOD 0.05

/***********************************|Pin Declarations|*************************************************************/

//Feedback
AnalogIn speed(PTB0);   //tachometer
AnalogIn _right(PTB1);  //Right sensor
AnalogIn _left(PTB2);   //Left sensor

//Output
PwmOut servoSig(PTA13); //PWM output to control servo position
PwmOut gateDrive(PTA4); //PWM output to control motor speed
DigitalOut brake(PTD0);
//Communication
Serial bt(PTE22, PTE23); //Serial Pins (Tx, Rx)
//LEDs
DigitalOut blue(LED_BLUE);
DigitalOut red(LED_RED);
DigitalOut green(LED_GREEN);
//Checkpoint Interrupts
InterruptIn navRt(PTD2);
InterruptIn navLft(PTD3);
//Button Interrupts
InterruptIn stopButton(PTD4);
InterruptIn goButton(PTA12);
//Checkpoint LEDS
BusOut checkLED(PTB8,PTB9,PTB10,PTB11);



/***********************************|Variable Declarations|*************************************************************/

Ticker control;
Timer ctrlTimer;
Timer checkpointTimer;

bool lTrig = false;
bool rTrig = false;

volatile int rightCount;
volatile int leftCount;

float data[6];

float Setpoint = 0.0;
float spHolder;
float errSum = 0.0;

float fbPrev = 0.0f;
float Kps = 2.0E-2; //0.013 for setpoint = 0.0/0.05
float Kd = 1.0e-4;

float Kpm = 0.141;
int checkpoint = 0;

char state;


void waitState()
{
    Setpoint = 0; //Makes sure the car isn't moving
    state = 'w';
    //bt.putc(state);
    if (_left.read() == 0.0 &&  _right.read() == 0.0)
    {
        green = blue = 1;
        red = 0;   
    } else {
    brake.write(0);    
    green = red = 1; 
    blue = 0;        //blue LED indicating car is in wait state 
    }
} 
void offTrack()
{
    Setpoint = 0;
    brake.write(1);
    state = 'o';
    //bt.putc(state);
    green = blue = 1;
    red = 0;
    waitState();
}




void stopState()
{
    state = 's';
    //bt.putc('s');
    spHolder = Setpoint;
    Setpoint = 0;
    brake.write(1);
    waitState();
}
void letsGo()
{
        if (_left.read() == 0.0 &&  _right.read() == 0.0)
        {
            stopState();
                
        }
        state = 'g';
        //bt.putc(state);
        red = blue = 1;
        green = 0;
        brake.write(0);
        Setpoint = spHolder;
        
        
}
inline void applyBrake()
{  
    spHolder = Setpoint;
    brake.write(1);
    Setpoint = 0.0;
}

inline void releaseBrake()
{
    brake.write(0);
    Setpoint = spHolder;
}

void display()
{
    bt.printf("Setpoint = %f, Kps = %f, Kd = %f, Kpm = %f, Brake = %f\r\n", Setpoint, Kps, Kd, Kpm, brake.read());
}

void displayCheckpoint()
{
    bt.printf("Checkpoint reached: %i\r\n", checkpoint);   
} 


void serCb()
{
    char x = bt.getc();
    if (x == 'u')
    {
        Setpoint += 0.025;
        display();
    }
    else if(x == 'h')
    {
        Setpoint -= 0.025;
        display();
    }
    else if (x == 'i')
    {
        Kps += 1.0e-3;
        display();
    }
    else if (x == 'j')
    {
        Kps -= 1.0e-3;
        display();
    }
    else if (x == 'o')
    {
        Kd += 1.0e-5;
        display();
    }
    else if (x == 'k')
    {
        Kd -= 1.0e-5;
        display();
    }
    else if (x == 'b')
    {
        applyBrake();
        display();
       
    }
    else if (x == 'n')
    {
        releaseBrake();
        display();
    }
    else if (x == 'p')
    {
        display();
    }
    else if (x == 'y')
    {
        Kpm += 0.003;
        display();   
    }   
    else if(x == 'g')
    {
        Kpm -= 0.003; 
        display();   
    } 
    else if (x == 'z')
    {
        bt.printf("Right = %i Left = %i\r\n", rightCount, leftCount);
    }
    else if(x== 's')
    {
        stopState();
    }
    else if(x == 'a')
    {
        letsGo();   
    }    
    else
    {
        bt.printf("Invalid input");
    }
    if(Setpoint > MAXM) Setpoint = MAXM;
    if(Setpoint < MINM) Setpoint = MINM;    
}

void incrimentCheckpoint()
{
    if (checkpointTimer > 0.1)
    {
        checkpoint++;
        checkpointTimer.reset();
    }
    displayCheckpoint();
    
}    

void incL()
{      
    leftCount++;
    lTrig = true;
    incrimentCheckpoint();
    
}

void incR()
{
    rightCount++;
    rTrig = true;
    incrimentCheckpoint();
}

void steer()
{
    
    float L = _left.read();
    float R = _right.read();
    if (L == 0.0 && R == 0.0)
    {
        stopState();
    }           
    float fb = L - R; 
    float e = SET - fb;
    float Controlleroutput = Kps * e - (Kd * (fb - fbPrev)/TI)+ BIAS;//subtract derivative of error?? 
    if (Controlleroutput > MAXS) Controlleroutput = MAXS;
    else if (Controlleroutput < MINS) Controlleroutput = MINS;
    if (abs(Controlleroutput - servoSig.read()) > TOL || ctrlTimer.read() >= STEERUPDATEPERIOD) 
    {
        ctrlTimer.reset();
        servoSig.write(Controlleroutput);
    }
    fbPrev = fb;
    data[0] = _right.read();
    data[1] = _left.read();
    data[2] = Controlleroutput;
    data[3] = e;
    
}
void drive()
{
    float error = Setpoint-speed.read();
    errSum +=(error * TI);
    float iTerm = KI*errSum;
    if(iTerm > MAXM) iTerm = MAXM;
    if(iTerm < MINM) iTerm = MINM; 
    float output = KPM*error + iTerm;
    if(output > MAXM) output = MAXM;
    if(output < MINM) output = MINM;        

    gateDrive.write(output);
    data[4] = gateDrive.read();
    data[5] = speed.read(); 
}

void cb()
{
    steer();
    if (state == 'g'){
        drive();
    }
}

int main()
{   
    //startup checks
    bt.baud(115200);
    bt.attach(&serCb);
    servoSig.period(STEER_FREQ);
    gateDrive.period(.00005f);
    gateDrive.write(Setpoint);
    checkpointTimer.start();
   

    ctrlTimer.start();
    control.attach(&cb, TI);
    
    rightCount = 0;
    leftCount = 0; 
    
  
    navRt.fall(&incR);
    navLft.fall(&incL);
    
    goButton.fall(&letsGo);
    stopButton.fall(&stopState);
    waitState();
    while(1) {
        if (checkpoint > 10) checkpoint = 0;
        checkLED.write(checkpoint);
        if(lTrig){
            bt.putc('l');
            lTrig = false;
        }
        if(rTrig){
            bt.putc('r');
            rTrig = false;
        }
    

    }
}