#ifndef MBED_SPI
#define MBED_SPI
#include "mbed.h"
#include "L3GD20_Resister.h"
/** L3GD20(SPI) class
*
* This is the L3GD20 class.
* @code
#include "mbed.h"
#include "L3GD20.h"

Serial pc(USBTX,USBRX);//tx,rx
L3GD20 L3GD20(p11,p12,p13,p14,p15);//miso,mosi,sck,cs,interrupt

void interruption(anglerrates *val)
{
    printf("X:%+05d\tY:%+05d\tZ:%+05d level:%02d\r\n",(val->X),(val->Y),(val->Z),(L3GD20.FIFO.level));
}
int main()
{
    L3GD20.start(L3GD20::XYZ,&interruption);
    while(1) {
        sleep();
    }
}
* @endcode
*/
class L3GD20
{
public:
    L3GD20(PinName mosi, PinName miso, PinName scl, PinName cs,PinName interrupt2=NC);
    typedef enum {
        null=0x00,WhoAmI=0x0F,CtrlReg1=0x20,CtrlReg2=0x21,CtrlReg3=0x22,CtrlReg4=0x23,CtrlReg5=0x24,Reference=0x25,OutTemp=0x26,StatusReg=0x27,OutXL=0x28,OutXH=0x29,OutYL=0x2A,OutYH=0x2B,OutZL=0x2C,OutZH=0x2D,FIFOCtrlReg=0x2E,FIFOSrcReg=0x2F,INT1Cfg=0x30,INT1Src=0x31,INT1ThsXH=0x32,INT1ThsXL=0x33,INT1ThsYH=0x34,INT1ThsYL=0x35,INT1ThsZH=0x36,INT1ThsZL=0x37,INT1Duration=0x38,READ=0x80
    } RESISTER;
    /** @enum DIRECTION
    *   enable direction\n
    *   Example...L3GD20::XY,L3GD20::XYZ
    */
    typedef enum {
        Y=0x1,
        X=0x1<<1,
        Z=0x1<<2,
        XY=X|Y,
        XZ=X|Z,
        YZ=Y|Z,
        XYZ=X|Y|Z
    } DIRECTION;
    /** @enum FIFO_mode
    *   FIFO mode\n
    *   Example...L3GD20::FIFOmode\n
    *   See Datasheet 4.2_FIFO(p.16/44)
    */
    typedef enum {
        BYPASSmode=0x0,FIFOmode,STREAMmode,STREAMtoFIFOmode,BYPASStoSTREAMmode
    } FIFO_mode;
    /**
    *   FIFO status for cause of interruption\n
    *   Example...L3GD20::watermark\n
    *   See Datasheet 7.4_CTRL_REG3(p.33/44)
    */
    typedef enum {
        none=0,empty,watermark,overrun
    } FIFOstatus;
    /** 
    *   @brief  FIFO status info
    */
    struct {
        FIFOstatus status;///< Type of status is enum "FIFOstatus". cause of interruption(none,empty,watermark,overrun)
        int level;        ///< FIFO buffer level
    } FIFO;
    struct config {
//read and write resister
        union CTRL_REG1 CTRL_REG1;
        union CTRL_REG2 CTRL_REG2;
        union CTRL_REG3 CTRL_REG3;
        union CTRL_REG4 CTRL_REG4;
        union CTRL_REG5 CTRL_REG5;
        union REF_DATACAP REF_DATACAP;
        union OUT_TEMP OUT_TEMP;
        union STATUS_REG STATUS_REG;
        union FIFO_CTRL_REG FIFO_CTRL_REG;
        union INT1_CFG INT1_CFG;
        union INT1_TSH_XH INT1_TSH_XH;
        union INT1_TSH_XL INT1_TSH_XL;
        union INT1_TSH_YH INT1_TSH_YH;
        union INT1_TSH_YL INT1_TSH_YL;
        union INT1_TSH_ZH INT1_TSH_ZH;
        union INT1_TSH_ZL INT1_TSH_ZL;
        union INT1_DURATION INT1_DURATION;
    } _config;
    struct status {
//read only resister
        int OUT_TEMP;
        int STATUS_REG;
        union FIFO_SRC_REG FIFO_SRC_REG;
        int INT1_SRC;
    } _status;
//Class method
    /** @fn void L3GD20::start(DIRECTION enable);
    *   Start command send to module
    *   @param enable
    */
    void start(DIRECTION enable);
    /** @fn void L3GD20::start(DIRECTION enable,void (*func)(anglerrates*));
    *   Start with interrupt
    *   @param enable L3GD20 channel
    *   @param func user function(call by InterrtptIn)
    */
    void start(DIRECTION enable,void (*func)(anglerrates*));
    /** @fn void L3GD20::stop()
    *    stop sampling command send
    */
    void stop();
    /** @fn void L3GD20::sleep()
    *    sleep command send
    */
    void sleep();
    /** @fn void L3GD20::read(anglerrates* val,DIRECTION direction)
    *   read angler rates with direction(L3GD20::XY)\n
    *   This is read and calcurate dps, reading value is set to class instanse
    */
    void read(anglerrates* val,DIRECTION direction);
    /** @fn int L3GD20::readTemperature()
    *   read temperature without format
    *   (I don't know meaning of value.)
    */
    int readTemperature();
    /** @fn void L3GD20::enableFIFO(FIFO_mode mode,FIFOstatus interrupt,const int threshold)
    *   @brief   L3GD20mode need to reset(to bypass mode) when filled FIFObuffor
    *   @param mode        FIFOmode(L3GD20::BYPASSmode,FIFOmode,STREAMmode,STREAMtoFIFOmode,BYPASStoSTREAMmode)
    *   @param interrupt   cause of interrupt(L3GD20::none,empty,watermark,overrun)
    *   @param threshold   interruption threshold(1 to 30 vaild)
    */
    void enableFIFO(FIFO_mode mode,FIFOstatus interrupt,const int threshold=0);
    /** @fn int L3GD20::updateFIFO(void)
    *   @brief   Update FIFO status
    *   @return FIFO buffer level
    */
    int updateFIFO(void);
    void allReadOut();
    anglerrates value;///< @brief latest angler rates
protected:
    //write command to resister
    void write(RESISTER reg,int val);
    //read resister for resister
    void read(RESISTER reg,int* val);
    //just send reboot command
    void reboot();
    //reset FIFO To restart data collection, resister must be written back to Bypass mode.
    void resetFIFO();
    //read all configration rester
    void configReadOut(void);
    //read all status rester
    void statusReadOut(void);
    //for InterruptIn function ,call userFunction in this methed
    void interrupt(void);
    /*
    void datarate(uint8_t rate,uint8_t bandwidth);
    void setDataFormat();
    void filter(uint8_t mode,uint8_t frequency);
    void channelSource(uint8_t channnel,uint8_t dataSelection,uint8_t interruptSelection);
    void FIFO(uint8_t mode,uint8_t watermark);
    void interrupt(uint8_t source,uint8_t threthold,uint8_t duration,uint8_t Wait);
    */
    void (*userFunction)(anglerrates*);
    SPI _spi;
    DigitalOut _cs;
    InterruptIn _int2;
};

#endif