/**
 * @file    main.h
 * @brief   Main header file for includes and defs 
 * @author  John Wilkey
 * @author  Alec Guertin
 * @author  Chester Chu
 */
 
#ifndef _MAIN_H
#define _MAIN_H

#include "mbed.h"
#include "m3pi.h"
#include <string.h>
#include <stdarg.h>
#include <stdio.h>
#include <math.h>

/* Constants used in main.c */
#define TURN_SPEED              0.15    /**< Motor power for turning */
#define DRIVE_SPEED             0.25    /**< Motor power for drawing/moving */
#define TIME_FACT               1780    /**< Multiplier for forward() and backward() */
#define CAL_SPEED               0.25    /**< Drive speed during calibration */
#define CLOSE_ENOUGH            0.0008  /**< Threshold for calibration line centering */
#define WIGGLE_MAX              30      /**< Max 'wiggles' during calibration */
#define CORNER_THRESHOLD        0.3     /**< Threshold for checking if line position denotes a corner */
#define INST_BUF_SIZE           250     /**< Size of input buffer for instructions */
#define CORRECTION_SPEED        0.2*DRIVE_SPEED     /**< Amount to change speed of one wheel when following line */
#define CORRECTION_THRESHOLD    0.05    /**< Maximum tolerable deviation from line when following line */
#define RAD_TO_DEG              57.29   /**< Factor to convert radians to degrees */
#define FULL_TURN               360     /**< Degrees for a full turn */
#define HALF_TURN               180     /**< Degrees for a half turn */
#define QUARTER_TURN            90      /**< Degrees for a quarter (right angle) turn */
#define CAL_FACTOR              1.1     /**< Scaling factor for time to drive */
#define DEGREE_CORRECTION       0       /**< Amount (in degrees) added to the angle for each turn */
#define SEC_TO_MSEC             1000    /**< Scaling factor to convert seconds to milliseconds */
#define SEC_TO_USEC             1000000 /**< Scaling factor to convert seconds to microseconds */
#define MSEC_TO_USEC            1000    /**< Scaling factor to convert milliseconds to microseconds */

/**
 * @brief get values of next PostScript instruction.
 *
 * @param       buf     Buffer with PS instructions.
 * @param       x       Pointer to storage for x coordinate.
 * @param       y       Pointer to storage for y coordinate.
 * @param       draw    Pointer to storage for draw/move boolean.
 * 
 * @return              Success or failure code.
 */
int retrieve_inst(char *buf, int *x, int *y, int *draw);

/**
 * @brief Driver forward for a time.
 *
 * @param[in]   amt     Amount to drive forward. In milliseconds.
 */
void forward(int amt);

/**
 * @brief Drive backward for a time.
 *
 * @param[in]   amt     Amount to drive backward. In milliseconds.
 */
void backward(int amt);

/**
 * @brief Turn right by some angle.
 *
 * @param[in]   deg     Desired final turn angle in degrees from start.
 *                      Note that a negative angle will turn in the opposite
 *                      direction.
 */
void right(float deg);

/**
 * @brief Turn left by some angle.
 *
 * @param[in]   deg     Desired final turn angle in degrees from start.
 *                      Note that a negative angle will turn in the opposite
 *                      direction.
 */
void left (float deg);

/**
 * @brief   Wait for a number of seconds, possibly fractional.
 * 
 * Timer resolution is on the order of microseconds. A negative wait
 * time does nothing. 
 *
 * @param[in]   amt     Time to wait, in seconds.
 */
void timerWait(float amt);

/**
 * @brief       Follows the non-reflective line to a corner
 *
 * @param[in]   left    boolean specifying if the corner to look for
 *                      is a left corner
 *
 * @return      returns an error code
 */
int find_corner(int left);

/**
 * @brief   Turns towards the base line and moves to reach it
 */
void find_line();

/**
 * @brief   Computes the Euclidean distance between two points
 *
 * @param[in]   x1      the x-coordinate of the first point
 * @param[in]   y1      the y-coordinate of the first point
 * @param[in]   x2      the x-coordinate of the second point
 * @param[in]   y2      the y-coordinate of the second point
 *
 * @return      the distance between points (x1,y1) and (x2,y2)
 */
float distance(int x1, int y1, int x2, int y2);

/**
 * @brief   prints to the m3pi's screen
 *
 * @param[in]   line    the line on the screen to print to, in {0,1}
 * @param[in]   format  the format string to print
 * @param[in]   ...     the arguments to pass into the format string
 */
void robot_printf(int line, const char *format, ...);

/**
 * @brief   computes the angle to turn to face towards (x,y)
 *
 * @param[in]   last_x  the current x-coordinate
 * @param[in]   last_y  the current y-coordinate
 * @param[in]   x       the desired x-coordinate
 * @param[in]   y       the desired y-coordinate
 * @param[in]   angle   the current angle the m3pi is facing
 *
 * @return      the angle to turn left
 */
float compute_turn_angle(int last_x, int last_y, int x, int y, float angle);

/**
 * @brief       sends the signal on the output pin for the led
 *              to turn it on
 */
void pen_up();

/**
 * @brief       turns off the signal to the led
 */
void pen_down();

#endif