#include "mbed.h"
#include "uLCD_4DGL.h"
#include "rtos.h"
#include "ultrasonic.h"
#define PI 3.14159265359
#include "Motor.h"
#include "PID.h"
#include "HALLFX_ENCODER.h"

/************* Common Application Declarations *****************/
//Mutex pc_mutex;
Mutex serial_mutex;
Serial pc(USBTX, USBRX);
volatile int sonar_distance = 0;
volatile int last_sonar_distance = 0;
volatile bool connected = false;
volatile bool move = false;
/************************* uLCD ******************************/               
uLCD_4DGL uLCD(p9,p10,p11); // serial tx, serial rx, reset pin;

void uLCD_func()
{
    int offset = 4;
    int connected_row = 1;
    int sonar_row = connected_row + offset + 1; // connected + offset + 1
    serial_mutex.lock();
    uLCD.color(WHITE);
    
    uLCD.locate(1,sonar_row);
    uLCD.printf("Sonar Measurement");
    
    bool connected_written;
    if(connected)
    {
        uLCD.locate(1,connected_row);
        uLCD.printf("             ");
        uLCD.locate(1,connected_row);
        uLCD.color(BLUE);
        uLCD.printf("CONNECTED");
        connected_written = true;
    }
    else
    {
        uLCD.locate(1,connected_row);
        uLCD.printf("             ");
        uLCD.locate(1,connected_row);
        uLCD.color(RED);
        uLCD.printf("NOT CONNECTED");
        connected_written = false;
    }
    serial_mutex.unlock();
    
    while(1)
    {
        serial_mutex.lock();
        if(connected && !connected_written)
        {
            uLCD.locate(1,connected_row);
            uLCD.printf("             ");
            uLCD.locate(1,connected_row);
            uLCD.color(BLUE);
            uLCD.printf("CONNECTED");
            connected_written = true;
        }
        
        if(!connected && connected_written) 
        {
            uLCD.locate(1,connected_row);
            uLCD.printf("             ");
            uLCD.locate(1,connected_row);
            uLCD.color(RED);
            uLCD.printf("NOT CONNECTED");
            connected_written = false;
        }
        uLCD.locate(1,sonar_row+2);
        uLCD.printf("             ");
        uLCD.color(GREEN);
        uLCD.locate(1,sonar_row+2);
        uLCD.printf("%d cm", sonar_distance/10);
        
        serial_mutex.unlock();
        
        Thread::wait(300);
    }
}

/************************* SONAR ******************************/

volatile int tmpcount = 0;
int sum_distance = 0;
void dist(int distance)
{
    sum_distance += distance;
    ++tmpcount;
    if(tmpcount == 3)
    {
        sonar_distance = sum_distance/3;
        tmpcount = 0;
        sum_distance = 0;
        if( sonar_distance < 200)
        {
            move =  false;
        }
        else if(!move) move = true;
    }
}

ultrasonic mu(p5, p6, .1, 1, &dist);    //Set the trigger pin to D8 and the echo pin to D9
                                        //have updates every .1 seconds and a timeout after 1
                                        //second, and call dist when the distance changes

void sonar_func()
{
    mu.startUpdates();//start measuring the distance
        
    while(1)
    {
        //Do something else here
        mu.checkDistance();     //call checkDistance() as much as possible, as this is where
                                //the class checks if dist needs to be called.
        
        Thread::wait(50);
    }
}
/********************** Bluetooth ****************************/
Serial bluemod(p13,p14);
Timer bluemod_timer;
bool bluemod_t_started = false;


//C union can convert 4 chars to a float - puts them in same location in memory
//trick to pack the 4 bytes from Bluetooth serial port back into a 32-bit float
union f_or_char 
{
    float f;
    char  c[4];
};

/************************* MOTORS ******************************/   
volatile float working_setpointL, working_setpointR;
Motor mR(p21, p22, p23); // pwm, fwd, rev
Motor mL(p24, p29, p30); // pwm, fwd, rev


/*********PID CONTROLLER SPECIFIC DECLARATIONS********************************/
/*****************************************************************************/
float kp, ki, kd;                               // Working gain vars
float setpointL, setpointR;                                 // This is used by PID objects, because 
                                                // encoders are not quadrature and 
                                                // do not have direction information
                                                // we use this to store the absolute
                                                // value of the working_setpoint.
                                                // Specifically setpoint is used only
                                                // for PID objects. Refer to 
                                                // pid_callback() function
float feedbackL, outputL;                       // Should these be volatile?
float feedbackR, outputR;                       // Should these be volatile? 
const float output_lower_limit = 0.0;          
const float output_upper_limit = 1.0;
const float FEEDBACK_SCALE = 1.0/384.0;         // Scale feedback to 1rev/3000cnts
                                                // this is encoder specific.
enum CONTROL_MODE{PID_OFF = 0, PID_ON = 1};
int control_mode = PID_ON;
const float Ts = 0.04;                         // 25Hz Sample Freq (40ms Sample Time)
const float Ts_PID_CALLBACK = Ts/2.0;          // Update Motors and sensers twice as 
                                               // fast as PID sample rate, ensures
                                               // PID feedback is upto date every 
                                               // time PID calculations run

const float kp_init = 0.01;        // Good Kp for Speed Control; Start with this
const float ki_init= 0.015;        // Good Ki for Speed Control; Start with this
const float kd_init = 0.0001;      // Good Kd for Speed Control; Start with this

PID pidL(&setpointL, &feedbackL, &outputL,
        output_lower_limit, output_upper_limit,
        kp_init, ki_init, kd_init, Ts); 
PID pidR(&setpointR, &feedbackR, &outputR,
        output_lower_limit, output_upper_limit, 
        kp_init, ki_init, kd_init, Ts);
HALLFX_ENCODER encR(p15);
HALLFX_ENCODER encL(p16);
void pid_callback();            // Updates encoder feedback and motor output
Ticker motor;                   // Interrupt for feedback and motor updates           
float clip(float value, float lower, float upper);

/***************************************************************************/
/***************************************************************************/
/******************************* MAIN **************************************/
/***************************************************************************/
/***************************************************************************/

int main()
{
    Thread thread2;
    thread2.start(sonar_func);
    //Thread thread3;
    //thread3.start(uLCD_func);
    
    char bchecksum=0;
    char temp=0;
    union f_or_char x,y,z;
    encL.reset();
    encR.reset();
    feedbackL = encL.read();
    feedbackR = encR.read();
     
    // Update sensors and feedback twice as fast as PID sample time
    // this makes pid react in real-time avoiding errors due to 
    // missing counts etc. 
    motor.attach(&pid_callback, Ts_PID_CALLBACK);
    
    // Start PID sampling
    pidL.start();
    pidR.start();
    control_mode = PID_ON ;
    
    int dir;
    
    while(1){
        
        bchecksum=0;
        
        serial_mutex.lock();
        
        if(bluemod.readable())
        {
            if (bluemod.getc()=='!') 
            {
                if (bluemod.getc()=='A') //Accelerometer data packet
                { 
                    for (int i=0; i<4; i++) 
                    {
                        temp = bluemod.getc();
                        x.c[i] = temp;
                        bchecksum = bchecksum + temp;
                    }
                    for (int i=0; i<4; i++) 
                    {
                        temp = bluemod.getc();
                        y.c[i] = temp;
                        bchecksum = bchecksum + temp;
                    }
                    for (int i=0; i<4; i++) 
                    {
                        temp = bluemod.getc();
                        z.c[i] = temp;
                        bchecksum = bchecksum + temp;
                    }
                    //serial_mutex.unlock();
                    
                    if(connected == false)
                    {
                        connected = true;
                    }
                    
                    if(bluemod_t_started == true)
                    {
                        bluemod_timer.stop();
                        bluemod_timer.reset();
                        bluemod_t_started = false;
                    }
                    
                    float radius = sqrt(x.f*x.f + y.f*y.f + z.f*z.f);
                    
                    
                    float theta  = 180*atan(y.f/x.f)/(PI); //[-90, 90] degrees
                    float phi = 180*atan(sqrt(x.f*x.f+y.f*y.f)/z.f)/(PI); //[-90, 90] degrees
                    float mag_phi = fabs(phi);    //magnitude of phi
                    float mag_theta = fabs(theta);//magnitude of theta
                    
                    x.f = clip(x.f, -1.0, 1.0); // |x.f|>1.0 => |x.f|=1.0
                    y.f = clip(y.f, -1.0, 1.0); // |y.f|>1.0 => |y.f|=1.0
                    if(fabs(y.f) < 0.15) y.f = 0.0; // Thresholding at +-0.5
                    if(fabs(x.f) < 0.15) x.f = 0.0;
                                    
                    x.f = -x.f;
                    dir = (x.f > 0)? 1: -1; //1 = reverse (x.f<0), -1 forward
                    
                    if(mag_phi > 90) mag_phi = 90;
                    
                    if(x.f == 0.0)
                    {
                        dir = 0;
                        if(y.f == 0.0)
                        {
                            working_setpointR=0.0;
                            working_setpointL=0.0;
                           
                        }
                        else 
                        { 
                            working_setpointL = 90*y.f/2; //increase forward speed
                            working_setpointR = -90*y.f/2; //increase reverse speed
                        }
                        
                    }
                    else
                    {
                        if(dir > 0) //if reverse
                        {
                            working_setpointR=-dir*mag_phi; //negate
                            working_setpointL=-dir*mag_phi; //for correct functioning
                        }
                        else if(dir == -1 && move) //if want to move backward, only do so if move == true
                        {
                            working_setpointR=-dir*mag_phi;
                            working_setpointL=-dir*mag_phi;
                        }
                    }
                    if(dir == -1)
                    {
                        if( sonar_distance < 150 + 100*(3*mag_phi/90))
                        {
                            move =  false;
                            working_setpointR = 0.0;
                            working_setpointL = 0.0;
                            mR.speed(0.0);
                            mL.speed(0.0);
                        }
                        else
                        {
                            move = true;
                        }
                    }
                }
            }
        } // end of: if(bluemod.readable())
        
        else // if not readable()
        {
            if(connected == true && bluemod_t_started == false)
            {
                bluemod_timer.start();
                bluemod_t_started = true;
            }
            
            if(connected == true && bluemod_t_started == true && bluemod_timer.read_ms() > 600)
            {
                connected = false;
                bluemod_timer.stop();
                bluemod_timer.reset();
                bluemod_t_started = false;
                working_setpointR = 0;
                working_setpointL = 0;
                mL.speed(0);
                mR.speed(0);
            }
        }
        serial_mutex.unlock();
    }
                        
}

void pid_callback(){
    static int lastMode = control_mode;
    
    // If control_mode is PID_OFF turn off PID controllers and run motors in open loop
    if(control_mode == PID_OFF){
        // Mode changed; turn off PID controllers
        if(lastMode != control_mode){
            pidL.stop(); pidR.stop();
            lastMode = PID_OFF;
        }
        // Set motor speed in open loop mode
        // Motor direction based on working setpoint var
        int dirL, dirR;
        if(working_setpointR < 0.0){
             dirR = -1;//reverse
        }
        else{
            dirR = 1;//forward
        }
        if(working_setpointL < 0.0){
             dirL = -1;
        }
        else{
            dirL = 1;
        }
        float speedR = abs(working_setpointR) / 90.0; // Normalize based on 90 RPM
        float speedL = abs(working_setpointL) / 90.0; // Normalize based on 90 RPM
        if(move)
        {
            mL.speed(speedR * dirL);
            mR.speed(speedL * dirR);
        }
        else
        {
            mL.speed(0.0);
            mR.speed(0.0);
        }
    }
    // else control_mode is PID_ON, turn on PID controllers and run motors in closed loop
    else{
        // Mode changed; turn on PID controllers
        if(lastMode != control_mode){
            pidL.start(); pidR.start();
            lastMode = PID_ON;
        }
        // Deal with feedback and update motors
        // Motor direction based on working setpoint var
        int dirL, dirR;
        if(working_setpointR < 0.0){
             dirR = -1; //reverse
        }
        else{
            dirR = 1;//forward
        }
        if(working_setpointL < 0.0){
             dirL = -1;
        }
        else{
            dirL = 1;
        }
        
        
        
        if(move || (dirR == -1 && dirL == -1)) //allow only backward movement if move==false
        {
            // Setpoint vars used by PID objects are concerned with 
            // only SPEED not direction. 
            setpointR = abs(working_setpointR);
            setpointL = abs(working_setpointL);
        
            float k = Ts_PID_CALLBACK;    // Discrete time, (Ts/2 because this callback is called
                                          // at interval of Ts/2... or twice as fast as pid controller)
            static int last_count_L = 0;
            static int last_count_R = 0;
            int countL = encL.read();
            int countR = encR.read();
            
            // Because encoders are not quadrature we must handle the sign ourselves, 
            // i.e. explicitly make calcs based on the direction we have set the motor
            float raw_speed_L = ((countL - last_count_L)*FEEDBACK_SCALE) / k; 
            float rpm_speed_L = raw_speed_L * 30.0;     // Convert speed to RPM
            
            float raw_speed_R = ((countR - last_count_R)*FEEDBACK_SCALE) / k; 
            float rpm_speed_R = raw_speed_R * 30.0;     // Convert speed to RPM
            
            last_count_L = countL;                      // Save last count
            last_count_R = countR;
            feedbackL = rpm_speed_L;
            feedbackR = rpm_speed_R;
            mL.speed(outputL * dirL);
            mR.speed(outputR * dirR);
        }
        else
        {
            mL.speed(0.0);
            mR.speed(0.0);
        }
    }
}

/*
    Clips value to lower/ uppper
    @param value    The value to clip
    @param lower    The mininum allowable value
    @param upper    The maximum allowable value
    @return         The resulting clipped value
*/
float clip(float value, float lower, float upper){
    return std::max(lower, std::min(value, upper));
}