/**************************************************************************
 * @file     ESCMControlApp.h
 * @brief    Main class for wrapping the interface with ESCM Control 
 *           Application
 * @version: V1.0
 * @date:    9/17/2019

 *
 * @note
 * Copyright (C) 2019 E3 Design. All rights reserved.
 *
 * @par
 * E3 Designers LLC is supplying this software for use with Cortex-M3 LPC1768
 * processor based microcontroller for the ESCM 2000 Monitor and Display.  
 *  *
 * @par
 * THIS SOFTWARE IS PROVIDED "AS IS".  NO WARRANTIES, WHETHER EXPRESS, IMPLIED
 * OR STATUTORY, INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE.
 * ARM SHALL NOT, IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 *
 ******************************************************************************/
 #ifndef _ESCM_CONTROL_APP_
#define _ESCM_CONTROL_APP_


#include "mbed.h"
#include "Menu.h"
#include "EditTimeMenu.h"
#include "EditAddressMenu.h"
#include "DisplayCodesMenu.h"
#include "FactoryResetMenu.h"
#include "LCD.h"
#include "AddressMap.h"
#include "EventLog.h"
#include "TimeUtilities.h"

#include "emic2.h"


#define LIMIT(x,low,high) x = ((x<=low)? low    : ((x>=high) ? high : x ));
#define WRAP(x,low,high) ((x<low)?(high-1):((x>high)?0:x));

typedef struct {
    int    event;
} event_t;

typedef struct {
    char   message[64];
} playbackMessage_t;

extern Serial pc;

extern LCD    lcd;
extern Menu                    rootMenu;
extern EditTimeMenu            editTime;
extern EditAddressMenu         editAddress;
extern DisplayCodesMenu        showEvents;
extern FactoryResetMenu        factoryReset;

extern AddressMap addressMap;

extern ESCM_EventLog escmEventLog;

extern RealTimeClock rtc;

extern Serial     escmRs485_Input;
extern DigitalOut escmRs485_Mode; 

extern Serial     microCommRs485_Tx; 
extern DigitalOut microCommRs485_Mode; 

extern CAN        microCommCanItf;       
extern DigitalOut microCommCan_Mode; 

extern emic2 speaker;

extern CircularBuffer<event_t, 64> event_queue;

extern CircularBuffer<playbackMessage_t, 10> playback_queue;


class ESCMControlApp 
{
    
    public:
        
        void init(void);
        void update(void);
        void processMessageQueue ();
        void processSoundQueue ();
        
        void relayMessage(int address);
        void tx485Message(int address); 
        void txCanMessage501 (int address);
        void txCanMessage502 (int address);
        void txCanMessage580 (int address);

        static void info    (char *format, ...);
        static void warning (char *format, ...);
        static void error   (char *format, ...);
        
        static void say (int address);
        static void say (char *format, ...);
        static void refresh_display (void);
        
        static void getTime(char *timeBuffer);
        static void setTime(int hours, int mins, int sec, int months, int days, int years );
        
        static void postEvent (uint16_t address, uint16_t source);
     
        int cur_address;
        
        char fault[4];
        
        int dataRxCnt;
        int dataTxCnt;
        
        Mutex event_ctrl ;
        Mutex emic_ctrl ;
        Mutex can_ctrl  ;
        Mutex rs485_ctrl;
};

extern ESCMControlApp escmController;

#endif