/**************************************************************************
 * @file     emic2.h
 * @brief    Base class for wrapping the interface with the EMIC2 Sound Care 
 * @version: V1.0
 * @date:    9/17/2019

 *
 * @note
 * Copyright (C) 2019 E3 Design. All rights reserved.
 *
 * @par
 * E3 Designers LLC is supplying this software for use with Cortex-M3 LPC1768
 * processor based microcontroller for the ESCM 2000 Monitor and Display.  
 *  *
 * @par
 * THIS SOFTWARE IS PROVIDED "AS IS".  NO WARRANTIES, WHETHER EXPRESS, IMPLIED
 * OR STATUTORY, INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE.
 * ARM SHALL NOT, IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 *
 ******************************************************************************/
#ifndef _EMIC2_DRIVER_
#define _EMIC2_DRIVER_

#include "mbed.h"

#define speakf printf

class emic2 : public Stream
{
public :
    emic2(PinName tx, PinName rx): _cmd(tx,rx) {
        _cmd.baud(9600);
        _cmd.putc('X'); //stop talking if reset and not a power on
        _cmd.putc('\r'); // Send a CR in case the system is already up
        wait(1); //delay for emic power on boot or reset respone
        while (_cmd.getc() != ':');   // When the Emic 2 has initialized and is ready, it will send a single ':'
        while (_cmd.readable()) _cmd.getc();//flush out buffer just in case
    };
    void ready() {
        while (_cmd.getc() != ':')
        {
            ThisThread::sleep_for(1);
        }
        while (_cmd.readable()) {
            _cmd.getc();//flush out recieve buffer just in case
            ThisThread::sleep_for(1);
        }
    };
    int readable() {
        return _cmd.readable();
    };
    int getc() {
        return _cmd.getc();
    }
    void volume(int x) {
        speakf("V%d\r",x);
        ready();
    }
    void voice(int x) {
        speakf("N%d\r",x);
        ready();
    }
protected :
    Serial     _cmd;
    //used by printf - supply it and printf works!
    virtual int _putc(int c) {
        _cmd.putc(c);
        ThisThread::sleep_for(1);
        return 0;
    };
    virtual int _getc() {
        return -1;
    };
};

#endif
