/**************************************************************************
 * @file     DisplayCodeMenu.cpp
 * @brief    Base class for implementing the Display Code Menu display
 * @version: V1.0
 * @date:    9/17/2019

 *
 * @note
 * Copyright (C) 2019 E3 Design. All rights reserved.
 *
 * @par
 * E3 Designers LLC is supplying this software for use with Cortex-M3 LPC1768
 * processor based microcontroller for the ESCM 2000 Monitor and Display.  
 *  *
 * @par
 * THIS SOFTWARE IS PROVIDED "AS IS".  NO WARRANTIES, WHETHER EXPRESS, IMPLIED
 * OR STATUTORY, INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE.
 * ARM SHALL NOT, IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 *
 ******************************************************************************/
#include "mbed.h"
#include "DisplayCodesMenu.h"
#include "TimeUtilities.h"
#include "ESCMControlApp.h"


/******************************************************************************/
DisplayCodesMenu::DisplayCodesMenu(char* id): Menu(id)
{

    active_selection = 0;
    row=0;
    column=0;
    update_needed=1;
    current_index=0;
    current_line = 1;
    top = 0;

}



/******************************************************************************/
void DisplayCodesMenu::init()
{
    active_selection = 0;
    row=0;
    column=0;
    update_needed=1;
    current_index= 0;
    current_line = 1;
    top = 0;
}


/******************************************************************************/
void DisplayCodesMenu::display(LCD * lcd)
{
    static int lastPos  = 0;
    static int lastSize = 0;
    
    int hr = 0;
    int update_page = 0;
    char buf[40];
    
    //displayCurrentTime(lcd);

    if(escmEventLog.head_ != lastPos ) {
        
        current_index = 0;
        top = current_index;
        bottom = top + 2;
        
        lastPos  = escmEventLog.head_;
        lastSize = escmEventLog.size();
        update_needed=1;
    }
    

    // paging
    if (current_index < top )
    {
        top    = current_index ;
        bottom = current_index + 2;
        update_page = 1;
        update_needed=1;
    }
    else if (current_index > bottom)
    {
        top    = current_index - 2;
        bottom = current_index;
        update_page = 1;
        update_needed=1;
    } 
    else
    {
        
    }
    

    if (update_needed) {
        
        lock();
        
        switch (active_selection ) {
            case 2:
                lcd->cls();
                lcd->locate(1,0);
                lcd->printf("Are you sure you want to erase");
                lcd->locate(2,0);
                lcd->printf("all events (Press Set)?");
                update_needed = 0;
                break;
            case 3:
                lcd->cls();
                lcd->locate(1,0);
                lcd->printf("Deleting Events...");
                erase_log     = 0;
                escmEventLog.reset();
                escmEventLog.save();
                update_needed = 1;
                active_selection = 0;
                current_index  = 0;
                break;

            case 1:
                // DISPLAY EXPANDED INFO
                lcd->cls();
                lcd->locate(0,0);
                lcd->printf(" ID | Port | Unit | Time");
                lcd->locate(0,33);
                lcd->printf("Addr=%02d",escmController.cur_address);

                // update display
                for(int i=0; i<3; i++) {

                    int index = top + i;
                    int line  = (1+i);

                    int selected = current_index == index;

                    ESCM_Event *event = escmEventLog.index (index) ;

                    if (event != NULL && index < escmEventLog.size() ) {
                        
                        if ( event->hours < 12 ) {
                            
                            hr = event[i].hours; 
                            hr = (hr == 0) ? 12 : hr;
                            sprintf(buf,"%02d | %02d  | %02d  | %02d:%02d:%02dam %02d/%02d/%04d",
                                index + 1,
                                event->port,
                                event->address,
                                hr,
                                event->mins,
                                event->secs,
                                event->month,
                                event->day,
                                event->year
                               );
                               
                        }
                        else  {
                            hr = (event->hours - 12);     
                            hr = (hr == 0) ? 12 : hr;
                            ;                   
                            sprintf(buf,"%02d | %02d  | %02d  | %02d:%02d:%02dpm %02d/%02d/%04d",
                                index + 1,
                                event->port,
                                event->address,
                                hr,
                                event->mins,
                                event->secs,
                                event->month,
                                event->day,
                                event->year
                               );
                               
                        }
                            
                    } else {
                        sprintf(buf,"%-s | %-s  | %-s  | %-s",
                                "--",
                                "--",
                                "--",
                                "- N/A -");
                    }
                    
                    if(selected) {
                        lcd->locate(line,0);
                        lcd->printf(">%-39s",buf);
                    } else {
                        lcd->locate(line,0);
                        lcd->printf(" %-39s",buf);
                    }

                    //lcd->locate(line,2);
                    //lcd->printf(buf);

                }
                update_page = 0;
                update_needed=0;
                break;
                break;
            default:

                lcd->cls();
                lcd->locate(0,0);
#if 0
                lcd->printf(" ID | Port | Unit | Time");
#else
                lcd->printf(" ID | Description  | Time");
#endif

                lcd->locate(0,33);
                lcd->printf("Addr=%02d",escmController.cur_address);
                
                // update display
                for(int i=0; i<3; i++) {

                    int index = top + i;
                    int line  = (1+i);

                    int selected = current_index == index;


                    ESCM_Event *event = escmEventLog.index (index) ;

                    if (event != NULL && index < escmEventLog.size() ) {
#if 0
                        sprintf(buf,"%02d | %02d   | %02d   | %02d:%02d:%02d %02d/%02d/%04d",
                                index + 1,
                                event->port,
                                event->address,
                                event->hours,
                                event->mins,
                                event->secs,
                                event->month,
                                event->day,
                                event->year
                               );
#else
                        char * ev_desc = addressMap.getDescription(event->address);
                    
                        if ( event->hours < 12 ) {
                            hr = (event[i].hours == 0) ? 12 : event[i].hours;
                            hr = (hr == 0) ? 12 : hr;
                            sprintf(buf,"%02d | %-12s | %02d:%02d:%02dam %02d/%02d/%02d",
                                
                                event->address,
                                ev_desc,
                                hr,
                                event->mins,
                                event->secs,
                                event->month,
                                event->day,
                                (event->year % 100)
                               );
                        } else  {
                            
                            hr = (event->hours - 12);
                            hr = (hr == 0) ? 12 : hr;
                            
                            sprintf(buf,"%02d | %-12s | %02d:%02d:%02dpm %02d/%02d/%02d",
                                
                                event->address,
                                ev_desc,
                                hr,
                                event->mins,
                                event->secs,
                                event->month,
                                event->day,
                                (event->year % 100)
                               );
                        }
                    
#endif

                    } else {
#if 0
                        sprintf(buf,"%-s | %-s   | %-s   | %-s",
                                "--",
                                "--",
                                "--",
                                "- N/A -");
#else
                        sprintf(buf,"%-s | %-s        | %-s",
                                "--",
                                "Empty",
                                "- N/A -");
#endif
                    }
                    
                    if(selected) {
                        lcd->locate(line,0);
                        lcd->printf(">%-39s",buf);
                    } else {
                        lcd->locate(line,0);
                        lcd->printf(" %-39s",buf);
                    }

                    //lcd->locate(line,2);
                    //lcd->printf(buf);

                }
                update_page = 0;
                update_needed=0;
                break;
        }
        
        unlock();
    }
}


/******************************************************************************/
void DisplayCodesMenu::pressSet()
{
    switch(active_selection) {

        case 0:  
        case 1: 
        { // normal

            ESCM_Event *event = escmEventLog.index(current_index);
            if (event!=NULL)
            {
                escmController.say(event->address );
                escmController.relayMessage(event->address );
            }
            update_needed = 0;
            break;
        }
        case 2: // press yes
            active_selection = 3;
            update_needed = 1;
            break;
        case 3: // press yes
        default:
            active_selection = 0;
            break;
    }
}

/******************************************************************************/
void DisplayCodesMenu::pressMode()
{

    switch(active_selection) {
        case 0: // normal
            active_selection = 1;
            update_needed = 1;
            break;
        case 1: // normal
            active_selection = 2;
            update_needed = 1;
            break;
        case 2: // normal
        case 3: // normal
        default:
            active_selection = 0;
            update_needed = 1;
            break;
    }
}

/******************************************************************************/
void DisplayCodesMenu::pressDown()
{

    current_line++;
    current_index++;
    
    if (current_index >= escmEventLog.size() )
        current_index = 0;
        
    
    if (current_line > 3 ) 
        current_line = 3;
    
    update_needed=1;
}

/******************************************************************************/
void DisplayCodesMenu::pressUp()
{
    current_line--;
    current_index--;
    if (current_index < 0)
        current_index = escmEventLog.size()-1;
            
    if (current_line < 1 ) 
        current_line = 1;
        
    update_needed =1;
}
/******************************************************************************/
