/**************************************************************************
 * @file     Menu.h
 * @brief    Base class for wrapping the interface with LCD Menu display
 * @version: V1.0
 * @date:    9/17/2019

 *
 * @note
 * Copyright (C) 2019 E3 Design. All rights reserved.
 *
 * @par
 * E3 Designers LLC is supplying this software for use with Cortex-M3 LPC1768
 * processor based microcontroller for the ESCM 2000 Monitor and Display.  
 *  *
 * @par
 * THIS SOFTWARE IS PROVIDED "AS IS".  NO WARRANTIES, WHETHER EXPRESS, IMPLIED
 * OR STATUTORY, INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE.
 * ARM SHALL NOT, IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 *
 ******************************************************************************/
 
#include "mbed.h"
#include "Menu.h"
#include "ESCMControlApp.h"

/******************************************************************************/
Menu* Menu::currentMenu = NULL;


/******************************************************************************/
Menu * Menu::getCurrentMenu () 
{ 
    return currentMenu; 
}

/******************************************************************************/
Menu * Menu::setCurrentMenu (Menu * value) 
{ 

    //printf("\n Goto Menu :%s ", value->getText() );
    currentMenu = value; 
    currentMenu->init();
    return currentMenu;
}


/******************************************************************************/
Menu::Menu(char *id) : menuID(id), parent(NULL),update_needed(1)
{
    children.clear();
    
    selectedMenu =  0; // default menu
    update_needed = 1;
}

/******************************************************************************/
void Menu::add(Menu *submenu)
{
    submenu->parent = this;
    children.push_back(submenu);
}


/******************************************************************************/
void Menu::init()
{
    update_needed=1;
}
        
/******************************************************************************/
void Menu::select(void)
{
    Menu::setCurrentMenu ( children[selectedMenu]) ;
}


/******************************************************************************/
void Menu::back()
{
    if (parent != NULL )
    {
        Menu::setCurrentMenu (parent);
    }
    else
    {
        printf("ERROR: missing parent ");
    }
}

/******************************************************************************/
void Menu::DrawDisplay(LCD * lcd)
{
    this->lcd = lcd;
    
    static unsigned long time_max = 0 ;
    unsigned long time_delta = 0;
    unsigned long time_ms1 = us_ticker_read() / 1000L; 
    
      
    lock();
    display(lcd);
    unlock();

    unsigned long time_ms2 = us_ticker_read() / 1000L;
    
    time_delta = time_ms2-time_ms1;
    if (time_max  < time_delta )
    {   
        time_max  = time_delta;
        printf("Tmax=%d\n\r", time_max);
    }
    
}

/******************************************************************************/
void Menu::display(LCD * lcd)
{   
    static int top    = 1;
    static int bottom = 3;
    static int size   = children.size();
    
    // paging
    if (selectedMenu < top )
    {
        top    = selectedMenu ;
        bottom = selectedMenu + 2;
        update_needed=1;
    }
    else if (selectedMenu > bottom)
    {
        top    = selectedMenu - 2;
        bottom = selectedMenu;
        update_needed=1;
    } 
    else
    {
        
    }

    if (update_needed) {
        
        lcd->cls();
        lcd->locate(0,0);
#if 1
        lcd->printf("ESCM Controller");
#else

        lcd->printf("(%d,%d,%d)", top,selectedMenu,size);
#endif
        displayVersion(lcd); 
        
        for ( int i=0; i< 3;i++)
        {   
            int index = top + i;
            int line  = (1+i);
            int selected = (selectedMenu == index);
            
            lcd->locate(line,0);
            
            if (selected)
                lcd->printf(">%s",children[index]->getText());
            else
                lcd->printf(" %s",children[index]->getText());
            
        }
        
        update_needed = 0;
    }
    
    displayCurrentTime(lcd);
    
}


/******************************************************************************/
void Menu::moveUp()
{        

    cursorLine--;
    selectedMenu--;
    
    if(selectedMenu < 0)
    {
        selectedMenu = children.size()-1;
    }
    
    if (cursorLine < 1 ) 
        cursorLine = 1;
        
        
    update_needed = 1;
}
/******************************************************************************/
void Menu::moveDown()
{
    cursorLine++;
    selectedMenu++;
    
    if(selectedMenu >= children.size())
    {
        selectedMenu=0;
    }
    
    if (cursorLine > 3 ) 
        cursorLine = 3;
        
    update_needed = 1;
    
}

/******************************************************************************/
void Menu::pressMode()
{
    select();
}
        
/******************************************************************************/
void Menu::pressSet()
{
    select();
}

/******************************************************************************/
void Menu::pressUp()
{
    moveUp();
}
        
/******************************************************************************/
void Menu::pressDown()
{
    moveDown();
}

/******************************************************************************/
void Menu::pressClear()
{
    back();
}

/******************************************************************************/
void Menu::displayVersion (LCD * lcd)
{
    lcd->locate(3,35);
    lcd->printf("v1.02");
}
/******************************************************************************/
void Menu::displayCurrentTime (LCD * lcd)
{
   #if 1 
    time_t rawtime;
    struct tm * timeinfo;
    
    int cur_hours,cur_mins,cur_secs,cur_year,cur_month,cur_day;
                
    time ( &rawtime );
    timeinfo = localtime ( &rawtime );  timeinfo = localtime (&rawtime);
    
    cur_hours   = timeinfo->tm_hour;
    cur_mins    = timeinfo->tm_min;
    cur_secs    = timeinfo->tm_sec;
    cur_year    = timeinfo->tm_year+1900;
    cur_month   = timeinfo->tm_mon + 1;
    cur_day     = timeinfo->tm_mday;
    
#if 1

    if ( cur_hours < 12 ) {

        cur_hours = (cur_hours == 0) ? 12 : cur_hours;
        lcd->locate(0,20);
        lcd->printf(" %02d:%02d:%02dam %02d/%02d/%02d",
                    cur_hours, cur_mins, cur_secs, cur_month, cur_day, (cur_year%100));

    } else  {

        cur_hours = (cur_hours - 12);
        cur_hours = (cur_hours == 0) ? 12 : cur_hours;
        lcd->locate(0,20);
        lcd->printf(" %02d:%02d:%02dpm %02d/%02d/%02d",
                    cur_hours, cur_mins, cur_secs, cur_month, cur_day, (cur_year%100));


    }

#else

    lcd->locate(0,20);
    lcd->printf(" %02d:%02d:%02d %02d/%02d/%04d", 
        cur_hours, cur_mins, cur_secs, cur_month, cur_day, cur_year);

#endif
    
    lcd->locate(1,33);
    lcd->printf("Addr=%02d",escmController.cur_address);
    #endif
}
/******************************************************************************/
