/**************************************************************************
 * @file     LCD.h
 * @brief    Base class for wrapping the interface with the New Haven 40x4
 *           LCD monitor. 
 * @version: V1.0
 * @date:    9/17/2019

 *
 * @note
 * Copyright (C) 2019 E3 Design. All rights reserved.
 *
 * @par
 * E3 Designers LLC is supplying this software for use with Cortex-M3 LPC1768
 * processor based microcontroller for the ESCM 2000 Monitor and Display.  
 *  *
 * @par
 * THIS SOFTWARE IS PROVIDED "AS IS".  NO WARRANTIES, WHETHER EXPRESS, IMPLIED
 * OR STATUTORY, INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE.
 * ARM SHALL NOT, IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 *
 ******************************************************************************/
#ifndef _LCD_
#define _LCD_

#include "mbed.h"

/*
 * LCD commands.
 */
// Clear display:
#define LCD_CLEAR   (1 << 0)

// return cursor and LCD to home position:
#define LCD_HOME   (1 << 1)

// set cursor move direction:
#define LCD_MOVEDIR   (1 << 2)
// shift display when byte written to display
#define LCD_MOVEDIR_SHIFT   (1 << 0)
// increment the cursor after each byte written:
#define LCD_MOVEDIR_ID   (1 << 1)

// Enable display/cursor:
#define LCD_DISPEN   (1 << 3)
// turn cursor blink on:
#define LCD_DISPEN_BLINK   (1 << 0)
// turn cursor on:
#define LCD_DISPEN_CURSOR   (1 << 1)
// turn display on:
#define LCD_DISPEN_DISP   (1 << 2)

// move cursor/shift display:
#define LCD_MCSD         (1 << 4)
// direction of shift (right if set):
#define LCD_MCSD_RL      (1 << 2)
// turn on display shift:
#define LCD_MCSD_SC      (1 << 3)

// set interface length:
#define LCD_SETIFLEN     (1 << 5)
// set character font 5x10 (1) or 5x7 (0)
#define LCD_SETIFLEN_F   (1 << 2)
// set # of display lines 1 if 0, 2 if 1:
#define LCD_SETIFLEN_N   (1 << 3)
// set interface length 8 bits if 1, 4 bits if 0
#define LCD_SETIFLEN_DL   (1 << 4)

// move cursor into CGRAM:
#define LCD_MOVERAM       (1 << 6)

// Move cursor to display:
#define LCD_MOVEDISPLAY   (1 << 7)
#define LCD_DATAREAD  0x00
#define LCD_DATAWRITE  0xFF
#define LCD_READ  1
#define LCD_WRITE  0
#define LCD_HIGH  1
#define LCD_LOW  0

#define LCD_RS_COMMANDREGISTERS 0
#define LCD_RS_DATAREGISTERS    1

#define LCD_wakeUp       0x30  // 
#define LCD_functionSet  0x28  // Sets the interface data type to 4 bits, 2 lines, and font size 5x8. 0x28 = 0010 1000
#define LCD_returnHome   0x02  // Sets the cursor to the beginning of the screen. 0x02 = 0000 0010
#define LCD_setCursor    0x10  // Sets cursor moving and display shift, and direction without changing DDRAM data. 0x10 = 0001 0000
#define LCD_setDisplayOn 0x0C  // Sets Entire Display on, Display on, Cursor on. 0x0C = 0000 1011
#define LCD_setDisplayEdit 0x0E  // Sets Entire Display on, Display on, Cursor on. 0x0C = 0000 1011
#define LCD_setDisplayBlink 0x0F// Sets Entire Display on, Display on, Cursor on. 0x0C = 0000 1011
#define LCD_setDisplayOff 0x08  // Sets Entire Display off, Display on, Cursor on. 0x0C = 0000 1000
#define LCD_entryModeSet 0x06  // Sets Entry Mode. 0x06 = 0000 0110
#define LCD_cursorModeSet 0x06 // Sets Entry Mode. 0x06 = 0000 0110
#define LCD_clearDisplay 0x01  // Resets the whole display to blank



class LCD
{
public:

    LCD();
    
    virtual ~LCD();

    void init(void) ;

    void display(char *show);
    
    void clear();
    
    void cls();

    void locate(int row, int column);

    void putc(const char c);

    void printf(const char* fmt, ...);

    int rows()
    {
        return 4;
    }

    int columns()
    {
        return 40;
    }

private:
    void setCursorMode (int mode);

    void clear(int e1,int e2);

    void writeByte1(int value);
    void writeByte2(int value);

    void command1(char i) ;
    void command2(char i) ;

    void writedata1(char i) ;
    void writedata2(char i) ;

    void nextline1();
    void nextline2();

    int isBusy(int e1,int e2);

    int isBusy1(void);
    int isBusy2(void);

    void waitBusy();

    void write(const char* text);
    void writeLine(int line, const char* text );
    void writeCharacter(const char c, int row, int column);
    void writeCharacters(const char* text, int row, int column);

    void writeCommand(char value,int e1,int e2);
    void writeChar   (char value,int e1,int e2);
    void writeData   (char value,int e1,int e2);

    int address(int row, int column);

    void character (int row, int column, char c);

    DigitalInOut LCD_E1;
    DigitalInOut LCD_E2;
    DigitalInOut LCD_RS;
    DigitalInOut LCD_RW;

    BusInOut LCD_DATA;

    Mutex mutex;

    volatile int _row;
    volatile int _column;
    volatile char isOutput;

};

#endif
