/**************************************************************************
 * @file     EventLog.h
 * @brief    Base class for wrapping the interface with the ESCM Event Log 
 * @version: V1.0
 * @date:    9/17/2019

 *
 * @note
 * Copyright (C) 2019 E3 Design. All rights reserved.
 *
 * @par
 * E3 Designers LLC is supplying this software for use with Cortex-M3 LPC1768
 * processor based microcontroller for the ESCM 2000 Monitor and Display.  
 *  *
 * @par
 * THIS SOFTWARE IS PROVIDED "AS IS".  NO WARRANTIES, WHETHER EXPRESS, IMPLIED
 * OR STATUTORY, INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE.
 * ARM SHALL NOT, IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 *
 ******************************************************************************/

#ifndef _EVENT_LOG_
#define _EVENT_LOG_

/* fault log */
#include "mbed.h"
#define  EVENT_LOG_FORMAT 0xFAFA
#define  MAX_EVENTS       50

class ESCM_Event
{
public:

    uint16_t port;
    uint16_t address;
    uint16_t hours;
    uint16_t mins;
    uint16_t secs;
    uint16_t month;
    uint16_t day;
    uint16_t year;
    
    
    ESCM_Event();
    ESCM_Event(uint16_t address);
    
    
    void getTime (char* buffer);
    void setTimeNow(void);
    

};

class ESCM_EventLog
{

public:


    ESCM_EventLog();
    virtual ~ESCM_EventLog();
    
    ESCM_Event events[MAX_EVENTS] ;
    
    int init();
    int load();
    int save();
    void clear();
    
    void display(Serial *pc);
    void add(uint16_t address);
    void add(uint16_t address, uint16_t port);
    ESCM_Event * index (int pos );

    Mutex _mutex;
    
    static const int max_size_ = MAX_EVENTS;
    
    int head_ ;
    int tail_ ;
    bool full_ ;
    bool dirty ;

    void put(ESCM_Event item){
        
        events[head_] = item;

        head_ = (head_ + 1) % max_size_;
        full_ = head_ == tail_;
        
        if(full_)    
        {    
            tail_ = (tail_ + 1) % max_size_;
        }
        
        dirty = true;
    
    }
    
    ESCM_Event get() {
           
        if(empty())
        {
            return ESCM_Event();
        }
    
        //Read data and advance the tail (we now have a free space)
        ESCM_Event val = events[tail_];
        full_ = false;
        tail_ = (tail_ + 1) % max_size_;
        
        dirty = true;
        return val;
    
    }
    
    void reset() 
    {
        head_ = tail_;
        full_ = false;
        dirty = true;
    }
    
    bool empty() const
    {
        
        //if head and tail are equal, we are empty
        return (!full_ && (head_ == tail_));
    }
    
    bool full() const
    {
        //If tail is ahead the head by 1, we are full
        return full_;
    }
    
    int capacity() const
    {
        return max_size_;
    }
    
    int size() const
    {
        int size = max_size_;
        if(!full_)
        {
            if(head_ >= tail_)
            {
                size = head_ - tail_;
            }
            else
            {
                size = max_size_ + head_ - tail_;
            }
        }
        return size;
   }
};
#endif
