/**************************************************************************
 * @file     EventLog.cpp
 * @brief    Base class for wrapping the interface with the ESCM Event Log 
 * @version: V1.0
 * @date:    9/17/2019

 *
 * @note
 * Copyright (C) 2019 E3 Design. All rights reserved.
 *
 * @par
 * E3 Designers LLC is supplying this software for use with Cortex-M3 LPC1768
 * processor based microcontroller for the ESCM 2000 Monitor and Display.  
 *  *
 * @par
 * THIS SOFTWARE IS PROVIDED "AS IS".  NO WARRANTIES, WHETHER EXPRESS, IMPLIED
 * OR STATUTORY, INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE.
 * ARM SHALL NOT, IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 *
 ******************************************************************************/
#include "mbed.h"
#include "EventLog.h"
#include "stdio.h"




/*-------------------------------------------------------------------
 * define local file system
 *-------------------------------------------------------------------*/
LocalFileSystem local("local");

const char * filename = "/local/events.bin";
    

/******************************************************************************/
ESCM_Event::ESCM_Event(): address(0),port(0xFF)
{
    
}

/******************************************************************************/
ESCM_Event::ESCM_Event(uint16_t address): address(address),port(0xFF)
{
    setTimeNow();
}

/******************************************************************************/
void ESCM_Event::setTimeNow(void)
{
    time_t rawtime;
    struct tm * timeinfo;

    time ( &rawtime );
    timeinfo = localtime ( &rawtime );  timeinfo = localtime (&rawtime);
    
    hours   = timeinfo->tm_hour;
    mins    = timeinfo->tm_min;
    secs    = timeinfo->tm_sec;
    year    = timeinfo->tm_year+1900;
    month   = timeinfo->tm_mon + 1;
    day     = timeinfo->tm_mday;
}

/******************************************************************************/
void ESCM_Event::getTime (char * buffer) {
    sprintf(buffer,"%02d:%02d %02d/%02d/%04d", hours,mins,month,day,year);
}
    


/******************************************************************************/
ESCM_EventLog::ESCM_EventLog() {
    full_=false;
    dirty=false;
    head_=0;
    tail_=0;

}

/******************************************************************************/
ESCM_EventLog::~ESCM_EventLog() {
}

/******************************************************************************/
int ESCM_EventLog::init()
{
    full_=false;
    dirty = true;
    head_=0;
    tail_=0;

    printf("Initializing Event Log\n\r");
    if ( !load() ) {
        printf("..Load Failed \n\r");
        printf("..Setting Default Event Log\n\r");
        memset ( events,0,sizeof(ESCM_Event)*MAX_EVENTS);
        // for testing
        for ( int i=0; i<MAX_EVENTS; i++) {
            add(i*2);
        }
        save();
    } else {
        printf("..Loaded Event Log\n\r");
    }

    printf("Complete\n\r");
    return 1;
}

/******************************************************************************/
int ESCM_EventLog::load()
{    
    printf("Reading %s ...\n\r",filename );
    FILE *input = fopen(filename, "rb");
    if(input!= NULL){
        
        uint16_t code;
        uint16_t size;
        uint16_t hr;
        
        
        
        fread(&code,  sizeof(uint16_t),1,input);
        fread(&size,  sizeof(uint16_t),1,input);
        
        if (code != EVENT_LOG_FORMAT && size != MAX_EVENTS ) {
            fclose(input);
            return 0;
        }
            
        fread(&head_, sizeof(uint16_t),1,input);
        fread(&tail_, sizeof(uint16_t),1,input);
        
        for (int i=0;i<MAX_EVENTS;i++)
        {
            fread( &events[i].port,        sizeof(uint16_t),1 , input);
            fread( &events[i].address,     sizeof(uint16_t),1 , input);
            fread( &events[i].hours,       sizeof(uint16_t),1 , input);
            fread( &events[i].mins,        sizeof(uint16_t),1 , input);
            fread( &events[i].secs,        sizeof(uint16_t),1 , input);
            fread( &events[i].day,         sizeof(uint16_t),1 , input);
            fread( &events[i].month,       sizeof(uint16_t),1 , input);
            fread( &events[i].year,        sizeof(uint16_t),1 , input);
#if 0
            if ( events[i].hours < 12 ) {
                
                hr = (events[i].hours == 0) ? 12 : events[i].hours; 
                        
                printf("%02d) %02x %02d:%02d:%02dam\r",
                    i, 
                    events[i].address,
                    hr,
                    events[i].mins,
                    events[i].secs );
                    
            } else  {
                
                hr = (events[i].hours - 12);
                
                printf("%02d) %02x %02d:%02d:%02dpm\r",
                    i, 
                    events[i].address,
                    hr,
                    events[i].mins,
                    events[i].secs );
            }
#endif
                
        }
        fclose(input);
        
        printf("Number of Events = %d (%d) \n\r", this->size(),this->capacity());
    }
    else
    {
        printf("Could not Read %s\n\r",filename );
        return 0;
    }
    return 1;  
}

/******************************************************************************/
int ESCM_EventLog::save()
{
    int result = 1;
    
    if (!dirty) 
        return 0;

    printf("Saving %s ...\n\r",filename );
    
    FILE *output = fopen(filename, "wb");
    if (output != NULL) {
        
        uint16_t code = EVENT_LOG_FORMAT;
        uint16_t size = MAX_EVENTS;
        
        fwrite(&code,  sizeof(uint16_t),1,output);
        fwrite(&size,  sizeof(uint16_t),1,output);
        fwrite(&head_, sizeof(uint16_t),1,output);
        fwrite(&tail_, sizeof(uint16_t),1,output);
        
        for (int i=0;i<MAX_EVENTS;i++)
        {  
            fwrite( &events[i].port,        sizeof(uint16_t),1 , output);
            fwrite( &events[i].address,     sizeof(uint16_t),1 , output);
            fwrite( &events[i].hours,       sizeof(uint16_t),1 , output);
            fwrite( &events[i].mins,        sizeof(uint16_t),1 , output);
            fwrite( &events[i].secs,        sizeof(uint16_t),1 , output);
            fwrite( &events[i].day,         sizeof(uint16_t),1 , output);
            fwrite( &events[i].month,       sizeof(uint16_t),1 , output);
            fwrite( &events[i].year,        sizeof(uint16_t),1 , output);
            ThisThread::sleep_for(1);
                
        }
        fflush(output);
        fclose(output);
        dirty = false;
        result = 1;
    }
    else
    {
        printf("Could not save %s\n\r",filename );
        result = 0;
    }
    
    printf("Done \n\r" );
    return result;
}

/******************************************************************************/
void ESCM_EventLog::display(Serial *pc)
{
    int j = tail_;
    int i = head_ ;
    int count = 0 ;

    pc->printf("\n\r");
    pc->printf("-----------------------------\n\r");
    pc->printf("- Event Log \n\r");
    pc->printf("-----------------------------\n\r");
    pc->printf("%d %d %d \n\r", i,j, full_);
    
    
    while ( count < size() )
    {
        if (--i < 0 ) {
            i =( max_size_ - 1 );
        }
            
        pc->printf("%02d <%02d> %02d :: %02d:%02d:%02d %02d/%02d/%04d \n\r", 
        count, 
        events[i].port, 
        events[i].address, 
        events[i].hours, 
        events[i].mins,
        events[i].secs,
        events[i].month,
        events[i].day,
        events[i].year
        );

        count++;
    }
    pc->printf("-----------------------------\n\r");
}
/******************************************************************************/
void ESCM_EventLog::add(uint16_t address, uint16_t port)
{
    ESCM_Event newEvent;
    newEvent.address = address;
    newEvent.port    = port;
    newEvent.setTimeNow();
    put( newEvent );
}

/******************************************************************************/
void ESCM_EventLog::add(uint16_t address)
{
    ESCM_Event newEvent;
    newEvent.address = address;
    newEvent.port    = 0;
    newEvent.setTimeNow();
    put( newEvent );
}

/******************************************************************************/
ESCM_Event * ESCM_EventLog::index (int pos ){

    ESCM_Event * result = NULL;
    if (!empty())
    {
        if (pos < size() || pos >= 0 ) {
            
            int index = (head_ - 1 - pos );
            
            if (index < 0 )
                index += max_size_;
            
            result = &(events[index]); 
        }
    }
    return result;
}