/**************************************************************************//**
 * @file     AddressMap.cpp
 * @brief    Base class for storage of Address Mapping data.
 * @version: V1.0
 * @date:    9/17/2019

 *
 * @note
 * Copyright (C) 2019 E3 Design. All rights reserved.
 *
 * @par
 * E3 Designers LLC is supplying this software for use with Cortex-M3 LPC1768
 * processor based microcontroller for the ESCM 2000 Monitor and Display.  
 *  *
 * @par
 * THIS SOFTWARE IS PROVIDED "AS IS".  NO WARRANTIES, WHETHER EXPRESS, IMPLIED
 * OR STATUTORY, INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE.
 * ARM SHALL NOT, IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 *
 ******************************************************************************/
#include "mbed.h"
#include "AddressMap.h"
#include "stdio.h"
#include "stdlib.h"

#define ADDRESS_MAP_VERSION 0xFAFA0001

const char * filePath = "/local/address.bin";

/******************************************************************************/
int AddressMap::reset()
{
    int result = 0;
    int i=0;
    
    printf("Initializing Default Data\n\r");

#if 0
    addresses[i].address = i;
    sprintf(addresses[i].description, "Basement", i);
    i++;
    addresses[i].address = i;
    sprintf(addresses[i].description, "Lobby", i);
    i++;
#endif
    int floor = 0;
    while (i<MAX_ADDRESSES) {
        addresses[i].address = i;
        sprintf(addresses[i].description, "Unit %d", i);
        i++;
    }

    result = save();
    return result;
}


/******************************************************************************/
int AddressMap::init()
{
    int result = 0;
    int i=0;
    
    printf("Initializing Address Map\n\r");
    if (!load())
    {
        result = reset();
    } 
    else
    {
        result = 1;
    }
    
    printf("Complete Init\n\r");
    return result;

}

/******************************************************************************/
int AddressMap::load()
{   
    int result = 0;
        
    printf("Loading Address Map Data\n\r");
        
    FILE *input = fopen(filePath, "rb");
    if(input){
        
        printf("Reading %s\n\r",filePath );
        int format;
        int size;;
        
        fread(&format, sizeof(uint32_t),1,input);
        fread(&size  , sizeof(uint32_t),1,input);
        
        if (format == ADDRESS_MAP_VERSION && size == MAX_ADDRESSES) 
        { 
            for (int i=0;i<MAX_ADDRESSES;i++)
            {
                fread( &addresses[i].address,     sizeof(uint32_t),1 , input);
                fread(  addresses[i].description, sizeof(char)    ,MAX_ADDR_LENGTH, input);
            }
            
            result = 1;
        } 
        else {
            printf("Error: Wrong version cannot read %s\n\r",filePath );
            result = 0;
        }
        
        fclose(input);
    }
    else
    {
        printf("Error: Cannot read %s\n\r",filePath );
        result = 0;        
    }
    
    return result;
}



/******************************************************************************/
int AddressMap::save()
{
    int result = 0;
    printf("Saving %s\n\r",filePath );
    

    FILE *output = fopen(filePath, "wb");

    if (output!=NULL) {
        uint32_t format = ADDRESS_MAP_VERSION;
        uint32_t size   = MAX_ADDRESSES;

        fwrite(&format, sizeof(uint32_t),1,output);
        fwrite(&size, sizeof(uint32_t),1,output);

        for (int i=0; i<MAX_ADDRESSES; i++) {
            printf("." );
            fwrite( &addresses[i].address,     sizeof(uint32_t),1, output);
            fwrite(  addresses[i].description, sizeof(char),MAX_ADDR_LENGTH, output);
        }
        fflush(output);
        fclose(output);
        result = 1;
    }
    else
    {
        printf("Error: Cannot write %s\n\r",filePath);
        result = 0;
    }
    
    printf("Done \n\r" );
    return result;
}

/******************************************************************************/
void AddressMap::display(Serial *pc)
{
    pc->printf("\n\r");
    pc->printf("-----------------------------\n\r");
    pc->printf("-- Addresses  \n\r");
    pc->printf("-----------------------------\n\r");
    for (int i=0;i<MAX_ADDRESSES;i++)
    {
        pc->printf("%02d : [%s]\n\r",
            addresses[i].address, addresses[i].description);
    }
}


/******************************************************************************/
char* AddressMap::getDescription(unsigned char idx )
{
    if (idx < MAX_ADDRESSES && idx >= 0 )
        return addresses[idx].description;
    else
        return "Invalid Address";
}
/******************************************************************************/
