/*
TEAM 2 E2PRO2 Test Runners for embedded smart house controller

DESCRIPTION:
This module
- Runs test cases (user stories).
- Only tests the logic of the house controller.
- Does not test the RFID readers. 

REQUIREMENT:
- Prior initialization of a serial object for print results.
- Prior definition of macro constants to define relevant pins.

METHOD:
- Each test is encapsulated in a function.
- Each test has SETUP, TEST CASE, TEST RESULT and TEAR DOWN sections, which
  create needed objects and sets postconditions, performs the story, tests the 
  result and then resets/destroys the created objects.
- Object lifetimes are managed by function scope and object destructors.
- The test result is evaluated using a custom test_assert function.

Changelog:
Date        By          Change
--------    --------    ------------------------------------------------------
20190504    Janus       Adding test structure
20190504    Janus       Adding Test case 1
20190505    Janus       Defining test cases 1 and 2 (passing).
20190506    Janus       Defining test case 3-6 (passing).

*/

#include "mbed.h"

//define test stuff
#define TEST_WAIT_MS 1500
#define USER1 0xAAAAAAAA
#define USER2 0xBBBBBBBB
#define FAST_GATE_S 0
#define DEBOUNCE_GATE_S 2

//define test house
#define TEST_HOUSE_NAME "T2 TestHouse Front Door"
#define TEST_HOUSE_SET_PIN FRONTDOOR_SET
#define TEST_HOUSE_RST_PIN FRONTDOOR_RST

//define test room
#define TEST_ROOM1_NAME "T2 Room 1 Door"
#define TEST_ROOM1_SET_PIN ROOM1DOOR_SET
#define TEST_ROOM1_RST_PIN ROOM1DOOR_RST


void test_assert(const bool expr, const int testId) {
    if (expr) {
        printf("TestID: %d. OK.\r\n", testId);
    } else {
        printf("TestID: %d. Failed.\r\n", testId);
    }
}

/* **** test case 1 **** */
void runTest1(void) {
    
    const int TEST_ID = 1;
    const char * TEST_TEXT = "Som beboer kan jeg komme hjem og saa taender huset\r\n"\
                             "automatisk saa jeg ikke behoever at gaa og taende for alle\r\n"\
                             "apparater selv.";
    bool test_result = false;
    
    printf("=============================================================\r\n");
    printf("TestID: %d\r\n%s\r\n", TEST_ID, TEST_TEXT);
    printf("-------------------------------------------------------------\r\n");
    
    //SET UP: conditions for test
    printf("Preconditions:\r\n");
    Room test_house(TEST_HOUSE_NAME, TEST_HOUSE_SET_PIN, TEST_HOUSE_RST_PIN, FAST_GATE_S);
    test_house.reset();
    printf("-------------------------------------------------------------\r\n");
    
    //TEST CASE: flow of test
    test_house.RoomCall(USER1); //owner enters house
    
    //TEST RESULT: Post-conditions for passed test
    test_result = test_house.getState(); //check that the state is set right
    
    printf("-------------------------------------------------------------\r\n");
    test_assert(test_result, TEST_ID); //house state must be true (turned on).
    printf("=============================================================\r\n");
    
    //TEAR DOWN: scope end wil destroy objects, or if any mem dynamically allocated here, then free it
    test_house.reset();
    
    wait_ms(TEST_WAIT_MS);
}

void runTest2(void) {
    const int TEST_ID = 2;
    const char * TEST_TEXT = "Som beboer kan jeg forlade mit hjem og saa slukker\r\n"\
                             "huset automatisk saa jeg ikke skal gaa rundt og slukke\r\n"\
                             "for alt selv.";
    bool test_result = false;
    
    printf("=============================================================\r\n");    
    printf("TestID: %d\r\n%s\r\n", TEST_ID, TEST_TEXT);
    printf("-------------------------------------------------------------\r\n");

    //SET UP:
    printf("Preconditions:\r\n");
    Room test_house(TEST_HOUSE_NAME, TEST_HOUSE_SET_PIN, TEST_HOUSE_RST_PIN, FAST_GATE_S);
    test_house.reset();
    test_house.RoomCall(USER1); //owner enters house
    printf("-------------------------------------------------------------\r\n");
    
    //TEST CASE:
    test_house.RoomCall(USER1); //owner leaves the house
    
    //TEST RESULT:
    test_result = !test_house.getState();
    printf("-------------------------------------------------------------\r\n");
    test_assert(test_result, TEST_ID); //test result must always be true
    printf("=============================================================\r\n");
    
    //TEAR DOWN:
    test_house.reset();
    wait_ms(TEST_WAIT_MS);
}

void runTest3(void) {
    const int TEST_ID = 3;
    const char * TEST_TEXT = "Som beboer kan jeg gaa ind og ud af huset uden at noget\r\n"\
                             "aendrer tilstand, hvis der stadig er nogen hjemme,\r\n"\
                             "saa systemet ikke generer os.";
                    
    bool test_result = false;
    
    printf("=============================================================\r\n");    
    printf("TestID: %d\r\n%s\r\n", TEST_ID, TEST_TEXT);
    printf("-------------------------------------------------------------\r\n");

    //SET UP:
    printf("Preconditions:\r\n");
    Room test_house(TEST_HOUSE_NAME, TEST_HOUSE_SET_PIN, TEST_HOUSE_RST_PIN, FAST_GATE_S);
    test_house.reset();
    test_house.RoomCall(USER1); //user1 enters house
    test_house.RoomCall(USER2); //user2 enters house
    
    printf("-------------------------------------------------------------\r\n");
    
    //TEST CASE:
    test_house.RoomCall(USER1); //user1 leaves the house
    
    //TEST RESULT:
    printf("-------------------------------------------------------------\r\n");
    //state must be true (house still active as there is still someone home
    test_result = test_house.getState();
    test_assert(test_result, TEST_ID);  //test result must always be true
    printf("=============================================================\r\n");
    
    //TEAR DOWN:
    test_house.reset();
    wait_ms(TEST_WAIT_MS);
}

void runTest4(void) {
    const int TEST_ID = 4;
    const char * TEST_TEXT = "Som beboer kan jeg gaa ind i et rum og saa taender\r\n"\
                             "lyset automatisk, saa jeg ikke behoever\r\n"\
                             "at trykke paa kontakten.";
                             
    bool test_result = false;
    
    printf("=============================================================\r\n");    
    printf("TestID: %d\r\n%s\r\n", TEST_ID, TEST_TEXT);
    printf("-------------------------------------------------------------\r\n");

    //SET UP:
    printf("Preconditions:\r\n");
    Room test_house(TEST_HOUSE_NAME, TEST_HOUSE_SET_PIN, TEST_HOUSE_RST_PIN, FAST_GATE_S);
    Room test_room(TEST_ROOM1_NAME, TEST_ROOM1_SET_PIN, TEST_ROOM1_RST_PIN, FAST_GATE_S);
    test_house.reset();
    test_room.reset();
    test_house.RoomCall(USER1); //user1 enters house
    printf("-------------------------------------------------------------\r\n");
    
    //TEST CASE:
    test_room.RoomCall(USER1); //user1 enters room
    
    //TEST RESULT:
    //house and room must BOTH be active
    test_result = test_house.getState() && test_room.getState();
    printf("-------------------------------------------------------------\r\n");
    test_assert(test_result, TEST_ID);  //test result must always be true
    printf("=============================================================\r\n");
    
    //TEAR DOWN:
    test_house.reset();
    test_room.reset();
    wait_ms(TEST_WAIT_MS);
}

void runTest5(void) {
    const int TEST_ID = 5;
    const char * TEST_TEXT = "Som beboer kan jeg forlade et rum og saa slukker\r\n"\
                             "lyset automatisk, saa jeg ikke behoever\r\n"\
                             "at huske at slukke.";

    bool test_result = false;
    
    printf("=============================================================\r\n");    
    printf("TestID: %d\r\n%s\r\n", TEST_ID, TEST_TEXT);
    printf("-------------------------------------------------------------\r\n");

    //SET UP:
    printf("Preconditions:\r\n");
    Room test_house(TEST_HOUSE_NAME, TEST_HOUSE_SET_PIN, TEST_HOUSE_RST_PIN, FAST_GATE_S);
    Room test_room(TEST_ROOM1_NAME, TEST_ROOM1_SET_PIN, TEST_ROOM1_RST_PIN, FAST_GATE_S);
    test_house.reset();
    test_room.reset();
    test_house.RoomCall(USER1); //user1 enters house
    test_room.RoomCall(USER1); //user1 enters room
    printf("-------------------------------------------------------------\r\n");
    
    //TEST CASE:
    test_room.RoomCall(USER1); //user1 enters room
    
    //TEST RESULT:
    printf("-------------------------------------------------------------\r\n");
    //the room must be off but the house still on
    test_result = test_house.getState() && !test_room.getState();
    test_assert(test_result, TEST_ID);  //test result must always be true
    printf("=============================================================\r\n");
    
    //TEAR DOWN:
    test_house.reset();
    test_room.reset();

    wait_ms(TEST_WAIT_MS);
}

void runTest6(void) {
    const int TEST_ID = 6;
    const char * TEST_TEXT = "Som beboer kan jeg gaa ind og ud af et rum uden at\r\n"\
                             "noget aendrer tilstand, hvis der stadig er nogen i rummet,\r\n"\
                             "saa systemet ikke generer os.";

    bool test_result = false;
    
    printf("=============================================================\r\n");    
    printf("TestID: %d\r\n%s\r\n", TEST_ID, TEST_TEXT);
    printf("-------------------------------------------------------------\r\n");

    //SET UP:
    printf("Preconditions:\r\n");
    Room test_house(TEST_HOUSE_NAME, TEST_HOUSE_SET_PIN, TEST_HOUSE_RST_PIN, FAST_GATE_S);
    Room test_room(TEST_ROOM1_NAME, TEST_ROOM1_SET_PIN, TEST_ROOM1_RST_PIN, FAST_GATE_S);
    test_house.reset();
    test_room.reset();
    test_house.RoomCall(USER1); //user1 enters house
    test_house.RoomCall(USER2); //user2 enters house
    test_room.RoomCall(USER1); //user1 enters room
    test_room.RoomCall(USER2); //user2 enters room
    printf("-------------------------------------------------------------\r\n");
    
    //TEST CASE:
    test_room.RoomCall(USER1); //user1 leaves room
    
    //TEST RESULT:
    printf("-------------------------------------------------------------\r\n");
    //both house and room must be active
    test_result = test_house.getState() && test_room.getState();
    test_assert(test_result, TEST_ID);  //test result must always be true
    printf("=============================================================\r\n");
    
    //TEAR DOWN:
    test_house.reset();
    test_room.reset();
    wait_ms(TEST_WAIT_MS);
}

void runTest7(void) {
    const int TEST_ID = 7;
    const char * TEST_TEXT = "Som beboer kan jeg staa i doeraabningen af et rum\r\n"\
                             "uden at lyset blinker,\r\n"\
                             "saa systemet ikke generer os.";
    bool test_result = false;
    bool test_result7a = false;
    bool test_result7b = false;
    
    printf("=============================================================\r\n");    
    printf("TestID: %d\r\n%s\r\n", TEST_ID, TEST_TEXT);
    printf("-------------------------------------------------------------\r\n");

    //SET UP:
    printf("Preconditions:\r\n");
    Room test_house(TEST_HOUSE_NAME, TEST_HOUSE_SET_PIN, TEST_HOUSE_RST_PIN, DEBOUNCE_GATE_S);
    test_house.reset();
        
    printf("-------------------------------------------------------------\r\n");
    
    //TEST CASE 7a - entering a house:
    test_house.RoomCall(USER1); //user1 enters house
    test_house.RoomCall(USER1); //but user1 keeps standing in the doorway
    test_result7a = test_house.getState(); //house must still be on
    
    wait(DEBOUNCE_GATE_S);
    
    test_house.RoomCall(USER1); //user1 really leaves the house
    test_house.RoomCall(USER1); //but user1 keeps standing in the doorway
    test_result7b = !test_house.getState(); //house must be off now
    
    //TEST RESULT:
    test_result = test_result7a && test_result7b; //completes both part a and b
    printf("-------------------------------------------------------------\r\n");
    test_assert(test_result, TEST_ID);  //test result must always be true
    printf("=============================================================\r\n");
    
    //TEAR DOWN:
    
    wait_ms(TEST_WAIT_MS);
}


/* **** TEST RUNNER **** */
void runTests(void) {
    
    /* **** Boilerplate **** */
    printf("Running self test. Test cases 1-7:\r\n");
    
    //Execute tests
    runTest1();
    runTest2();
    runTest3();
    runTest4();
    runTest5();
    runTest6();
    runTest7();
    
    printf("=============================================================\r\n");
    printf("Tests done.\r\n");
    printf("=============================================================\r\n");
    
} //end test runner